﻿using System;
using System.Diagnostics;
using System.Text;

namespace HIPS.Configuration.Tracing
{
    /// <summary>
    /// Class to implement server side tracing.
    /// </summary>
    public class TraceWriter : IDisposable
    {
        #region Declarations

        /// <summary>
        /// The date time the tracing commenced.
        /// </summary>
        private DateTime _startDateTime;

        /// <summary>
        /// The Trace Source.
        /// </summary>
        private TraceSource _traceSource;

        #endregion Declarations

        #region Constructors

        /// <summary>
        /// Overloaded constructor
        /// </summary>
        /// <param name="traceSource">The name of the trace source form the configuration file.</param>
        /// <param name="writeStartAndEndMessages">A value that indicates if the start and end traces messages should be written or not.</param>
        public TraceWriter(string traceSource)
        {
            _traceSource = new TraceSource(traceSource);
            _startDateTime = DateTime.Now;
        }

        #endregion Constructors

        #region Public Methods

        /// <summary>
        /// Writes an Information trace message to the trace output.
        /// </summary>
        /// <param name="message"></param>
        public void WriteTrace(string message)
        {
            WriteTrace(message, TraceEventType.Information);
        }

        /// <summary>
        /// Writes a trace message with the specified TraceEventType.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="level"></param>
        public void WriteTrace(string message, TraceEventType type)
        {
            WriteTrace(message, type, -1, DateTime.Now);
        }

        /// <summary>
        /// Writes an information message at the specified index.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="index"></param>
        public void WriteTrace(string message, int index)
        {
            WriteTrace(message, TraceEventType.Information, -1, DateTime.Now);
        }

        /// <summary>
        /// Writes a trace message of the specified type at the specified index.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="type"></param>
        /// <param name="index"></param>
        public void WriteTrace(string message, TraceEventType type, int index, DateTime dateStarted)
        {
            WriteTrace(message, type, index, dateStarted, false, false);
        }

        /// <summary>
        /// Writes a trace message of the specified type at the specified index.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="type"></param>
        /// <param name="index"></param>
        public void WriteTrace(string message, TraceEventType type, int index, DateTime dateStarted, bool isStartTrace, bool isEndTrace)
        {
            if (_traceSource.Switch.Level == SourceLevels.Off)
            {
                // Don't bother.  Tracing is off.
                return;
            }

            StringBuilder sb = new StringBuilder();
            sb.Append(dateStarted.ToString("dd/MM/yyyy hh:mm:ss.fff tt"));
            sb.Append(" - ");
            sb.Append(message);

            TraceMessage traceMessage = new TraceMessage(sb.ToString(), type);
            traceMessage.Level = 0;
            traceMessage.Index = 0;

            traceMessage.IsEndMessage = isEndTrace;
            traceMessage.IsStartMessage = isStartTrace;
            _traceSource.TraceData(traceMessage.Type, 1, traceMessage.Message);
        }

        #endregion Public Methods

        #region IDisposable Methods

        /// <summary>
        /// Disposes the TraceWriter.
        /// </summary>
        public void Dispose()
        {
            try
            {
                _traceSource.Flush();
            }
            catch
            {
                // Do nothing.  Not a great deal we can do anyway!.
            }
        }

        #endregion IDisposable Methods
    }
}