﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;

using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Service;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.CommonSchemas.Cda.ParticipatingIndividual;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrHiBusinessLogic.Mapping;
using HIPS.PcehrHiBusinessLogic.Pcehr;
using Nehta.VendorLibrary.CDA.Generator;
using Nehta.VendorLibrary.CDA.SCSModel.DischargeSummary;
using Nehta.VendorLibrary.Common;

namespace HIPS.PcehrHiBusinessLogic.Cda
{
    public class DischargeSummaryService : BusinessServiceBase
    {
        /// <summary>
        /// Initialises a new instance of the <see cref="DischargeSummaryService" /> class.
        /// </summary>
        /// <param name="user">Identifies the person who is responsible for the action.</param>
        public DischargeSummaryService(UserDetails user)
            : base(user)
        {

        }

        /// <summary>
        /// Uploads a PDF discharge summary to the PCEHR National Repository, optionally superseding a previously-uploaded document.
        /// This takes in the PDF document, required metadata, and any attachments and creates a CDA document before packaging it.
        /// </summary>
        /// <param name="request">Request containing a PDF and required metadata for creating the CDA package.</param>
        /// <returns>Details of the document that has been queued for upload.</returns>
        /// <exception cref="ItemNotFoundException">Represents a required item not being found.</exception>
        /// <exception cref="HipsResponseException">Represents an error described by a HipsResponse object.</exception>
        public CdaDocumentDetails UploadDischargeSummaryLevel1A(
            byte[] pdfDocument,
            CdaHeaderMetadata metadata,
            PatientIdentifierBase patientIdentifier,
            List<CdaAttachment> attachments)
        {
            var details = this.CreateDischargeSummaryLevel1A(pdfDocument, metadata, patientIdentifier, attachments);
            try
            {
                DocumentUploadBeforeQueue dubq = new DocumentUploadBeforeQueue(
                    this.UserContext,
                    details.PatientIdentifier,
                    details.Metadata.AdmissionDateTime,
                    details.CdaDocumentXmlBytes,
                    ObjectMapper.Map<PcehrSchemas.Attachment[]>(details.Attachments),
                    details.DocumentFormatCode);
                details.Status = dubq.UploadOrSupersedeDocument();
                if (details.Status.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(details.Status);
                }
            }
            catch (HipsResponseException ex)
            {
                // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                throw ex;
            }
            catch (Exception ex)
            {
                details.Status.Status = HipsResponseIndicator.SystemError;
                details.Status.HipsErrorMessage = ex.Message;
                details.Status.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
                details.Status.ResponseCode = ex.GetType().Name;
                throw new HipsResponseException(details.Status);
            }
            return details;
        }

        /// <summary>
        /// Creates a CDA discharge summary document that wraps a PDF document body.
        /// This may be for the purpose of uploading the discharge summary to the
        /// PCEHR or for provider-to-provider (P2P) secure message delivery (SMD).
        /// This takes in the PDF document, required metadata, and any attachments
        /// and creates a CDA document before packaging it.
        /// </summary>
        /// <param name="request">Request containing a PDF and required metadata for creating the CDA package.</param>
        /// <returns>Details of the document that has been queued for upload.</returns>
        /// <exception cref="ItemNotFoundException">Represents a required item not being found.</exception>
        /// <exception cref="HipsResponseException">Represents an error described by a HipsResponse object.</exception>
        public CdaDocumentDetails CreateDischargeSummaryLevel1A(
            byte[] pdfDocument,
            CdaHeaderMetadata metadata,
            PatientIdentifierBase patientIdentifier,
            List<CdaAttachment> attachments)
        {
            var details = new CdaDocumentDetails();
            try
            {
                details.Metadata = metadata;
                details.PatientIdentifier = patientIdentifier;
                LookupLocalInformation(details);
                if (details.Status.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(details.Status);
                }
                details.Mrn = details.HospitalPatient.Mrn;
                SetOrganisation(details);
                ConstructAttachmentList(details, pdfDocument, attachments);
                LoadPatientNames(details);
                details.EventNarrativeText = "See attached Discharge Summary document.";

                // If the document author's HPI-I is supplied, use the HPI-I enforced template, otherwise use the HPI-I relaxed template.
                details.DocumentFormatCode = details.Metadata.DocumentAuthor.Hpii != null
                    ? HIPS.CommonSchemas.Cda.Constants.DocumentFormatCode.DISCHARGE_SUMMARY_1A_ENFORCED
                    : HIPS.CommonSchemas.Cda.Constants.DocumentFormatCode.DISCHARGE_SUMMARY_1A_RELAXED;

                SaveDocumentDetails(details);
                if (details.Status.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(details.Status);
                }
                EDischargeSummary eDischargeSummary = PopulateDischargeSummary(details);
                XmlDocument cdaDocument = CDAGenerator.GenerateEDischargeSummary(eDischargeSummary);
                details.CdaDocumentXmlBytes = ConvertToBinary(cdaDocument);
            }
            catch (HipsResponseException ex)
            {
                // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                throw ex;
            }
            catch (ItemNotFoundException ex)
            {
                // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                throw ex;
            }
            catch (ValidationException ex)
            {
                details.Status.Status = HipsResponseIndicator.ValidationError;
                details.Status.HipsErrorMessage = ex.GetMessagesString();
                throw new HipsResponseException(details.Status);
            }
            catch (Exception ex)
            {
                details.Status.Status = HipsResponseIndicator.SystemError;
                details.Status.HipsErrorMessage = ex.Message;
                details.Status.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
                details.Status.ResponseCode = ex.GetType().Name;
                throw new HipsResponseException(details.Status);
            }
            return details;
        }

        /// <summary>
        /// If the caller of the service has not supplied an employer HPI-O
        /// and Name for the participating providers, obtains this information
        /// from the HPI-O and Name configured for the Hospital in the database.
        /// </summary>
        /// <param name="details">Document upload details.</param>
        private void SetOrganisation(CdaDocumentDetails details)
        {
            List<ParticipatingProvider> providers = new List<ParticipatingProvider>()
            {
                details.Metadata.DocumentAuthor,
                details.Metadata.ResponsibleHealthProfessional,
                details.Metadata.LegalAuthenticator
            };
            providers.ForEach(provider =>
            {
                if (provider != null && provider.EmployerHpio == null && provider.EmployerName == null)
                {
                    provider.EmployerHpio = details.Hospital.HpiO;
                    provider.EmployerName = details.Hospital.HpioName;
                }
            });
        }

        /// <summary>
        /// Saves the document details into the CdaSetNumber and CdaDocumentNumber tables,
        /// allocating the version number and numeric IDs for the DocumentSetId and 
        /// DocumentId in the process.
        /// </summary>
        /// <param name="details">Document upload details.</param>
        private void SaveDocumentDetails(CdaDocumentDetails details)
        {
            DocumentType documentType = ListSingleton.Instance.AllDocumentTypes.Single(a => a.Code == DocumentTypeCodes.DischargeSummary);
            DocumentFormat documentFormat = ListSingleton.Instance.AllDocumentFormats.Single(a => a.Code == details.DocumentFormatCode);
            CdaDocumentNumberDl documentNumberDl = new CdaDocumentNumberDl(this.UserContext);
            CdaSetNumberDl setNumberDl = new CdaSetNumberDl(this.UserContext);
            CdaSetNumber setNumber = setNumberDl.GetByEpisodeAndDocumentType(details.Episode.EpisodeId.Value, documentType.DocumentTypeId.Value);
            setNumber.EpisodeId = details.Episode.EpisodeId.Value;
            setNumber.DocumentFormatId = documentFormat.DocumentFormatId.Value;
            setNumber.DocumentTypeId = documentType.DocumentTypeId.Value;
            setNumber.ModeOfSeparationId = (int)details.Metadata.ModeOfSeparation;
            setNumber.AdmissionDateTime = details.Metadata.AdmissionDateTime;
            setNumber.DischargeDateTime = details.Metadata.DischargeDateTime;
            CdaDocumentNumber documentNumber = new CdaDocumentNumber();
            documentNumber.DocumentCreationDateTime = details.Metadata.DocumentCreationDateTime;

            bool saved = false;
            SqlTransaction transaction = null;
            using (SqlCommand command = setNumberDl.GetCommand())
            {
                try
                {
                    transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;
                    saved = setNumberDl.Save(setNumber, transaction);
                    if (saved)
                    {
                        documentNumber.CdaSetNumberId = setNumber.CdaSetNumberId.Value;
                        saved = documentNumberDl.Insert(documentNumber, transaction);
                    }
                    if (saved)
                    {
                        transaction.Commit();
                    }
                    else
                    {
                        throw new Exception("Items did not save");
                    }
                }
                catch (Exception ex)
                {
                    try
                    {
                        if (transaction != null)
                        {
                            transaction.Rollback();
                        }
                        if (command.Connection != null)
                        {
                            command.Connection.Close();
                        }
                    }
                    catch (Exception)
                    {
                        // Ignore exceptions while rolling back or closing.
                    }
                    details.Status.Status = HipsResponseIndicator.DatabaseError;
                    details.Status.ResponseCode = ex.GetType().Name;
                    details.Status.ResponseCodeDescription = ex.Message;
                    details.Status.ResponseCodeDetails = ex.InnerException == null ? null : ex.InnerException.Message;
                    details.Status.HipsErrorMessage = "Error Saving Clinical Document Details";
                    return;
                }
            }
            details.DocumentSetId = new EntityIdentifier
            {
                Root = string.Format("{0}.{1}", HIPS.CommonSchemas.Cda.Constants.Oid.DOCUMENT_SET_ID, details.Hospital.HpiO),
                Extension = setNumber.CdaSetNumberId.ToString(),
                AssigningAuthorityName = string.Format("HIPS {0} Set", details.Hospital.HpioName)
            };
            details.DocumentId = new EntityIdentifier
            {
                Root = string.Format("{0}.{1}.{2}", HIPS.CommonSchemas.Cda.Constants.Oid.DOCUMENT_ID, details.Hospital.HpiO, documentNumber.CdaDocumentNumberId),
                Extension = documentNumber.CdaDocumentNumberId.ToString(),
                AssigningAuthorityName = string.Format("HIPS {0} Doc", details.Hospital.HpioName)
            };
            details.Version = setNumber.DocumentCount + 1;
        }

        /// <summary>
        /// Loads the patient's names into the document details. The patient's
        /// registered name must be included in the CDA document. If the
        /// current name is materially different (in the family or given names
        /// part), then the current name will also be included.
        /// </summary>
        /// <param name="details">The document upload details.</param>
        private void LoadPatientNames(CdaDocumentDetails details)
        {
            var registeredName = ObjectMapper.Map<IndividualName>(details.PatientMaster.LegalName);
            var preferredName = ObjectMapper.Map<IndividualName>(details.PatientMaster.CurrentName);
            registeredName.Usage = IndividualNameUsage.RegisteredName;
            preferredName.Usage = IndividualNameUsage.OtherName;

            details.PatientNames = new List<IndividualName>();
            details.PatientNames.Add(registeredName);
            if (registeredName.IsMateriallyDifferentFrom(preferredName))
            {
                details.PatientNames.Add(preferredName);
            }
        }

        /// <summary>
        /// Constructs the CDA attachment list, including the supplied PDF
        /// document body, the supplied attachment files, and the 
        /// organisational logo. If one of the supplied attachments has a file
        /// name of "logo.png" then it is used as the organisational logo, 
        /// otherwise if the hospital Logo column is populated then that value
        /// is used as the organisational logo, otherwise there will be no 
        /// organisational logo on the CDA document.
        /// </summary>
        /// <param name="details">Document upload details.</param>
        /// <param name="pdfDocument">Contents of the document body PDF file.</param>
        /// <param name="suppliedAttachments">List of supplied attachment files.</param>
        private void ConstructAttachmentList(CdaDocumentDetails details, byte[] pdfDocument, List<CdaAttachment> suppliedAttachments)
        {
            details.Attachments = new List<CdaAttachment>();

            // Add an attachment for the document body PDF. This one must be first.
            details.Attachments.Add(new CdaAttachment
            {
                AttachmentType = CommonSchemas.Cda.AttachmentType.DocumentBodyPdf,
                Caption = HIPS.CommonSchemas.Cda.Constants.AttachmentCaption.DISCHARGE_SUMMARY_UNSTRUCTURED_BODY,
                Content = pdfDocument,
                FileName = HIPS.CommonSchemas.Cda.Constants.AttachmentFileName.UNSTRUCTURED_BODY
            });

            // Add each of the supplied attachments.
            details.Attachments.AddRange(suppliedAttachments);

            // If no logo supplied as attachment, and hospital has a logo, add that logo.
            if (!details.Attachments.Exists(a => a.AttachmentType == CommonSchemas.Cda.AttachmentType.OrganisationalLogoPng))
            {
                if (details.Hospital.Logo != null)
                {
                    details.Attachments.Add(new CdaAttachment
                        {
                            AttachmentType = CommonSchemas.Cda.AttachmentType.OrganisationalLogoPng,
                            Caption = HIPS.CommonSchemas.Cda.Constants.AttachmentCaption.ORGANISATIONAL_LOGO,
                            Content = details.Hospital.Logo,
                            FileName = HIPS.CommonSchemas.Cda.Constants.AttachmentFileName.ORGANISATIONAL_LOGO
                        });
                }
            }
        }

        /// <summary>
        /// Looks up the hospital, patient and episode information from the database,
        /// and stores this information into the document upload details.
        /// </summary>
        /// <param name="details">Document upload details.</param>
        private void LookupLocalInformation(CdaDocumentDetails details)
        {
            PatientAccess patientAccess = new PatientAccess(this.UserContext);
            PatientMaster patientMaster;
            Hospital hospital;
            HospitalPatient hospitalPatient;
            details.Status = patientAccess.GetHospital(details.PatientIdentifier, out hospital);
            if (details.Status.Status == HipsResponseIndicator.InvalidHospital)
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Hospital);
            }
            if (details.Status.Status != HipsResponseIndicator.OK)
            {
                throw new HipsResponseException(details.Status);
            }
            details.Hospital = hospital;
            details.Status = patientAccess.GetPatient(details.PatientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (details.Status.Status == HipsResponseIndicator.InvalidPatient)
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Patient);
            }
            if (details.Status.Status != HipsResponseIndicator.OK)
            {
                throw new HipsResponseException(details.Status);
            }
            details.HospitalPatient = hospitalPatient;
            details.PatientMaster = patientMaster;
            Episode episode = patientAccess.GetEpisodeWithoutCancelled(details.PatientIdentifier, details.Metadata.AdmissionDateTime, hospitalPatient);
            details.Episode = episode;
            if (episode == null)
            {
                throw new ItemNotFoundException(ItemNotFoundException.ItemType.Episode);
            }
            if (details.Metadata.PatientAddress == null)
            {
                var dbAddress = patientMaster.Addresses.FirstOrDefault(a => !string.IsNullOrEmpty(a.AddressLine1));
                if (dbAddress == null)
                {
                    throw new ItemNotFoundException(ItemNotFoundException.ItemType.Address);
                }
                if (string.IsNullOrEmpty(dbAddress.CountryName))
                {
                    dbAddress.CountryName = HIPS.CommonSchemas.Cda.Constants.CountryName.AUSTRALIA;
                }
                details.Metadata.PatientAddress = ObjectMapper.Map<Address>(patientMaster.Addresses.FirstOrDefault());
            }
            if (details.Metadata.PatientContactDetails == null)
            {
                Contact dbContact = patientMaster.Contacts.FirstOrDefault(a => !string.IsNullOrEmpty(a.Detail));
                details.Metadata.PatientContactDetails = ObjectMapper.Map<HIPS.CommonSchemas.Cda.ElectronicCommunicationDetail>(dbContact);
            }
        }

        /// <summary>
        /// Converts the CDA document to a binary blob.
        /// </summary>
        /// <param name="cdaDocument">CDA document.</param>
        /// <returns>Binary blob.</returns>
        private static byte[] ConvertToBinary(XmlDocument cdaDocument)
        {
            byte[] cdaDocumentBytes;
            using (MemoryStream stream = new MemoryStream())
            {
                XmlWriterSettings settings = new XmlWriterSettings();
                settings.Encoding = new UTF8Encoding(encoderShouldEmitUTF8Identifier: false);
                using (XmlWriter writer = XmlWriter.Create(stream, settings))
                {
                    cdaDocument.Save(writer);
                }
                cdaDocumentBytes = stream.ToArray();
            }
            return cdaDocumentBytes;
        }

        /// <summary>
        /// Populates the NEHTA CDA Library's EDischargeSummary model from the details.
        /// </summary>
        /// <param name="details">Document upload details.</param>
        /// <returns>NEHTA eDischarge Summary model.</returns>
        private EDischargeSummary PopulateDischargeSummary(CdaDocumentDetails details)
        {
            EDischargeSummary model = ObjectMapper.Map<EDischargeSummary>(details);
            return model;
        }
    }
}
