﻿using System.Linq;

using AutoMapper;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.PcehrHiBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Level 1A Discharge Summary.
    /// </summary>
    internal class DischargeSummaryProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary>()
                .ForMember(dest => dest.CDAContext, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.DocumentCreationTime, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.DocumentStatus, opt => opt.MapFrom(src => src.Metadata.SourceDocumentStatus))
                .ForMember(dest => dest.IncludeLogo, opt => opt.MapFrom(src => src.Attachments.Exists(a => a.AttachmentType == ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng)))
                .ForMember(dest => dest.LogoPath, opt => opt.MapFrom(src => src.Attachments.SingleOrDefault(a => a.AttachmentType == ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng).FileName))
                .ForMember(dest => dest.LogoContent, opt => opt.MapFrom(src => src.Attachments.SingleOrDefault(a => a.AttachmentType == ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng).Content))
                .ForMember(dest => dest.SCSContent, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.SCSContext, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.ShowAdministrativeObservations, opt => opt.UseValue(false));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.ICDAContextEDischargeSummary>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateCDAContext())
                .ForMember(dest => dest.Custodian, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.DocumentId, opt => opt.MapFrom(src => src.DocumentId))
                .ForMember(dest => dest.InformationRecipients, opt => opt.UseValue(null))
                .ForMember(dest => dest.LegalAuthenticator, opt => opt.MapFrom(src => src.Metadata.LegalAuthenticator == null ? null : src))
                .ForMember(dest => dest.SetId, opt => opt.MapFrom(src => src.DocumentSetId))
                .ForMember(dest => dest.Version, opt => opt.MapFrom(src => src.Version));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IEDischargeSummaryContext>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateSCSContext())
                .ForMember(dest => dest.Attested, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Author, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Facility, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.SubjectOfCare, opt => opt.MapFrom(src => src));

            // Add all attachments except the Organisational Logo as body files.
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IEDischargeSummaryContent>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateSCSContent())
                .ForMember(dest => dest.Event, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.StructuredBodyFiles, opt => opt.MapFrom(src => src.Attachments.Where(a => a.AttachmentType != ModelSchemas.Cda.AttachmentType.OrganisationalLogoPng)));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Event>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateEvent())
                .ForMember(dest => dest.CustomNarrativeEvent, opt => opt.MapFrom(src => src.EventNarrativeText))
                .ForMember(dest => dest.Encounter, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Encounter>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateEncounter())
                .ForMember(dest => dest.ResponsibleHealthProfessional, opt => opt.MapFrom(src => src.Metadata.ResponsibleHealthProfessional))
                .ForMember(dest => dest.EncounterPeriod, opt => opt.MapFrom(src => src.Metadata))
                .ForMember(dest => dest.SeparationMode, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode
                    {
                        Code = ((int)src.Metadata.ModeOfSeparation).ToString(),
                        CodeSystem = "2.16.840.1.113883.13.65",
                        CodeSystemName = "AIHW Mode of Separation"
                    }));

        }

        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion Methods
    }
}