﻿// -----------------------------------------------------------------------
// <copyright file="NOC_DoesPcehrExist.cs" company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrHiBusinessLogic.Pcehr
{
    /// <summary>
    /// This class contains the business logic to check the patient demographic
    /// information in a document that is downloaded from the PCEHR matches the
    /// locally stored information for the patient.
    /// </summary>
    public class DemographicCheck
    {
        #region Private Properties

        private const double PATIENT_DATE_OF_BIRTH_TOLERANCE_HOURS = 12.0;

        private XmlDocument documentField;
        private XmlNamespaceManager namespaceManagerField;
        private PatientMaster patientMasterField;
        private UserDetails userField;

        private PatientMaster LocalPatient
        {
            get
            {
                return patientMasterField;
            }
        }

        private UserDetails User
        {
            get
            {
                return userField;
            }
        }

        /// <summary>
        /// The XmlDocument object that backs this DemographicCheck object.
        /// </summary>
        private XmlDocument Document
        {
            get
            {
                if (documentField == null)
                {
                    documentField = new XmlDocument();
                }
                return documentField;
            }
        }

        /// <summary>
        /// An XmlNamespaceManager that includes the HL7 v3 (abbreviated to "x")
        /// and Australian CDA Extensions (abbreviated to "ext") namespaces.
        /// </summary>
        private XmlNamespaceManager NamespaceManager
        {
            get
            {
                if (namespaceManagerField == null)
                {
                    namespaceManagerField = new XmlNamespaceManager(new NameTable());
                    namespaceManagerField.AddNamespace("ext", XmlStringMap.AustralianCdaExtensionsV3Namespace);
                    namespaceManagerField.AddNamespace("x", XmlStringMap.HL7V3Namespace);
                }
                return namespaceManagerField;
            }
        }

        #endregion Private Properties

        #region Constructor

        /// <summary>
        /// Initialises the demographic check business logic for a local patient.
        /// </summary>
        /// <param name="localPatient">The local patient demographic information.</param>
        /// <param name="user">The person responsible for downloading the document (for logging).</param>
        public DemographicCheck(PatientMaster localPatient, UserDetails user)
        {
            patientMasterField = localPatient;
            userField = user;
        }

        #endregion Constructor

        #region Public Methods

        /// <summary>
        /// Checks the patient IHI, sex, date of birth and family names in the
        /// document match the IHI, registered or current sex, date of birth
        /// and the registered family name or any of the family names from the PAS.
        /// </summary>
        /// <param name="cdaDocument">The downloaded CDA document contents.</param>
        /// <param name="response">Response to which warnings will be added if the
        /// IHI, sex, date of birth or family names do not match or if unable to
        /// get the information from the document.</param>
        public void ValidateDemographics(byte[] cdaDocument, HipsResponse response)
        {
            HipsResponseIndicator originalStatus = response.Status;
            if (response.HipsErrorMessage == null)
            {
                response.HipsErrorMessage = string.Empty;
            }
            LoadDocument(cdaDocument, response);
            if (response.Status != HipsResponseIndicator.InvalidDocument)
            {
                ValidateIhi(response);
                ValidateSex(response);
                ValidateDateOfBirth(response);
                ValidateFamilyNames(response);
            }

            // TODO: Further consider what logging should be done for validation failures.
            if (response.Status != originalStatus)
            {
                Exception ex = new Exception(response.HipsErrorMessage);
                EventLogger.WriteLog("Demographic Mismatch", ex, User, LogMessage.HIPS_MESSAGE_088);
            }
        }

        #endregion Public Methods

        #region Private Validation Methods

        /// <summary>
        /// Loads the CDA document.
        /// </summary>
        /// <param name="cdaDocument">The downloaded CDA document contents.</param>
        /// <param name="response">Response to which warnings will be added if
        /// unable to load the document.</param>
        private void LoadDocument(byte[] cdaDocument, HipsResponse response)
        {
            using (MemoryStream stream = new MemoryStream(cdaDocument))
            {
                try
                {
                    Document.Load(stream);
                }
                catch (XmlException ex)
                {
                    response.Status = HipsResponseIndicator.InvalidDocument;
                    response.HipsErrorMessage += "[Cannot load document]";
                    response.ResponseCodeDescription = ex.Message;
                }
            }
        }

        /// <summary>
        /// Checks whether the patient IHI in the document matches the IHI
        /// stored locally.
        /// </summary>
        /// <param name="response">Response to which warnings will be added if the
        /// IHI does not match or if unable to get the IHI from the document.</param>
        private void ValidateIhi(HipsResponse response)
        {
            string documentIhi = PatientIhiFromDocument;
            if (!string.IsNullOrEmpty(documentIhi))
            {
                if (LocalPatient.Ihi != PatientIhiFromDocument)
                {
                    response.Status = HipsResponseIndicator.DemographicMismatchWarning;
                    response.HipsErrorMessage += "[Different IHI]";
                }
            }
            else
            {
                response.Status = HipsResponseIndicator.InvalidDocument;
                response.HipsErrorMessage += "[Cannot retrieve patient IHI from document]";
            }
        }

        /// <summary>
        /// Checks whether the patient sex in the document matches either
        /// of the patient sexes stored locally (either the Medicare validated
        /// information or the current sex supplied by the PAS).
        /// </summary>
        /// <param name="response">Response to which warnings will be added if the
        /// sex does not match or if unable to get the sex from the document.</param>
        private void ValidateSex(HipsResponse response)
        {
            SexEnumerator? documentSex = PatientSexFromDocument;
            if (documentSex.HasValue)
            {
                if (LocalPatient.RegisteredSexId != (int)documentSex.Value)
                {
                    if (LocalPatient.CurrentSexId != (int)documentSex.Value)
                    {
                        response.Status = HipsResponseIndicator.DemographicMismatchWarning;
                        response.HipsErrorMessage += "[Different Sex]";
                    }
                }
            }
            else
            {
                response.Status = HipsResponseIndicator.InvalidDocument;
                response.HipsErrorMessage += "[Cannot retrieve patient sex from document]";
            }
        }

        /// <summary>
        /// Checks whether the patient date of birth in the document matches
        /// the patient date of birth stored locally.
        /// </summary>
        /// <param name="response">Response to which warnings will be added if
        /// the two dates of birth do not match or if unable to get the date of
        /// birth from the document.</param>
        private void ValidateDateOfBirth(HipsResponse response)
        {
            DateTime? documentDateOfBirth = PatientDateOfBirthFromDocument;
            if (documentDateOfBirth.HasValue)
            {
                if ((LocalPatient.DateOfBirth - documentDateOfBirth.Value).Duration().TotalHours > PATIENT_DATE_OF_BIRTH_TOLERANCE_HOURS)
                {
                    response.Status = HipsResponseIndicator.DemographicMismatchWarning;
                    response.HipsErrorMessage += "[Different DOB]";
                }
            }
            else
            {
                response.Status = HipsResponseIndicator.InvalidDocument;
                response.HipsErrorMessage += "[Cannot retrieve patient DOB from document]";
            }
        }

        /// <summary>
        /// Checks whether any of the patient family names in the document match any
        /// of the patient family names stored locally (either in the Medicare registered
        /// information or the other names supplied by the PAS).
        /// </summary>
        /// <param name="response">Response to which warnings will be added if none
        /// of the family names match or if unable to get the family names from the
        /// document.</param>
        private void ValidateFamilyNames(HipsResponse response)
        {
            List<PatientMasterName> documentNames = PatientNamesFromDocument;
            if (documentNames.Count > 0)
            {
                bool matched = false;
                foreach (PatientMasterName name in documentNames)
                {
                    if (PatientHasFamilyName(name.FamilyName))
                    {
                        matched = true;
                        break;
                    }
                }
                if (!matched)
                {
                    response.Status = HipsResponseIndicator.DemographicMismatchWarning;
                    response.HipsErrorMessage += "[Different Family Name]";
                }
            }
            else
            {
                response.Status = HipsResponseIndicator.InvalidDocument;
                response.HipsErrorMessage += "[Cannot retrieve patient names from document]";
            }
        }

        /// <summary>
        /// Checks whether the local patient has the given family name, either
        /// as the registered (Medicare) family name or as one of the other
        /// names supplied by the PAS.
        /// </summary>
        /// <param name="familyName">The family name from the document</param>
        /// <returns>True if the patient has had the given family name</returns>
        private bool PatientHasFamilyName(string familyName)
        {
            if (CompareName(LocalPatient.RegisteredFamilyName, familyName))
            {
                return true;
            }
            foreach (PatientMasterName name in LocalPatient.Names)
            {
                if (CompareName(name.FamilyName, familyName))
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// Performs case-insensitive matching of two names.
        /// </summary>
        /// <param name="expected">The local name that was expected</param>
        /// <param name="actual">The actual name in the document</param>
        /// <returns>True if the names match</returns>
        private bool CompareName(string expected, string actual)
        {
            if (string.IsNullOrEmpty(expected) || string.IsNullOrEmpty(actual))
            {
                return false;
            }
            return expected.ToUpper() == actual.ToUpper();
        }

        #endregion Private Validation Methods

        #region Private Document Extraction Properties

        /// <summary>
        /// The patient names from the downloaded document.
        /// </summary>
        private List<PatientMasterName> PatientNamesFromDocument
        {
            get
            {
                XmlNodeList nameNodes = Document.SelectNodes(XmlStringMap.PatientNameXPath, NamespaceManager);
                List<PatientMasterName> names = new List<PatientMasterName>();
                foreach (XmlNode node in nameNodes)
                {
                    PatientMasterName name = new PatientMasterName();
                    name.FamilyName = node.SelectSingleNode("x:family", NamespaceManager).InnerText;
                    List<string> givenNames = new List<string>();
                    foreach (XmlNode given in node.SelectNodes("x:given", NamespaceManager))
                    {
                        givenNames.Add(given.InnerText);
                    }
                    List<string> prefixes = new List<string>();
                    foreach (XmlNode prefix in node.SelectNodes("x:prefix", NamespaceManager))
                    {
                        prefixes.Add(prefix.InnerText);
                    }
                    List<string> suffixes = new List<string>();
                    foreach (XmlNode suffix in node.SelectNodes("x:suffix", NamespaceManager))
                    {
                        suffixes.Add(suffix.InnerText);
                    }
                    name.GivenNames = string.Join(" ", givenNames);
                    name.Title = string.Join(" ", prefixes);
                    name.Suffix = string.Join(" ", suffixes);
                    names.Add(name);
                }
                return names;
            }
        }

        /// <summary>
        /// The patient date of birth from the document.
        /// </summary>
        private DateTime? PatientDateOfBirthFromDocument
        {
            get
            {
                XmlNode dobNode = Document.SelectSingleNode(XmlStringMap.PatientDateOfBirthXPath, NamespaceManager);
                return HL7DateTime.Parse(dobNode.Attributes["value"].Value);
            }
        }

        /// <summary>
        /// The patient sex from the document.
        /// </summary>
        private SexEnumerator? PatientSexFromDocument
        {
            get
            {
                XmlNode sexNode = Document.SelectSingleNode(XmlStringMap.PatientSexXPath, NamespaceManager);
                if (sexNode != null)
                {
                    switch (sexNode.Attributes["code"].Value)
                    {
                        case "M": return SexEnumerator.Male;
                        case "F": return SexEnumerator.Female;
                        case "I": return SexEnumerator.IntersexOrIndeterminate;
                        case "N": return SexEnumerator.NotStatedOrInadequatelyDescribed;
                    }
                }
                return null;
            }
        }

        /// <summary>
        /// The patient IHI from the document.
        /// </summary>
        private string PatientIhiFromDocument
        {
            get
            {
                string ihi = null;
                XmlNode ihiNode = Document.SelectSingleNode(XmlStringMap.PatientIhiXPath, NamespaceManager);
                if (ihiNode != null)
                {
                    ihi = ihiNode.Attributes["root"].Value;
                    string oidQualifier = "1.2.36.1.2001.1003.0.";
                    if (ihi.StartsWith(oidQualifier))
                    {
                        ihi = ihi.Substring(oidQualifier.Length);
                    }
                }
                return ihi;
            }
        }

        #endregion Private Document Extraction Properties
    }
}