﻿using System;
using System.Configuration;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Text;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Exceptions;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrHiBusinessLogic.Ihi;
using HIPS.PcehrSchemas;
using HIPS.PcehrSchemas.Exceptions;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.PCEHRProfile;

namespace HIPS.PcehrHiBusinessLogic.Pcehr
{
    public class DoesPcehrExist
    {
        #region Events

        /// <summary>
        /// Validates the service certificate.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="certificate">The certificate.</param>
        /// <param name="chain">The chain.</param>
        /// <param name="sslPolicyErrors">The SSL policy errors.</param>
        /// <returns></returns>
        private bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return DocumentHelper.ValidateServiceCertificate(sender, certificate, chain, sslPolicyErrors);
        }

        #endregion Events

        #region Methods

        /// <summary>
        /// For a given MRN, State Patient Identifier, PCEHR Data Store PatientMasterId or a validated IHI,
        /// checks whether the patient has advertised the existence of his/her PCEHR,
        /// whether access has been gained, and whether access can be gained without a code.
        /// </summary>
        /// <param name="patientIdentifier">An identifier of the patient (MRN, State ID, PatientMasterId or IHI)</param>
        /// <param name="hospitalCode">Hospital code</param>
        /// <param name="hospitalCodeSystem">Hospital code system (pasFacCd)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action</param>
        /// <returns>Response containing operation success indicator, whether advertised, and access status</returns>
        public DoesPcehrExistResponse IsPcehrAdvertised(PatientIdentifierBase patientIdentifier, DateTime dateOfBirth, UserDetails user)
        {
            DoesPcehrExistResponse response = new DoesPcehrExistResponse();
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            response.HipsResponse = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }
            response.HipsResponse = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK && response.HipsResponse.Status != HipsResponseIndicator.InvalidIhi)
            {
                return response;
            }
            try
            {
                IhiSearchResponse ihiResponse = new PatientIhiValidation().GetValidatedIhi(patientIdentifier, hospital, user, patientMaster, dateOfBirth);
                if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    response.HipsResponse = ihiResponse.HipsResponse;
                    return response;
                }
                patientAccess.ValidateLocalIhiInformation(patientMaster, response.HipsResponse);
                if (response.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return response;
                }
                response = PcehrExists(patientIdentifier, hospital, patientMaster, user);
            }
            catch (IhiServiceUnavailableException ex)
            {
                response.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                response.HipsResponse.HipsErrorMessage = ResponseStrings.IhiRequiresValidation;
                response.HipsResponse.ResponseCodeDescription = ex.Message;
            }
            catch (PcehrServiceUnavailableException ex)
            {
                response.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                response.HipsResponse.HipsErrorMessage = ResponseStrings.PcehrSystemTemporarilyUnavailable;
                response.HipsResponse.ResponseCodeDescription = ex.Message;
            }
            catch (Exception ex)
            {
                response.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                response.HipsResponse.HipsErrorMessage = ResponseStrings.PcehrSystemTemporarilyUnavailable;
                response.HipsResponse.ResponseCodeDescription = ex.Message;
            }
            return response;
        }

        /// <summary>
        /// <para>Calls the doesPCEHRExist web service for the given hospital, patient and user, audits the operation, stores the result in
        /// the HealthProviderOrganisationPatient table in the database, and returns the response.
        /// Note that the result of doesPCEHRExist depends on which HPI-O (associated with the hospital) is making the call.</para>
        /// <para>If the HPI-O is on the Provider Access List of a consumer's PCEHR, and the HPI-O calls doesPCEHRExist, the result will depend on the consumer's chosen Read permissions for the organisation:</para>
        /// <list type="bullet">
        /// <item><description>If Restricted Access, PCEHRExists will be true and AccessCodeRequired will be GainedAccess</description></item>
        /// <item><description>If General Access, PCEHRExists will be true and AccessCodeRequired will be GainedAccess</description></item>
        /// <item><description>If Revoked Access, PCEHRExists will be false and AccessCodeRequired will be unspecified</description></item>
        /// </list>
        /// <para>If the HPI-O is not on the Provider Access List of the PCEHR, the result will depend on the consumer's chosen Advertisement setting for the PCEHR</para>
        /// <list type="bullet">
        /// <item><description>If advertised, PCEHRExists will be true and AccessCodeRequired will be WithCode or WithoutCode</description></item>
        /// <item><description>If not advertised, PCEHRExists will be false and AccessCodeRequired will be unspecified</description></item>
        /// </list>
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier (in which the alternate organisation name may be supplied)</param>
        /// <param name="hospital">The hospital from which this call is being made</param>
        /// <param name="patientMaster">The patient about which this call is being made</param>
        /// <param name="user">The person responsible for this call</param>
        /// <returns>A HIPS 'does PCEHR exist' response object</returns>
        public DoesPcehrExistResponse PcehrExists(PatientIdentifierBase patientIdentifier, Hospital hospital, PatientMaster patientMaster, UserDetails user)
        {
            DoesPcehrExistResponse result = new DoesPcehrExistResponse();

            try
            {
                X509Certificate2 signingCert = Helpers.GetHpioCertificate(hospital);
                Uri url = Helpers.GetDoesPcehrUrl();
                result = InvokeDoesPcehrExist(patientIdentifier, hospital, patientMaster, url, signingCert, user);
                StoreResult(hospital, patientMaster, user, result);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(LogMessages.ErrorUnableToConnectDoesPcehrExist, ex, user, LogMessage.HIPS_MESSAGE_089);
                throw ex;
            }
            return result;
        }

        /// <summary>
        /// Builds and submits the Does PCEHR Exist web service request, audits the operation, and returns the response.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier (in which the alternate organisation name may be supplied)</param>
        /// <param name="ihiNumber">The ihi number.</param>
        /// <param name="certificate">A NASH-compliant HPI-O eHealth Record certificate</param>
        /// <param name="endpoint">The Does PCEHR Exist service URL</param>
        /// <param name="hospital">The hospital from which this call is being made (note, result differs per HPI-O)</param>
        /// <param name="patientMaster">The patient about which this call is being made</param>
        /// <param name="user">The person responsible for this call</param>
        /// <returns>A HIPS 'does PCEHR exist' response object</returns>
        private DoesPcehrExistResponse InvokeDoesPcehrExist(PatientIdentifierBase patientIdentifier, Hospital hospital, PatientMaster patientMaster, Uri endpoint, X509Certificate2 certificate, UserDetails user)
        {
            DoesPcehrExistResponse result = new DoesPcehrExistResponse();
            bool checkService = bool.Parse(ConfigurationManager.AppSettings["CheckDoesPcehrExist"].ToString());
            if (!checkService)
            {
                result.HipsResponse.Status = HipsResponseIndicator.OK;
                return result;
            }
            CommonPcehrHeader pcehrHeader = Helpers.GetHeader(patientIdentifier, patientMaster.Ihi, user, hospital);

            DoesPCEHRExistClient doesPcehrExistClient = new DoesPCEHRExistClient(endpoint, certificate, certificate);

            ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

            System.Reflection.FieldInfo clientfield = doesPcehrExistClient.GetType().GetField("profileClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            object ppc = clientfield.GetValue(doesPcehrExistClient);
            System.Reflection.FieldInfo clientField2 = ppc.GetType().GetField("pcehrProfileClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            PCEHRProfilePortTypeClient ptc = clientField2.GetValue(ppc) as PCEHRProfilePortTypeClient;
            CustomBinding binding = ptc.Endpoint.Binding as CustomBinding;
            HttpsTransportBindingElement https = binding.Elements.First(a => a.GetType() == typeof(HttpsTransportBindingElement)) as HttpsTransportBindingElement;

            //Set the connection timeout for the DoesPCEHRExist service
            binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DoesPCEHRExistTimeoutSeconds);
            binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DoesPCEHRExistTimeoutSeconds);
            binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DoesPCEHRExistTimeoutSeconds);

            //If avoiding the proxy then set the UseDefaultWebProxy to false
            if (Settings.Instance.AvoidProxy)
            {
                https.UseDefaultWebProxy = false;
            }

            StringBuilder errorMessages = new StringBuilder();

            try
            {
                // Invoke the service
                doesPCEHRExistResponse response = doesPcehrExistClient.DoesPCEHRExist(pcehrHeader);
                result.HipsResponse.Status = HipsResponseIndicator.OK;
                result.PcehrExists = response.PCEHRExists;
                if (response.accessCodeRequiredSpecified)
                {
                    switch (response.accessCodeRequired)
                    {
                        case doesPCEHRExistResponseAccessCodeRequired.AccessGranted:
                            result.AccessCodeRequired = AccessCodeRequired.AccessGranted;
                            break;

                        case doesPCEHRExistResponseAccessCodeRequired.WithCode:
                            result.AccessCodeRequired = AccessCodeRequired.WithCode;
                            break;

                        case doesPCEHRExistResponseAccessCodeRequired.WithoutCode:
                            result.AccessCodeRequired = AccessCodeRequired.WithoutCode;
                            break;

                        default:
                            result.AccessCodeRequired = AccessCodeRequired.Unknown;
                            break;
                    }
                }
                else
                {
                    result.AccessCodeRequired = AccessCodeRequired.Unknown;
                }
            }

            catch (FaultException<StandardErrorType> ex)
            {
                errorMessages.Append(string.Format(ResponseStrings.ErrorCodeAndDetailsFormat, ex.Detail.errorCode, ex.Detail.message));
                result.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                result.HipsResponse.HipsErrorMessage = ResponseStrings.UnableToConnectDoesPcehrExist;
                result.HipsResponse.ResponseCode = ex.Detail.errorCode.ToString();
                result.HipsResponse.ResponseCodeDescription = ex.Detail.message;
                string logMessage = string.Format(LogMessages.ErrorUnableToConnectDoesPcehrExist, patientMaster.PatientMasterId);
                EventLogger.WriteLog(logMessage, new Exception(errorMessages.ToString()), user, LogMessage.HIPS_MESSAGE_090);
            }
            catch (Exception ex)
            {
                if (ex.InnerException != null)
                {
                    errorMessages.AppendLine(ex.InnerException.Message);
                    result.HipsResponse.ResponseCodeDetails = ex.InnerException.Message;
                }
                errorMessages.Append(ex.Message);
                result.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                result.HipsResponse.HipsErrorMessage = ResponseStrings.UnableToConnectDoesPcehrExist;
                result.HipsResponse.ResponseCodeDescription = ex.Message;
                string logMessage = string.Format(LogMessages.ErrorUnableToConnectDoesPcehrExist, patientMaster.PatientMasterId);
                EventLogger.WriteLog(logMessage, ex, user, LogMessage.HIPS_MESSAGE_091);
            }
            finally
            {
                doesPcehrExistClient.Close();
            }
            Helpers.InsertAudit(patientMaster, user, hospital, AuditOperationNames.DoesPcehrExist, result.HipsResponse, doesPcehrExistClient.SoapMessages);
            return result;
        }

        /// <summary>
        /// Stores the result of doesPCEHRExist into the database.
        /// </summary>
        /// <param name="hospital">The hospital.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="user">The user.</param>
        /// <param name="result">The response from doesPCEHRExist.</param>
        private static void StoreResult(Hospital hospital, PatientMaster patientMaster, UserDetails user, DoesPcehrExistResponse result)
        {
            if (result.HipsResponse.Status == HipsResponseIndicator.OK)
            {
                HealthProviderOrganisationPatient hpop;
                HealthProviderOrganisationPatientDl dataAccess = new HealthProviderOrganisationPatientDl(user);
                if (dataAccess.Get(hospital.HpiO, patientMaster.PatientMasterId.Value, out hpop))
                {
                    hpop.PcehrAdvertised = result.PcehrExists;
                    hpop.AccessCodeRequiredId = (int)result.AccessCodeRequired;
                    dataAccess.Update(hpop);
                }
                else if (result.PcehrExists.HasValue && result.PcehrExists.Value)
                {
                    hpop.PatientMasterId = patientMaster.PatientMasterId.Value;
                    hpop.HealthProviderOrganisationId = hospital.HealthProviderOrganisationId;
                    hpop.PcehrAdvertised = result.PcehrExists;
                    hpop.AccessCodeRequiredId = (int)result.AccessCodeRequired;
                    hpop.PcehrDisclosed = false;
                    if (!dataAccess.Insert(hpop))
                    {
                        throw new Exception(string.Format(ConstantsResource.DatabaseError, dataAccess.GetType().FullName));
                    }
                }
            }
            else
            {
                // Update the IHI status to Service Unavailable.
                // This will cause the IHI cleanup background process (if configured) to
                // retry the IHI lookup and PCEHR exists check for this patient, until
                // both services are available.

                PatientMasterIhi ihi = new PatientMasterIhi();
                ihi.PatientMasterId = patientMaster.PatientMasterId;
                ihi.Ihi = patientMaster.Ihi;
                ihi.IhiStatusId = (int)IhiStatus.ServiceUnavailable;
                patientMaster.IhiStatusId = ihi.IhiStatusId;
                ihi.IhiRecordStatusId = patientMaster.IhiRecordStatusId;
                ihi.DateLastValidated = patientMaster.IhiLastValidated.Value;
                ihi.RegisteredFamilyName = patientMaster.RegisteredFamilyName ?? patientMaster.CurrentName.FamilyName;
                ihi.RegisteredGivenName = patientMaster.RegisteredGivenName ?? patientMaster.CurrentName.GivenNames;
                ihi.RegisteredSexId = patientMaster.RegisteredSexId;
                ihi.HospitalId = hospital.HospitalId.Value;
                ihi.MedicareNumber = patientMaster.MedicareNumber;
                ihi.MedicareNumberSequence = patientMaster.MedicareIrn;
                ihi.DvaNumber = patientMaster.DvaNumber;
                ihi.DateOfBirth = patientMaster.DateOfBirth;
                PatientMasterIhiDl dataAccess = new PatientMasterIhiDl(user);
                if (!dataAccess.Update(ihi))
                {
                    throw new Exception(string.Format(ConstantsResource.DatabaseError, dataAccess.GetType().FullName));
                }
            }
        }

        #endregion Methods
    }
}