﻿using System;
using System.Configuration;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Transactions;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.PCEHRProfile;

namespace HIPS.PcehrHiBusinessLogic.Pcehr
{
    public static class Helpers
    {
        /// <summary>
        /// Inserts an audit record for a call to PCEHR.
        /// </summary>
        /// <param name="patientMaster">The patient master record</param>
        /// <param name="user">The user who initiated the action</param>
        /// <param name="hospital">The hospital whose participating healthcare provider organisation accessed the PCEHR.</param>
        /// <param name="serviceName">Identifies the PCEHR B2B service</param>
        /// <param name="serviceMessage">Message returned by the PCEHR B2B Gateway</param>
        /// <param name="soapRequest">Full outgoing SOAP message</param>
        /// <param name="soapResponse">Full incoming SOAP message</param>
        /// <returns>Whether the audit record was inserted</returns>
        private static bool InsertAudit(PatientMaster patientMaster, UserDetails user, Hospital hospital, string serviceName, string serviceMessage, string soapRequest, string soapResponse)
        {
            PcehrAudit audit = new PcehrAudit();
            audit.PatientMasterId = patientMaster.PatientMasterId.Value;
            audit.Ihi = patientMaster.Ihi;
            audit.HpiO = hospital.HpiO;
            audit.UserName = user.Name;
            audit.ServiceName = serviceName;
            audit.ServiceMessage = serviceMessage;
            audit.Request = soapRequest;
            audit.Response = soapResponse;

            PcehrAuditDl dataAccess = new PcehrAuditDl(user);
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                return dataAccess.Insert(audit);
            }
        }

        /// <summary>
        /// Inserts an audit record for a call to PCEHR.
        /// </summary>
        /// <param name="patientMaster">The patient master record</param>
        /// <param name="user">The clinical information system user who initiated the action.</param>
        /// <param name="hospital">The hospital whose participating healthcare provider organisation accessed the PCEHR.</param>
        /// <param name="serviceName">Identifies the PCEHR B2B service that was invoked.</param>
        /// <param name="response">The HIPS response, including indicator of success or failure and reason for failure.</param>
        /// <param name="soapMessages">The full outgoing and incoming messages.</param>
        /// <returns>Whether the audit record was inserted.</returns>
        internal static bool InsertAudit(PatientMaster patientMaster, UserDetails user, Hospital hospital, string serviceName, HipsResponse response, SoapMessages soapMessages)
        {
            string serviceMessage = string.Format("{0} {1} {2} {3} {4}", response.Status, response.HipsErrorMessage, response.ResponseCode, response.ResponseCodeDescription, response.ResponseCodeDetails).Trim();
            return InsertAudit(patientMaster, user, hospital, serviceName, serviceMessage, soapMessages.SoapRequest, soapMessages.SoapResponse);
        }

        /// <summary>
        /// XMLs the view model.
        /// </summary>
        /// <param name="xml">The XML.</param>
        /// <returns></returns>
        public static string XmlViewModel(string xml)
        {
            XmlDocument document = new XmlDocument();
            document.Load(new StringReader(xml));

            StringBuilder builder = new StringBuilder();
            using (XmlTextWriter writer = new XmlTextWriter(new StringWriter(builder)))
            {
                writer.Formatting = Formatting.Indented;
                document.Save(writer);
            }

            return builder.ToString();
        }

        /// <summary>
        /// Gets the nash certificate.
        /// </summary>
        /// <param name="hospital">The hospital.</param>
        /// <returns>X509Certificate2</returns>
        /// <exception cref="System.Exception">PCEHR certificate Serial has not been found via the hospital</exception>
        public static X509Certificate2 GetHpioCertificate(Hospital hospital)
        {
            string serialNumber = hospital.PcehrCertSerial;

            //string serialNumber = ConfigurationManager.AppSettings["PcehrHpioCertificate"];
            if (string.IsNullOrEmpty(serialNumber))
            {
                throw new Exception("PCEHR certificate Serial has not been found via the hospital");
            }

            // Obtain the certificate for use with TLS
            return X509CertificateUtil.GetCertificate(serialNumber, X509FindType.FindBySerialNumber, StoreName.My, StoreLocation.LocalMachine, false);
        }

        /// <summary>
        /// Gets the URL for the Does PCEHR Exist web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GetDoesPcehrUrl()
        {
            string url = ConfigurationManager.AppSettings["DoesPCEHRExistUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Does PCEHR Exist Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the Upload Document web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GetUploadDocumentUrl(DocumentType docType)
        {
            string url = null;
            if (docType.RepositoryId == (int)Repository.NPDR)
            {
                url = ConfigurationManager.AppSettings["NpdrUploadDocumentUrl"];
            }
            else if (docType.RepositoryId == (int)Repository.PCEHR)
            {
                url = ConfigurationManager.AppSettings["UploadDocumentUrl"];
            }
            else
            {
                throw new Exception("Repository has not been defined in the Document Type");
            }

            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Upload Document Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the 'Get Document List' web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GetDocumentListUrl()
        {
            string url = ConfigurationManager.AppSettings["ListDocumentUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("List Document Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the 'Get Change History View' web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GetChangeHistoryViewUrl()
        {
            string url = ConfigurationManager.AppSettings["GetChangeHistoryViewUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Get Change History View Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the 'Get Document' web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GetDocumentUrl()
        {
            string url = ConfigurationManager.AppSettings["GetDocumentUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Get Document Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the 'Get View' web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GetViewUrl()
        {
            string url = ConfigurationManager.AppSettings["GetViewUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Get View Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the 'Gain PCEHR Access' web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GainPCEHRAccessUrl()
        {
            string url = ConfigurationManager.AppSettings["GainPCEHRAccessUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Gain PCEHR Access Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the 'Remove Document' web service.
        /// </summary>
        /// <returns>Uri</returns>
        public static Uri GetRemoveDocumentUrl(DocumentType docType)
        {
            string url = null;
            if (docType.RepositoryId == (int)Repository.NPDR)
            {
                url = ConfigurationManager.AppSettings["NpdrRemoveDocumentUrl"];
            }
            else if (docType.RepositoryId == (int)Repository.PCEHR)
            {
                url = ConfigurationManager.AppSettings["RemoveDocumentUrl"];
            }
            else
            {
                throw new Exception("Repository has not been defined in the Document Type");
            }

            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Remove Document Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the URL for the 'Register PCEHR' web service.
        /// </summary>
        /// <returns></returns>
        /// <exception cref="System.Exception">Register PCEHR Url has not been defined in the App.Config</exception>
        public static Uri GetRegisterPcehrUrl()
        {
            string url = ConfigurationManager.AppSettings["RegisterPcehrUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Register PCEHR Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Constructs the common PCEHR header that identifies the clinical information system,
        /// the consumer whose PCEHR is to be accessed, the individual responsible for the
        /// action, and the healthcare provider organisation that is accessing the PCEHR.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier which contains the alternate organisation name</param>
        /// <param name="ihi">The IHI of the consumer whose PCEHR is to be accessed.</param>
        /// <param name="user">The individual responsible for the action.</param>
        /// <param name="hospital">The hospital whose linked Healthcare Provider Organisation is accessing the PCEHR.</param>
        public static CommonPcehrHeader GetHeader(PatientIdentifierBase patientIdentifier, string ihi, UserDetails user, Hospital hospital)
        {
            CommonPcehrHeader pcehrHeader = new CommonPcehrHeader();
            if (user.Role == UserRole.ProviderIndividual)
            {
                pcehrHeader.UserIdType = CommonPcehrHeaderUserIDType.HPII;
                pcehrHeader.UserId = user.HpiI;
            }
            else if (user.Role == UserRole.InteractiveUser)
            {
                pcehrHeader.UserIdType = CommonPcehrHeaderUserIDType.LocalSystemIdentifier;
                pcehrHeader.UserId = user.Login;
            }
            else if (user.Role == UserRole.AuthorisedEmployee)
            {
                pcehrHeader.UserIdType = CommonPcehrHeaderUserIDType.LocalSystemIdentifier;
                pcehrHeader.UserId = user.AuthorisedEmployeeUserId;
            }
            else
            {
                throw new Exception("Unknown User Role");
            }

            pcehrHeader.UserName = user.Name;
            pcehrHeader.IhiNumber = ihi;
            if (patientIdentifier != null && !string.IsNullOrWhiteSpace(patientIdentifier.AlternateOrganisationName))
            {
                pcehrHeader.AlternateOrganisationName = patientIdentifier.AlternateOrganisationName.Trim();
            }
            pcehrHeader.OrganisationName = hospital.HpioName;
            pcehrHeader.OrganisationId = hospital.HpiO;

            pcehrHeader.ClientSystemType = CommonPcehrHeaderClientSystemType.CIS;
            pcehrHeader.ProductName = ConfigurationManager.AppSettings["PcehrProductName"];
            pcehrHeader.ProductPlatform = System.Environment.OSVersion.ToString();
            pcehrHeader.ProductVersion = ConfigurationManager.AppSettings["PcehrProductVersion"];
            pcehrHeader.ProductVendor = ConfigurationManager.AppSettings["PcehrVendorId"];
            return pcehrHeader;
        }

        /// <summary>
        /// Extracts the ihi from the fully qualified string.
        /// </summary>
        /// <param name="ihi">The ihi.</param>
        /// <returns>IHI Value</returns>
        public static string ExtractIhi(string ihi)
        {
            if (string.IsNullOrEmpty(ihi))
            {
                return string.Empty;
            }
            string result = ihi;
            if (ihi != null && ihi.StartsWith(HIQualifiers.IHIQualifier))
            {
                result = ihi.Substring(HIQualifiers.IHIQualifier.Length);
            }

            return result;
        }

        /// <summary>
        /// Inserts the ihi identifier in fromt of the IHI if required
        /// This should not need to be done, but for some reason the Medicare
        /// API needs it in the IHI field.
        /// </summary>
        /// <param name="ihi">The ihi.</param>
        /// <returns></returns>
        public static string InsertIhiIdentifier(string ihi)
        {
            string result = ihi;
            if (ihi != null && !ihi.StartsWith(HIQualifiers.IHIQualifier))
            {
                result = HIQualifiers.IHIQualifier + ihi;
            }

            return result;
        }

        /// <summary>
        /// Clones the specified source.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="source">The source.</param>
        /// <returns></returns>
        public static T Clone<T>(this T source)
        {
            if (!typeof(T).IsSerializable)
            {
                throw new ArgumentException("The type must be serializable.", "source");
            }

            // Don't serialize a null object, simply return the default for that object
            if (Object.ReferenceEquals(source, null))
            {
                return default(T);
            }

            IFormatter formatter = new BinaryFormatter();
            Stream stream = new MemoryStream();
            using (stream)
            {
                formatter.Serialize(stream, source);
                stream.Seek(0, SeekOrigin.Begin);
                return (T)formatter.Deserialize(stream);
            }
        }

        /// <summary>
        /// Updates the specified target item.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="targetItem">The target item.</param>
        /// <param name="sourceItem">The source item.</param>
        public static void Update<T>(T targetItem, T sourceItem)
        {
            PropertyInfo[] fromFields = typeof(T).GetProperties();
            for (int count = 0; count < fromFields.Length; count++)
            {
                PropertyInfo fromField = (PropertyInfo)fromFields[count];
                if (fromField.CanWrite)
                {
                    object value = fromField.GetValue(sourceItem, null);
                    fromField.SetValue(targetItem, value, null);
                }
            }
        }

        /// <summary>
        /// Maps from the Nehta.VendorLibrary.PCEHR.PCEHRProfile.recordStatusType to the HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus
        /// </summary>
        /// <param name="pcehrIhiRecordStatus"></param>
        /// <returns>IhiRecordStatus</returns>
        public static IhiRecordStatus MaptoHIPSIhiRecordStatus(recordStatusType pcehrIhiRecordStatus)
        {
            switch (pcehrIhiRecordStatus)
            {
                case recordStatusType.Verified:
                    return IhiRecordStatus.Verified;

                case recordStatusType.Unverified:
                    return IhiRecordStatus.Unverified;

                default:
                    return IhiRecordStatus.Unknown;
            }
        }

        /// <summary>
        /// Maps from the Nehta.VendorLibrary.PCEHR.PCEHRProfile.statusType to the HIPS.PcehrDataStore.Schemas.Enumerators.IhiStatus
        /// </summary>
        /// <param name="pcehrIhiStatus"></param>
        /// <returns>IhiStatus</returns>
        public static IhiStatus MaptoHIPSIhiStatus(statusType pcehrIhiStatus)
        {
            switch (pcehrIhiStatus)
            {
                case statusType.Active:
                    return IhiStatus.Active;

                case statusType.Deceased:
                    return IhiStatus.Deceased;

                case statusType.Expired:
                    return IhiStatus.Expired;

                case statusType.Resolved:
                    return IhiStatus.Resolved;

                case statusType.Retired:
                    return IhiStatus.Retired;

                default:
                    return IhiStatus.Unknown;
            }
        }
    }
}