﻿using System.Runtime.Serialization;

using EntLibVal = Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.ServiceContracts.Common.DTO
{
    /// <summary>
    /// International Address DTO. This object represents a physical address, which may be a
    /// street address or a postal address. Validations based on HI Service requirements.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    [EntLibVal.Validators.HasSelfValidation]
    public class InternationalAddress
    {
        #region Properties

        /// <summary>
        /// Gets or sets the unstructured address line of the international address.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 120)]
        [EntLibVal.Validators.IgnoreNulls]
        public string AddressLine { get; set; }

        /// <summary>
        /// Gets or sets the international state or province name.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 40)]
        [EntLibVal.Validators.IgnoreNulls]
        public string StateProvince { get; set; }

        /// <summary>
        /// Gets or sets the international postcode.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 10)]
        [EntLibVal.Validators.IgnoreNulls]
        public string Postcode { get; set; }

        /// <summary>
        /// Gets or sets the country code. Refer to TECH.SIS.HI.02 for values.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 4)]
        [EntLibVal.Validators.IgnoreNulls]
        public string CountryCode { get; set; }

        /// <summary>
        /// Gets or sets the country name. Refer to TECH.SIS.HI.02 for values.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 256)]
        [EntLibVal.Validators.IgnoreNulls]
        public string CountryName { get; set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Validate the instance is valid.
        /// </summary>
        /// <param name="results">Validation results to add results of validation to.</param>
        [EntLibVal.Validators.SelfValidation]
        protected void Validate(EntLibVal.ValidationResults results)
        {
            if (string.IsNullOrEmpty(this.CountryName) && string.IsNullOrEmpty(this.CountryCode))
            {
                results.AddResult(new EntLibVal.ValidationResult("Either CountryCode or CountryName must be supplied.", this, "InternationalAddress", string.Empty, null));
            }
        }

        #endregion Methods
    }
}