﻿using AutoMapper;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using DataSchemas = HIPS.PcehrDataStore.Schemas;

using ServiceSchemas = HIPS.PcehrSchemas;

namespace HIPS.AppServer.ServiceHost.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the ClinicalDocument entity.
    /// </summary>
    internal class ClinicalDocumentProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            //Data --> DTO:
            this.CreateMap<DataSchemas.LocalClinicalDocumentMetaData, ServiceSchemas.LocalClinicalDocumentMetaData>()
                .ForMember(dest => dest.Status, opt => opt.ResolveUsing<DocumentStatusResolver>().FromMember(src => src))
                .ForMember(dest => dest.DocumentType, opt => opt.ResolveUsing(src => src.DocumentTypeDescription));

            this.CreateMap<DataSchemas.LocalClinicalDocumentMetaData, HIPS.ServiceContracts.Common.DTO.LocalClinicalDocumentMetaData>()
                .ForMember(dest => dest.Status, opt => opt.ResolveUsing<DocumentStatusResolver>().FromMember(src => src))
                .ForMember(dest => dest.DocumentType, opt => opt.ResolveUsing(src => src.DocumentTypeDescription));
        }

        /// <summary>
        /// Resolves the value of the LocalDocumentStatus
        /// </summary>
        internal class DocumentStatusResolver : ValueResolver<DataSchemas.LocalClinicalDocumentMetaData, LocalDocumentStatus>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override LocalDocumentStatus ResolveCore(DataSchemas.LocalClinicalDocumentMetaData source)
            {
                LocalDocumentStatus returnStatus;

                // Check what type of operation is on the queue
                switch (source.QueueOperationId)
                {
                    // An upload operation is on the queue
                    case (int)QueueOperation.UploadOrSupersede:
                        // Check the status of the upload operation
                        switch (source.QueueStatusId)
                        {
                            //success
                            case (int)QueueStatus.Success:
                                returnStatus = LocalDocumentStatus.Uploaded;
                                break;
                            //pending
                            case (int)QueueStatus.Pending:
                                returnStatus = LocalDocumentStatus.PendingUpload;
                                break;
                            //failure
                            case (int)QueueStatus.Failure:
                                returnStatus = LocalDocumentStatus.FailedUpload;
                                break;
                            //unknown queue status
                            default:
                                returnStatus = LocalDocumentStatus.Unknown;
                                break;
                        }

                        break;

                    // An remove operation is on the queue
                    case (int)QueueOperation.Remove:
                        // Check the status of the remove operation
                        switch (source.QueueStatusId)
                        {
                            //success
                            case (int)QueueStatus.Success:
                                returnStatus = LocalDocumentStatus.Removed;
                                break;
                            //pending
                            case (int)QueueStatus.Pending:
                                returnStatus = LocalDocumentStatus.PendingRemove;
                                break;
                            //failure
                            case (int)QueueStatus.Failure:
                                returnStatus = LocalDocumentStatus.FailedRemove;
                                break;
                            //unknown queue status
                            default:
                                returnStatus = LocalDocumentStatus.Unknown;
                                break;
                        }
                        break;

                    // There are no operations on the queue
                    default:
                        // Check the true status of the clinical document
                        switch (source.ClinicalDocumentStatusId)
                        {
                            // Active (uploaded and not removed)
                            case (int)ClinicalDocumentStatus.Active:
                                returnStatus = LocalDocumentStatus.Uploaded;
                                break;
                            // Removed
                            case (int)ClinicalDocumentStatus.Removed:
                                returnStatus = LocalDocumentStatus.Removed;
                                break;
                            // Unknown clinical document status.
                            default:
                                returnStatus = LocalDocumentStatus.Unknown;
                                break;
                        }
                        break;
                }

                return returnStatus;
            }

            #endregion Methods
        }

        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion Methods
    }
}