﻿using System;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Hpii;
using HIPS.CommonSchemas;
using HIPS.HpiiSchemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.ServiceInterfaces;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for searching for a healthcare provider individual using the HI Service.
    /// </summary>
    public partial class HpiiService : BaseService, IHpiiService
    {
        /// <summary>
        /// Retrieves the results of a batch of HPI-I queries from the HI Service.
        /// </summary>
        /// <param name="hpiiBatchAsyncRetrieveRequest">The batch retrieval request.</param>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <returns>Batch retrieval response</returns>
        public HpiiBatchAsyncRetrieveResponse HpiiBatchRetrieve(HpiiBatchAsyncRetrieveRequest hpiiBatchAsyncRetrieveRequest, UserDetails user)
        {
            try
            {
                return new HpiiSearch().HpiiBatchRetrieve(hpiiBatchAsyncRetrieveRequest, user);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToRetrieveHpiiQueryBatch, 
                    ex, 
                    user,
                    LogMessage.HIPS_MESSAGE_176);
                return new HpiiBatchAsyncRetrieveResponse { HipsResponse = new HipsResponse(ex) };
            }
        }

        /// <summary>
        /// Submits a batch of HPI-I queries to the HI Service.
        /// </summary>
        /// <param name="hpiiBatchAsyncSubmitRequest">The HPI-I batch asynchronous submit request.</param>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <returns>Batch submission response</returns>
        public HpiiBatchAsyncSubmitResponse HpiiBatchSubmit(HpiiBatchAsyncSubmitRequest hpiiBatchAsyncSubmitRequest, UserDetails user)
        {
            try
            {
                return new HpiiSearch().HpiiBatchSubmit(hpiiBatchAsyncSubmitRequest, user);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToSubmitHpiiQueryBatch, 
                    ex, 
                    user, 
                    LogMessage.HIPS_MESSAGE_177);
                return new HpiiBatchAsyncSubmitResponse { HipsResponse = new HipsResponse(ex) };
            }
        }

        /// <summary>
        /// Performs an HPI-I Demographic query against the HPI-I Service.
        /// </summary>
        /// <param name="hpiiDemographicQuery">The HPII demographic query.</param>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <returns>Found HPII information</returns>
        public HpiiQueryResponse HpiiDemographicSearch(HpiiDemographicQuery hpiiDemographicQuery, UserDetails user)
        {
            try
            {
                return new HpiiSearch().HpiiDemographicSearch(hpiiDemographicQuery, user);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToPerformHpiiDemographicSearch, 
                    ex, 
                    user, 
                    LogMessage.HIPS_MESSAGE_178);
                return new HpiiQueryResponse { HipsResponse = new HipsResponse(ex) };
            }
        }

        /// <summary>
        /// Performs an HPI-I Validation query against the HI Service.
        /// </summary>
        /// <param name="hpiiIdentifierQuery">The HPII identifier query.</param>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <returns>Validated HPII information</returns>
        public HpiiQueryResponse HpiiValidation(HpiiIdentifierQuery hpiiIdentifierQuery, UserDetails user)
        {
            try
            {
                return new HpiiSearch().HpiiValidation(hpiiIdentifierQuery, user);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToPerformHpiiValidation, 
                    ex,
                    user, 
                    LogMessage.HIPS_MESSAGE_179);
                return new HpiiQueryResponse { HipsResponse = new HipsResponse(ex) };
            }
        }
    }
}