﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;

using HIPS.AppServer.HIPSServiceHost.Services;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Hpii;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.HpiiSchemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Common.DTO;
using HIPS.ServiceContracts.Common.Fault;
using HIPS.ServiceContracts.Hpii;
using HIPS.ServiceContracts.Hpii.DTO;
using HIPS.ServiceContracts.Hpii.Message;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for searching for a healthcare provider individual using the HI Service.
    /// </summary>
    public partial class HpiiService : BaseService, IHpiiServiceV2
    {
        /// <summary>
        /// This operation retrieves the results of a batch of HPI-I searches from the HI Service.
        /// </summary>
        /// <param name="request">HpiiBatchRetrieveRequest request parameters.</param>
        /// <returns>Response including the results of the batch.</returns>
        public HpiiBatchRetrieveResponse HpiiBatchRetrieve(HpiiBatchRetrieveRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiBatchAsyncRetrieveRequest>(request);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiBatchRetrieve(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<BatchRetrieveResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiBatchRetrieveResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToRetrieveHpiiQueryBatch, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_176);
                throw error;
            }
        }

        /// <summary>
        /// This operation submits a batch of HPI-I searches to the HI Service.
        /// </summary>
        /// <param name="request">HpiiBatchSubmitRequest request parameters.</param>
        /// <returns>Response including the batch identifier.</returns>
        public HpiiBatchSubmitResponse HpiiBatchSubmit(HpiiBatchSubmitRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiBatchAsyncSubmitRequest>(request);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiBatchSubmit(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<BatchSubmitResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiBatchSubmitResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToSubmitHpiiQueryBatch, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_177);
                throw error;
            }
        }

        /// <summary>
        /// This operation submits an HPI-I demographic search to the HI Service.
        /// </summary>
        /// <param name="request">HpiiDemographicSearchRequest request parameters.</param>
        /// <returns>Response including the search results.</returns>
        public HpiiDemographicSearchResponse HpiiDemographicSearch(HpiiDemographicSearchRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiDemographicQuery>(request.Query);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiDemographicSearch(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<SearchResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiDemographicSearchResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToPerformHpiiDemographicSearch, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_178);
                throw error;
            }
        }

        /// <summary>
        /// This operation submits an HPI-I identifier search to the HI Service.
        /// </summary>
        /// <param name="request">HpiiIdentifierSearchRequest request parameters.</param>
        /// <returns>Response including the search results.</returns>
        public HpiiIdentifierSearchResponse HpiiIdentifierSearch(HpiiIdentifierSearchRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiIdentifierQuery>(request.Query);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiValidation(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<SearchResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiIdentifierSearchResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToPerformHpiiValidation, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_179);
                throw error;
            }
        }

        /// <summary>
        /// List all of the current local providers for the specified facility.
        /// </summary>
        /// <param name="request">List Local Provider request parameters.</param>
        /// <returns>List Local Provider response.</returns>
        public ListLocalProviderResponse ListLocalProviders(ListLocalProviderRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;
            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var hospitalIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.HospitalIdentifier>(request.HospitalIdentifier);

                // Invoke Business Service
                List<HospitalHealthProviderIndividual> dataProviderIdentifiers;
                List<HealthProviderIndividual> dataProviders;

                var logic = new HpiiLocalManagement();
                var hipsResponse = logic.ListLocalProviders(hospitalIdentifier, user, out dataProviders, out dataProviderIdentifiers);
                if (hipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(hipsResponse);
                }
                var messages = new List<string>() { hipsResponse.HipsErrorMessage, hipsResponse.ResponseCodeDescription, hipsResponse.ResponseCodeDetails };

                var localProviders = ObjectMapper.Map<List<LocalProvider>>(dataProviders);
                var localProviderIdentifiers = ObjectMapper.Map<List<LocalProviderIdentifier>>(dataProviderIdentifiers);
                foreach (var localProvider in localProviders)
                {
                    localProvider.LocalProviderIdentifiers = localProviderIdentifiers.Where(i => i.LocalProviderCode != null && i.LocalProviderCode == localProvider.LocalProviderCode).ToList();
                }

                // Map Common Model to DTO
                var serviceResult = localProviders;
                var serviceMessages = ObjectMapper.Map<List<ServiceContracts.Common.DTO.Message>>(messages.Where(message => !string.IsNullOrEmpty(message)));
                var serviceStatus = (hipsResponse.Status == HipsResponseIndicator.OK ? ServiceContracts.Common.ResponseStatus.OK : ServiceContracts.Common.ResponseStatus.Warning);

                return new ListLocalProviderResponse() { Status = serviceStatus, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToListLocalProviders, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_192);
                throw error;
            }
        }

        /// <summary>
        /// Add or Edit a Local Provider.
        /// </summary>
        /// <param name="request">Add or Edit Local Provider request.</param>
        /// <returns>Add or Edit Local Provider response.</returns>
        public AddEditLocalProviderResponse AddEditLocalProvider(AddEditLocalProviderRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common object
                user = ObjectMapper.Map<UserDetails>(request.User);
                var hospitalIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.HospitalIdentifier>(request.HospitalIdentifier);

                // Call Business Logic
                MessageLevel serviceMessageLevelCode;
                var logic = new HpiiLocalManagement();
                var hipsResponse = logic.AddEditLocalProviders(hospitalIdentifier, request.LocalProviderCode, request.IamProviderIdentifier, request.Hpii, request.FamilyName, request.GivenNames, request.Suffix, request.Title, user);
                switch (hipsResponse.Status)
                {
                    case HipsResponseIndicator.OK:
                        serviceMessageLevelCode = MessageLevel.Information; // Information
                        break;
                    case HipsResponseIndicator.HpiiWarningRaised:
                        serviceMessageLevelCode = MessageLevel.Warning; // Warning
                        break;
                    case HipsResponseIndicator.UnresolvedHpiiAlert:
                        serviceMessageLevelCode = MessageLevel.Error; // Error
                        break;
                    default: // if we got anything else then throw a HipsResponse Error
                        throw new HipsResponseException(hipsResponse);
                }
                
                // Map Common Model to DTO
                var message = new Message()
                {
                    Code = hipsResponse.ResponseCode,
                    Description = hipsResponse.HipsErrorMessage,
                    Level = (MessageLevel)serviceMessageLevelCode
                };
                var serviceMessages = (new Message[] { message }).Where(m => !string.IsNullOrEmpty(m.Description)).ToList();
                var serviceStatus = (hipsResponse.Status == HipsResponseIndicator.OK ? ServiceContracts.Common.ResponseStatus.OK : ServiceContracts.Common.ResponseStatus.Warning);

                return new AddEditLocalProviderResponse() { Status = serviceStatus, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToAddOrEditLocalProvider, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_193);
                throw error;
            }
        }

        /// <summary>
        /// Add or Replace a Local Provider Identifier.
        /// </summary>
        /// <param name="request">Add or Replace Local Provider Identifier request.</param>
        /// <returns>Add or Replace Local Provider Identifier response.</returns>
        public AddReplaceLocalProviderIdentifierResponse AddReplaceLocalProviderIdentifier(AddReplaceLocalProviderIdentifierRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common object
                user = ObjectMapper.Map<UserDetails>(request.User);
                var hospitalIdentifier = new HIPS.CommonSchemas.HospitalIdentifier(request.LocalProviderIdentifier.HospitalCode, request.LocalProviderIdentifier.HospitalCodeSystem);

                // Invoke Business Service
                var logic = new HpiiLocalManagement();
                var hipsResponse = logic.AddReplaceLocalProviderIdentifier(hospitalIdentifier, request.LocalProviderIdentifier.LocalProviderCode, request.LocalProviderIdentifier.PasProviderIdentifier, request.LocalProviderIdentifier.CisProviderIdentifier, request.Replace, user);
                if (hipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(hipsResponse);
                }
                var messages = new List<string>() { hipsResponse.HipsErrorMessage, hipsResponse.ResponseCodeDescription, hipsResponse.ResponseCodeDetails };

                // Map Common Model to DTO
                var serviceMessages = ObjectMapper.Map<List<ServiceContracts.Common.DTO.Message>>(messages.Where(message => !string.IsNullOrEmpty(message)));
                var serviceStatus = (hipsResponse.Status == HipsResponseIndicator.OK ? ServiceContracts.Common.ResponseStatus.OK : ServiceContracts.Common.ResponseStatus.Warning);

                return new AddReplaceLocalProviderIdentifierResponse() { Status = serviceStatus, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToAddOrReplaceLocalProviderIdentifier, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_194);
                throw error;
            }
        }

        /// <summary>
        /// Deactivates or removes deactivation a local provider.
        /// </summary>
        /// <param name="request">Deactivate Local Provider request parameters.</param>
        /// <returns>Deactivate Local Provider response.</returns>
        public DeactivateLocalProviderResponse DeactivateLocalProvider(DeactivateLocalProviderRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common object
                user = ObjectMapper.Map<UserDetails>(request.User);
                var hospitalIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.HospitalIdentifier>(request.HospitalIdentifier);

                // Invoke Business Service
                var logic = new HpiiLocalManagement();
                var hipsResponse = logic.DeactivateLocalProvider(hospitalIdentifier, request.LocalProviderCode, request.DeactivatedDate, user);
                if (hipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(hipsResponse);
                }
                var messages = new List<string>() { hipsResponse.HipsErrorMessage, hipsResponse.ResponseCodeDescription, hipsResponse.ResponseCodeDetails };

                // Map Common Model to DTO
                var serviceMessages = ObjectMapper.Map<List<ServiceContracts.Common.DTO.Message>>(messages.Where(message => !string.IsNullOrEmpty(message)));
                var serviceStatus = (hipsResponse.Status == HipsResponseIndicator.OK ? ServiceContracts.Common.ResponseStatus.OK : ServiceContracts.Common.ResponseStatus.Warning);

                return new DeactivateLocalProviderResponse() { Status = serviceStatus, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                var error = ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
                EventLogger.WriteLog(
                    string.Format("{0} FaultIdentifier: [{1}]", ResponseStrings.ErrorUnableToDeactivateLocalProvider, error.Detail.FaultIdentifier),
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_195);
                throw error;
            }
        }

        /// <summary>
        /// Builds a list of HIPS service messages from the NEHTA ServiceMessagesType object.
        /// </summary>
        /// <param name="nehtaMessages">The list of messages returned from the HI Service.</param>
        /// <returns>The HIPS service message list.</returns>
        private List<Message> BuildMessageList(nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType nehtaMessages)
        {
            var messages = new List<Message>();
            if (nehtaMessages != null)
            {
                messages.AddRange(ObjectMapper.Map<List<Message>>(nehtaMessages.serviceMessage));
            }
            return messages;
        }

        
    }
}