﻿using System;
using System.Collections.Generic;
using System.ServiceModel;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Cda;
using HIPS.CommonBusinessLogic.Consent;
using HIPS.CommonBusinessLogic.Service;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.Common.Fault;
using HIPS.ServiceContracts.Pcehr;
using HIPS.ServiceContracts.Pcehr.Message;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.PcehrSchemas;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for uploading, removing, listing and downloading documents from the PCEHR Repository.
    /// </summary>
    public partial class PCEHRService : IPcehrServiceV2
    {
        #region Methods

        /// <summary>
        /// Uploads a PDF discharge summary to the PCEHR National Repository, optionally superseding a previously-uploaded document.
        /// This takes in the PDF document, required metadata, and any attachments and creates a CDA document before packaging it.
        /// </summary>
        /// <param name="request">Request containing a PDF and required metadata for creating the CDA package.</param>
        /// <returns>Response indicating that the request was queued or the reason why not.</returns>
        public UploadDischargeSummaryLevel1AResponse UploadOrSupersedeDischargeSummaryLevel1A(UploadDischargeSummaryLevel1ARequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;
            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var metadata = ObjectMapper.Map<HIPS.CommonSchemas.Cda.CdaHeaderMetadata>(request.CdaHeaderMetadata);
                var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(request.PatientIdentifier);
                var attachments = ObjectMapper.Map<List<HIPS.CommonSchemas.Cda.CdaAttachment>>(request.Attachments);

                // Invoke Business Service
                var businessService = new DischargeSummaryService(user);
                var businessResponse = businessService.UploadDischargeSummaryLevel1A(request.PdfDocument, metadata, patientIdentifier, attachments);

                // Map Common Model to DTO
                var svcResponse = ObjectMapper.Map<UploadDischargeSummaryLevel1AResponse>(businessResponse);
                return svcResponse;
            }
            catch (PcehrServiceException ex)
            {
                // Catch any business exceptions, log them in the “SystemErrorLog”
                // and map them to SOAP faults using AutoMapper. Ensure that the
                // message or code in each business exception is returned in the
                // SOAP fault.
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<PcehrServiceFault>>(ex);
            }
            catch (HiServiceException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<HiServiceFault>>(ex);
            }
            catch (ItemNotFoundException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<ItemNotFoundFault>>(ex);
            }
            catch (InvalidUserException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<InvalidUserFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// This operation looks up a patient record in the current facility (for advertised status) and disclosure facility (for disclosed status),
        /// gets a validated IHI for the patient, and optionally refreshes the advertised status from the PCEHR system.
        /// </summary>
        /// <param name="request">RefreshPatientParticipationStatusRequest request parameters</param>
        /// <returns>Response indicating the participation status of the patient.</returns>
        public RefreshPatientParticipationStatusResponse RefreshPatientParticipationStatus(RefreshPatientParticipationStatusRequest request)
        {
            // 1. Validate the request using Enterprise Library Validation.
            ValidateRequest(request);

            // 2. Map the DTOs in the request into common schema objects using AutoMapper.
            var user = ObjectMapper.Map<UserDetails>(request.User);
            var disclosureFacility = ObjectMapper.Map<HIPS.CommonSchemas.HospitalIdentifier>(request.DisclosureFacility);
            var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(request.PatientIdentifier);
            var forceRefresh = ObjectMapper.Map<HIPS.CommonSchemas.ForceRefreshOption>(request.ForceRefresh);

            try
            {
                // 3. Create an instance of the “CheckPatientParticipation” business logic class.
                CheckPatientParticipation businessService = new CheckPatientParticipation();

                // 4. Invoke the “RefreshPatientParticipationStatus” method.
                var businessResponse = businessService.RefreshPatientParticipationStatus(patientIdentifier, disclosureFacility, forceRefresh, user);

                // 5. Map the business response to the service response using AutoMapper.
                var svcResponse = ObjectMapper.Map<RefreshPatientParticipationStatusResponse>(businessResponse);
                return svcResponse;
            }
            catch (PcehrServiceException ex)
            {
                // 6. Catch any business exceptions, log them in the “SystemErrorLog”
                // and map them to SOAP faults using AutoMapper. Ensure that the message
                // or code in each business exception is returned in the SOAP fault.
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingPatientParticipationStatus, ex, user, LogMessage.HIPS_MESSAGE_175);
                throw ObjectMapper.Map<FaultException<PcehrServiceFault>>(ex);
            }
            catch (HiServiceException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingPatientParticipationStatus, ex, user, LogMessage.HIPS_MESSAGE_175);
                throw ObjectMapper.Map<FaultException<HiServiceFault>>(ex);
            }
            catch (ItemNotFoundException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingPatientParticipationStatus, ex, user, LogMessage.HIPS_MESSAGE_175);
                throw ObjectMapper.Map<FaultException<ItemNotFoundFault>>(ex);
            }
            catch (InvalidUserException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingPatientParticipationStatus, ex, user, LogMessage.HIPS_MESSAGE_175);
                throw ObjectMapper.Map<FaultException<InvalidUserFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingPatientParticipationStatus, ex, user, LogMessage.HIPS_MESSAGE_175);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Determines whether a PCEHR is advertised looking at local data only.
        /// </summary>
        /// <param name="request">IsPcehrAdvertisedLocalRequest request.</param>
        /// <returns>Response inditicating if the patients PCEHR is advertised and the current access code</returns>
        public IsPcehrAdvertisedLocalResponse IsPcehrAdvertisedLocal(IsPcehrAdvertisedLocalRequest request)
        {
            // 1. Validate the request using Enterprise Library Validation.
            ValidateRequest(request);

            // 2. Map the DTOs in the request into common schema objects using AutoMapper.
            var user = ObjectMapper.Map<UserDetails>(request.User);
            var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(request.PatientIdentifier);

            try
            {
                // 3. Create an instance of the “DoesPcehrExist” business logic class.
                var businessService = new DoesPcehrExist();

                // 4. Invoke the “IsPcehrAdvertised” method.
                var businessResponse = businessService.IsPcehrAdvertised(patientIdentifier, DateTime.MinValue, user, true);

                if (businessResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResponse.HipsResponse);
                }
                
                // 5. Map the business response to the service response using AutoMapper.
                var svcResponse = ObjectMapper.Map<IsPcehrAdvertisedLocalResponse>(businessResponse);

                return svcResponse;
            }
            catch (ItemNotFoundException ex)
            {
                // 6. Catch any business exceptions, log them in the “SystemErrorLog”
                // and map them to SOAP faults using AutoMapper. Ensure that the message
                // or code in each business exception is returned in the SOAP fault.
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingIsPcehrAdvertised, ex, user, LogMessage.HIPS_MESSAGE_180);
                throw ObjectMapper.Map<FaultException<ItemNotFoundFault>>(ex);
            }
            catch (InvalidUserException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingIsPcehrAdvertised, ex, user, LogMessage.HIPS_MESSAGE_180);
                throw ObjectMapper.Map<FaultException<InvalidUserFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorRetrievingIsPcehrAdvertised, ex, user, LogMessage.HIPS_MESSAGE_180);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Adds a pathology document onto the PcehrQueue so that eHISC can upload it to the PCEHR.
        /// </summary>
        /// <param name="request">UploadOrRemovePathologyRequest request.</param>
        /// <returns>Response indicating if the upload or remove pathology is successful</returns>
        public UploadOrRemovePathologyResponse UploadOrRemovePathology(UploadOrRemovePathologyRequest request)
        {
            // Validate the request using Enterprise Library Validation.
            ValidateRequest(request);

            // Map the DTOs in the request into common schema objects using AutoMapper.
            var user = ObjectMapper.Map<UserDetails>(request.User);

            try
            {
                // Invoke the QueuePathology method providing the parsed HL7® message, report or report location.
                var businessService = new PathologyImagingService(user);
                var businessResponse = businessService.QueuePathology(request.HL7Message, request.Report, request.ReportLocation);
                if (businessResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResponse);
                }
                // Map Common Model to DTO
                var svcResponse = ObjectMapper.Map<UploadOrRemovePathologyResponse>(businessResponse);
                return svcResponse;
            }
            catch (PcehrServiceException ex)
            {
                // Catch any business exceptions, log them in the “SystemErrorLog”
                // and map them to SOAP faults using AutoMapper. Ensure that the
                // message or code in each business exception is returned in the
                // SOAP fault.
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadPathologyResultReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<PcehrServiceFault>>(ex);
            }
            catch (HiServiceException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadPathologyResultReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<HiServiceFault>>(ex);
            }
            catch (ItemNotFoundException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadPathologyResultReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<ItemNotFoundFault>>(ex);
            }
            catch (InvalidUserException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadPathologyResultReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<InvalidUserFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadPathologyResultReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Adds a pathology document onto the PcehrQueue so that eHISC can upload it to the PCEHR.
        /// </summary>
        /// <param name="request">UploadOrRemoveImagingRequest request.</param>
        /// <returns>Response inditicating if the upload or remove imaging is successful/returns>
        public UploadOrRemoveImagingResponse UploadOrRemoveImaging(UploadOrRemoveImagingRequest request)
        {
            // 1. Validate the request using Enterprise Library Validation.
            ValidateRequest(request);

            // 2. Map the DTOs in the request into common schema objects using AutoMapper.
            var user = ObjectMapper.Map<UserDetails>(request.User);

            try
            {
                // Invoke the QueueImaging method providing the parsed HL7® message, report or report location.
                var businessService = new PathologyImagingService(user);
                var businessResponse = businessService.QueueImaging(request.HL7Message, request.Report, request.ReportLocation);

                // Map Common Model to DTO
                var svcResponse = ObjectMapper.Map<UploadOrRemoveImagingResponse>(businessResponse);
                return svcResponse;
            }
            catch (PcehrServiceException ex)
            {
                // Catch any business exceptions, log them in the “SystemErrorLog”
                // and map them to SOAP faults using AutoMapper. Ensure that the
                // message or code in each business exception is returned in the
                // SOAP fault.
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDiagnosticImagingExamReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<PcehrServiceFault>>(ex);
            }
            catch (HiServiceException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDiagnosticImagingExamReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<HiServiceFault>>(ex);
            }
            catch (ItemNotFoundException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDiagnosticImagingExamReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<ItemNotFoundFault>>(ex);
            }
            catch (InvalidUserException ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDiagnosticImagingExamReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<InvalidUserFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(HIPS.AppServer.HIPSServiceHost.Services.ResponseStrings.ErrorUnableToUploadDiagnosticImagingExamReport, ex, user, LogMessage.HIPS_MESSAGE_170);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }


        #endregion Methods
    }
}