﻿#region Using Directives

using System;
using System.Collections.Generic;
using System.ServiceModel;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.ConsentSchemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using HIPS.ServiceContracts.ServiceInterfaces;
using Nehta.VendorLibrary.PCEHR;

#endregion Using Directives

namespace HIPS.Client.Proxy
{
    /// <summary>
    /// This pre-built client class can be referenced by a .net client application in lieu of generating client classes from the WSDL.
    /// </summary>
    public class PCEHRProxy : ClientBase<IPCEHRService>, IPCEHRService
    {
        /// <summary>
        /// Default constructor
        /// </summary>
        /// <param name="endpointName"></param>
        public PCEHRProxy(string endpointName)
            : base(endpointName)
        {
        }

        /// <summary>
        /// Requests emergency access to the patient's records on the PCEHR
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>Gain PCEHR Access response.</returns>
        public GainPcehrAccessResponse GainAccessEmergency(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            return Channel.GainAccessEmergency(patientIdentifier, user);
        }

        /// <summary>
        /// Request access with the access code being sent to the PCEHR
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="accessCode">The access code provided by the patient.</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>Gain PCEHR Access response.</returns>
        public GainPcehrAccessResponse GainAccessWithCode(PatientIdentifierBase patientIdentifier, string AccessCode, UserDetails user)
        {
            return Channel.GainAccessWithCode(patientIdentifier, AccessCode, user);
        }

        /// <summary>
        /// Request for access without a code being required by PCEHR
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>Gain PCEHR Access response.</returns>
        public GainPcehrAccessResponse GainAccessWithOutCode(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            return Channel.GainAccessWithOutCode(patientIdentifier, user);
        }

        /// <summary>
        /// Gets the list of uploaded and changed documents from a single document set
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="documentUniqueId">The document entry UUID (NOC requirements disallow using the document ID).</param>
        /// <returns>List of versions of the document.</returns>
        public ChangeHistoryViewResponse<PatientIdentifierBase> GetChangeHistoryView(PatientIdentifierBase patientIdentifier, UserDetails user, string documentUniqueId)
        {
            return Channel.GetChangeHistoryView(patientIdentifier, user, documentUniqueId);
        }

        /// <summary>
        /// Gets the list of all avalible documents.
        /// This function is given the WS name of GetDocumentList.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            return Channel.GetDocumentList(patientIdentifier, user);
        }

        /// <summary>
        /// Gets the list of documents with the ability to filter by start and end date times for document creation and healthcare service
        /// This function is given the WS name of GetDocumentListFilterDates.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="creationTimeStart">Include documents created after this time.</param>
        /// <param name="creationTimeEnd">Include documents created before this time.</param>
        /// <param name="serviceTimeStart">Include documents for healthcare services after this time.</param>
        /// <param name="serviceTimeEnd">Include documents for healthcare services before this time.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd)
        {
            return Channel.GetDocumentList(patientIdentifier, user, creationTimeStart, creationTimeEnd, serviceTimeStart, serviceTimeEnd);
        }

        /// <summary>
        /// Gets the list of documents with the ability to filter by multiple statuses, start and end dates and times for document creation and the healthcare service.
        /// This function is given the WS name of GetDocumentListFilterStatusAndDates.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="documentStatus">Include documents with this status.</param>
        /// <param name="creationTimeStart">Include documents created after this time.</param>
        /// <param name="creationTimeEnd">Include documents created before this time.</param>
        /// <param name="serviceTimeStart">Include documents for healthcare services after this time.</param>
        /// <param name="serviceTimeEnd">Include documents for healthcare services before this time.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, IList<DocumentStatus> documentStatus, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd)
        {
            return Channel.GetDocumentList(patientIdentifier, user, documentStatus, creationTimeStart, creationTimeEnd, serviceTimeStart, serviceTimeEnd);
        }

        /// <summary>
        /// Get the list of documents corresponding to the specified document query.
        /// This function is given the WS name of GetDocumentListWithQuery.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="docQuery">The query parameters to be sent to the document registry for retrieval.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, DocumentQuery docQuery)
        {
            return Channel.GetDocumentList(patientIdentifier, user, docQuery);
        }

        /// <summary>
        /// Gets the list of documents that have the 'Approved' status.
        /// This function is given the WS name of GetDocumentListActive.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentListActive(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            return Channel.GetDocumentListActive(patientIdentifier, user);
        }

        /// <summary>
        /// Gets the list of documents that have the 'Approved' status with the ability to filter by start and end date for document creation and the healthcare service.
        /// This function is given the WS name of GetDocumentListActiveFilterDates.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="creationTimeStart">Include documents created after this time.</param>
        /// <param name="creationTimeEnd">Include documents created before this time.</param>
        /// <param name="serviceTimeStart">Include documents for services after this time.</param>
        /// <param name="serviceTimeEnd">Include documents for services before this time.</param>
        /// <returns>List of documents with response details.</returns>
        public DocumentListResponse<PatientIdentifierBase> GetDocumentListActive(PatientIdentifierBase patientIdentifier, UserDetails user, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd)
        {
            return Channel.GetDocumentListActive(patientIdentifier, user, creationTimeStart, creationTimeEnd, serviceTimeStart, serviceTimeEnd);
        }

        /// <summary>
        /// Gets the status of a single pcehr queue operation for the specified PcehrMessageQueueId.
        /// </summary>
        /// <param name="pcehrMessageQueueId">Identifier of an item in the PCEHR Message Queue.</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>List of queued operations and indicator of success or failure.</returns>
        public IndividualOperationStatus GetIndividualOperationStatus(int pcehrMessageQueueId, UserDetails user)
        {
            return Channel.GetIndividualOperationStatus(pcehrMessageQueueId, user);
        }

        /// <summary>
        /// Get a document uploaded for a specific patient in a defined hospital
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="sourceSystemSetId">The source system set identifier.</param>
        /// <param name="sourceSystemDocumentId">The source system document identifier.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <returns>
        /// Response containing a clinical document uploaded for a specific patient in a defined hospital
        /// </returns>
        public LocalClinicalDocumentResponse<PatientIdentifierBase> GetLocalUploadedDocument(UserDetails user, string sourceSystemSetId, string sourceSystemDocumentId, PatientIdentifierBase patientIdentifier)
        {
            return Channel.GetLocalUploadedDocument(user, sourceSystemSetId, sourceSystemDocumentId, patientIdentifier);
        }

        /// <summary>
        /// Gets the status of all operations for the specified patient and episode.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The episode admission date and time.</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>List of queued operations and indicator of success or failure.</returns>
        public OperationStatus GetOperationStatus(PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user)
        {
            return Channel.GetOperationStatus(patientIdentifier, admissionDate, user);
        }

        /// <summary>
        /// Gets a List of queued Operations
        /// </summary>
        /// <param name="dateTimeFrom">The date time from.</param>
        /// <param name="dateTimeTo">The date time to.</param>
        /// <param name="queueOperation">The queue operation.</param>
        /// <param name="queueStatus">The queue status.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>List of queued operations and indicator of success or failure.</returns>
        public QueuedOperationStatus GetQueuedOperationList(DateTime dateTimeFrom, DateTime dateTimeTo, IList<QueueOperation> queueOperation, IList<QueueStatus> queueStatus, IList<PatientIdentifierBase> patientIdentifier, UserDetails user)
        {
            return Channel.GetQueuedOperationList(dateTimeFrom, dateTimeTo, queueOperation, queueStatus, patientIdentifier, user);
        }

        /// <summary>
        /// Returns the Document and Attachments required to display the requested view.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The person responsible for the action.</param>
        /// <param name="request">The view request.</param>
        /// <returns>A view response containing success and error messages, and if successful, the document and attachments.</returns>
        public ViewResponse GetView(PatientIdentifierBase patientIdentifier, UserDetails user, ViewRequestBase request)
        {
            return Channel.GetView(patientIdentifier, user, request);
        }

        /// <summary>
        /// For a given MRN, State Patient Identifier, PCEHR Data Store PatientMasterId or a validated IHI,
        /// checks whether the patient has advertised the existence of his/her PCEHR,
        /// whether access has been gained, and whether access can be gained without a code.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="dateOfBirth">The patient's date of birth.</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <returns>Response containing operation success indicator, whether advertised, and access status</returns>
        public DoesPcehrExistResponse IsPcehrAdvertised(PatientIdentifierBase patientIdentifier, DateTime dateOfBirth, UserDetails user)
        {
            return Channel.IsPcehrAdvertised(patientIdentifier, dateOfBirth, user);
        }

        /// <summary>
        /// Lists the documents uploaded for a specific patient in a defined hospital
        /// </summary>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="sourceSystemSetId">The source system set identifier.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <returns>Response containing the a list of the admitted or discharged episodes for the patient with in the specified number of days.</returns>
        public LocalClinicalDocumentListResponse<PatientIdentifierBase> ListUploadedDocuments(UserDetails user, string sourceSystemSetId, PatientIdentifierBase patientIdentifier)
        {
            return Channel.ListUploadedDocuments(user, sourceSystemSetId, patientIdentifier);
        }

        /// <summary>
        /// This operation looks up a patient record in the current facility (for
        /// advertised status) and disclosure facility (for disclosed status),
        /// gets a validated IHI for the patient, and optionally refreshes the
        /// advertised status from the PCEHR system.
        /// </summary>
        /// <param name="user">Information to identify the person responsible for
        /// this action</param>
        /// <param name="patientIdentifier">An identifier of the patient and the
        /// current facility.</param>
        /// <param name="disclosureFacility">The Hospital Code and Hospital Code
        /// System of the facility whose healthcare provider organisation records
        /// the PCEHR disclosure status for the patient.</param>
        /// <param name="forceRefresh">The option of always refreshing the
        /// participation status, only if not advertised, or never refreshing the
        /// participation status.</param>
        /// <returns>Participation status for the requested patient.</returns>
        public PatientParticipationResponse RefreshPatientParticipationStatus(
            UserDetails user,
            PatientIdentifierBase patientIdentifier,
            HospitalIdentifier disclosureFacility,
            ForceRefreshOption forceRefresh)
        {
            return Channel.RefreshPatientParticipationStatus(user, patientIdentifier, disclosureFacility, forceRefresh);
        }

        /// <summary>
        /// Removes the specified discharge summary from the PCEHR system.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="admissionDate">The episode admission date and time.</param>
        /// <param name="documentSetId">The clinical document set id.</param>
        /// <param name="reason">Reason for removal.</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="auditInformation">Additional audit information to store.</param>
        /// <returns>Response indicating that the request was queued or the reason why not.</returns>
        public HipsResponse Remove(PatientIdentifierBase patientIdentifier, DateTime admissionDate, string documentSetId, RemovalReason reason, UserDetails user, byte[] auditInformation)
        {
            return Channel.Remove(patientIdentifier, admissionDate, documentSetId, reason, user, auditInformation);
        }

        /// <summary>
        /// Returns the Document and Attachments required to display the requested document.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="request">The request sent to get the document from the PCEHR repository.</param>
        /// <returns>DocumentResponse will also pass back success and error messages</returns>
        public DocumentResponse RetrieveDocument(PatientIdentifierBase patientIdentifier, UserDetails user, DocumentRequest request)
        {
            return Channel.RetrieveDocument(patientIdentifier, user, request);
        }

        /// <summary>
        /// Uploads a CDA document to the PCEHR National Repository, optionally superseding a previously-uploaded document.
        /// This takes in the XML document and any attachments and does the packaging itself.
        /// </summary>
        /// <param name="cdaDocument">The CDA root XML document.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action.</param>
        /// <param name="attachments">The attachments.</param>
        /// <param name="admissionDate">The episode admission date and time.</param>
        /// <param name="documentFormatCode">The document format code (template ID).</param>
        /// <returns>Response indicating that the request was queued or the reason why not.</returns>
        public HipsResponse UploadOrSupersedeDocument(byte[] cdaDocument, PatientIdentifierBase patientIdentifier, UserDetails user, Attachment[] attachments, DateTime admissionDate, string documentFormatCode = null)
        {
            return Channel.UploadOrSupersedeDocument(cdaDocument, patientIdentifier, user, attachments, admissionDate, documentFormatCode);
        }
    }
}