﻿using HIPS.CommonBusinessLogic.Mapping;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.CommonSchemas.Cda.ParticipatingIndividual;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.PcehrDataStore.Schemas;
using Nehta.VendorLibrary.CDA.Common;
using Nehta.VendorLibrary.CDA.Generator;
using Nehta.VendorLibrary.Common;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Text;
using System.Transactions;
using System.Xml;


namespace HIPS.CommonBusinessLogic.Cda
{
    /// <summary>
    /// Implements a business service for Pathology and Diagnostic Imaging allowing the documents to be uploaded to the PCEHR system.
    /// </summary>
    public class PDIUploadService
    {

        #region Declarations

        /// <summary>
        /// The UserDetails object to store the user context object
        /// </summary>
        private UserDetails User;


        #endregion Declarations

        #region Constructor

        /// <summary>
        /// Initialises a new instance of the <see cref="PDIUploadService"/> class.
        /// </summary>
        /// <param name="user">UserDetails object</param>
        public PDIUploadService(UserDetails user)
        {
            this.User = user;
        }

        #endregion Constructor

        #region Methods

        /// <summary>
        /// This method creates the Pathology Result CDA® Report for the purpose of uploading to the PCEHR.
        /// </summary>
        /// <param name="pdfDocument">The Pathology PDF report</param>
        /// <param name="metaData">The header metadata required for the CDA® document</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="orderDetails">The order details for the Pathology report Order</param>
        /// <param name="pathologyTestResult">A list of pathologyTestResult objects</param>
        /// <param name="requestingHospitalPatientMrn">The requesting Hospital’s Patient Mrn</param>
        /// <param name="attachments">A list of CDA® attachments, documents related to the report.</param>
        /// <param name="reportResultStatus">The Overall Report Result Status from RelatedDocumentDetails.ResultStatus</param>
        /// <param name="episode">Patient Episode details</param>
        /// <returns>
        /// The CdaDocumentDetails object.
        /// </returns>
        public CdaPathDIDocumentDetails CreatePathologyResultReport(byte[] pdfDocument, CdaHeaderMetadata metaData, PatientIdentifierBase patientIdentifier,
            OrderDetails orderDetails, List<PathologyTestResult> pathologyTestResult, string requestingHospitalPatientMrn, List<CdaAttachment> attachments, 
            string reportResultStatus, string patientIndigenousStatus, Episode episode)
        {
            // 1. Create a new CdaPathDIDocumentDetails object.
            var cdaPathDIDocumentDetails = new CdaPathDIDocumentDetails();
            var cdaDocumentDetails = new CdaDocumentDetails();
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                try
                {
                    // 2. Assign the metaData parameter to the CdaDocumentDetails Metadata attribute.
                    cdaDocumentDetails.Metadata = metaData;

                    // 3. Assign the patientIdentifier parameter to the CdaDocumentDetails PatientIdentifier attribute
                    cdaDocumentDetails.PatientIdentifier = patientIdentifier;
                    cdaDocumentDetails.PatientIndigenousStatus = patientIndigenousStatus;

                    // 4. Using the patientIdentifier parameter lookup the Patient’s hospital details from the Pcehr Data Store and assign this to the CdaDocumentDetails HospitalPatient attribute
                    // 5. Using the patientIdentifier parameter lookup the Hospital details from the Pcehr Data Store and assign this to the CdaDocumentDetails Hospital attribute.
                    // 6. Using the patientIdentifier parameter lookup the Patient’s master record from the Pcehr Data Store and assign this to the CdaDocumentDetails PatientMaster attribute.
                    PatientAccess patientAccess = new PatientAccess(User);
                    Hospital hospital;
                    HospitalPatient hospitalPatient;
                    PatientMaster patientMaster;

                    cdaDocumentDetails.Status = patientAccess.GetHospital(patientIdentifier, out hospital);
                    if (cdaDocumentDetails.Status.Status != HipsResponseIndicator.OK)
                    {
                        throw new HipsResponseException(cdaDocumentDetails.Status);
                    }
                    cdaDocumentDetails.Status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
                    if (cdaDocumentDetails.Status.Status != HipsResponseIndicator.OK)
                    {
                        throw new HipsResponseException(cdaDocumentDetails.Status);
                    }

                    cdaDocumentDetails.HospitalPatient = hospitalPatient;
                    cdaDocumentDetails.Hospital = hospital;
                    cdaDocumentDetails.PatientMaster = patientMaster;

                    // Added episode due to SaveDocumentDetails method requirement 
                    if (episode == null)
                    {
                        string responseMessage = string.Format(ResponseStrings.UnableToGetEpisode, cdaDocumentDetails.Metadata.AdmissionDateTime, cdaDocumentDetails.PatientIdentifier, hospital.Description);
                        throw new HipsResponseException(new HipsResponse(HipsResponseIndicator.InvalidEpisode));
                    }
                    cdaDocumentDetails.Episode = episode;

                    // 7. Assign the HospitalPatient Mrn attribute retrieved in step 4 above to the CdaDocumentDetails Mrn attribute.
                    cdaDocumentDetails.Mrn = hospitalPatient.Mrn;

                    // 8. For each of the participating providers assigned to the CdaDocumentDetails Metadata attribute, these being DocumentAuthor, ResponsibleHealthProfessional (ReportingPathologist) and LegalAuthenticator; 
                    //      a. Check that the Provider’s Organisation HpiO and Organisation Name has been assigned to the EmployerHpio and EmployerName attributes, 
                    //      if not default these to the Hospital’s HpiO and Name retrieved in step 5 above.
                    List<ParticipatingProvider> providers = new List<ParticipatingProvider>()
                    {
                        cdaDocumentDetails.Metadata.DocumentAuthor,
                        cdaDocumentDetails.Metadata.ResponsibleHealthProfessional,
                        cdaDocumentDetails.Metadata.LegalAuthenticator
                    };
                    providers.ForEach(provider =>
                    {
                        if (provider != null && provider.EmployerHpio == null && provider.EmployerName == null)
                        {
                            provider.EmployerHpio = cdaDocumentDetails.Hospital.HpiO;
                            provider.EmployerName = cdaDocumentDetails.Hospital.HpioName;
                        }
                    });


                    // 9. Create a new CdaAttachment object and assign the following attributes
                    var cdaAttachment = new CdaAttachment();
                    cdaAttachment.AttachmentType = CommonSchemas.Cda.AttachmentType.DocumentBodyPdf;
                    cdaAttachment.Content = pdfDocument;
                    cdaAttachment.FileName = CommonSchemas.Cda.Constants.AttachmentFileName.UNSTRUCTURED_BODY;

                    // Assign this new CdaAttachment to the CdaDocumentDetails Attachments attribute.
                    cdaDocumentDetails.Attachments = new List<CdaAttachment>();
                    cdaDocumentDetails.Attachments.Add(cdaAttachment);

                    // 10. Add the list of CdaAttachments included in the attachments parameter to the CdaDocumentDetails Attachments attribute.
                    if (attachments != null)
                    {
                        cdaDocumentDetails.Attachments.AddRange(attachments);
                    }

                    // 11. Check if an organisational logo has been supplied as one of the CdaAttachment objects in the attachment paramters:
                    //     If no logo has been supplied and the Hospital retrieved in step 5 has a logo then create a new CdaAttachment object
                    if (attachments != null && !attachments.Any(i => i.AttachmentType == AttachmentType.DocumentBodyPdf))
                    {
                        var logoAttachment = new CdaAttachment();
                        logoAttachment.AttachmentType = CommonSchemas.Cda.AttachmentType.OrganisationalLogoPng;
                        logoAttachment.Caption = CommonSchemas.Cda.Constants.AttachmentCaption.ORGANISATIONAL_LOGO;
                        logoAttachment.Content = hospital.Logo;
                        logoAttachment.FileName = CommonSchemas.Cda.Constants.AttachmentFileName.ORGANISATIONAL_LOGO;
                    }

                    // 12. Assign the Patient’s LegalName and CurrentName to the CdaDocumentDetails PatientNames attribute.
                    var registeredName = ObjectMapper.Map<IndividualName>(cdaDocumentDetails.PatientMaster.LegalName);
                    var preferredName = ObjectMapper.Map<IndividualName>(cdaDocumentDetails.PatientMaster.CurrentName);
                    registeredName.Usage = IndividualNameUsage.RegisteredName;
                    preferredName.Usage = IndividualNameUsage.OtherName;

                    cdaDocumentDetails.PatientNames = new List<IndividualName>();
                    cdaDocumentDetails.PatientNames.Add(registeredName);
                    if (registeredName.IsMateriallyDifferentFrom(preferredName))
                    {
                        cdaDocumentDetails.PatientNames.Add(preferredName);
                    }

                    // 13. Assign the CdaDocumentDetails NarrativeText attribute to be “See attached Pathology Result Report”.
                    cdaDocumentDetails.EventNarrativeText = "See attached Pathology Result Report";

                    // 14. Assign the requestingHospitalPatientMrn parameter to the CdaPathDIDocumentDetails RequestHospitalPatientMrn attribute.
                    cdaPathDIDocumentDetails.RequestingHospitalPatientMrn = requestingHospitalPatientMrn;

                    // 15. Assign the orderDetails parameter to the CdaPathDIDocumentDetails OrderDetails attribute
                    cdaPathDIDocumentDetails.OrderDetails = orderDetails;

                    // 16. Assign the pathologyTestResult objects to the CdaPathDIDocumentDetails attribute.
                    cdaPathDIDocumentDetails.PathologyTestResult = pathologyTestResult;

                    // Assign the reportResultStatus parameter to the cdaPathDIDocumentDetails.RelatedDocumentDetails.ResultStatus
                    cdaPathDIDocumentDetails.RelatedDocumentDetails = new RelatedDocumentDetails();
                    cdaPathDIDocumentDetails.RelatedDocumentDetails.ResultStatus = reportResultStatus;

                    // 17. Set the DocumentFormatCode as Pathology Result Report format code in the CdaDocumentDetails.
                    cdaDocumentDetails.DocumentFormatCode = Settings.Instance.PathologyReportDocumentFormatCode;

                    // 18. Save the CdaDocumentDetails to the Pcehr Data Store.
                    var discharSummaryService = new DischargeSummaryService(User);
                    discharSummaryService.SaveDocumentDetails(cdaDocumentDetails, ListSingleton.Instance.AllDocumentTypes.Single(a => a.Code == DocumentTypeCodes.PathologyReport));

                    cdaPathDIDocumentDetails.CdaDocumentDetails = cdaDocumentDetails;

                    // 19. Map the CdaDocumentDetails to the NEHTA CDA® Pathology Report document.
                    PathologyResultReport pathologyResultReport = PopulatePathologyResultReport(cdaPathDIDocumentDetails);

                    // 20. Invoke the “GeneratePathologyResultReport” passing the NEHTA CDA Pathology Report document created in step 17 as the parameter.
                    XmlDocument cdaDocument = CDAGenerator.GeneratePathologyResultReport(pathologyResultReport);

                    // 21. Convert the resulting cdaDocument to Binary and assign it to the CdaDocumentDetails CdaDocumentXmlBytes attribute.
                    cdaDocumentDetails.CdaDocumentXmlBytes = ConvertToBinary(cdaDocument);

                }
                catch (HipsResponseException ex)
                {
                    // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                    throw ex;
                }
                catch (ItemNotFoundException ex)
                {
                    // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                    throw ex;
                }
                catch (ValidationException ex)
                {
                    cdaDocumentDetails.Status.Status = HipsResponseIndicator.ValidationError;
                    cdaDocumentDetails.Status.HipsErrorMessage = ex.GetMessagesString();
                    throw new HipsResponseException(cdaDocumentDetails.Status);
                }
                catch (Exception ex)
                {
                    cdaDocumentDetails.Status.Status = HipsResponseIndicator.SystemError;
                    cdaDocumentDetails.Status.HipsErrorMessage = ex.Message;
                    cdaDocumentDetails.Status.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
                    cdaDocumentDetails.Status.ResponseCode = ex.GetType().Name;
                    throw new HipsResponseException(cdaDocumentDetails.Status);
                }

                cdaPathDIDocumentDetails.CdaDocumentDetails = cdaDocumentDetails;
                return cdaPathDIDocumentDetails;
            }
        }

        /// <summary>
        /// This method creates the Diagnostic Imaging CDA® Report for the purpose of uploading to the PCEHR.
        /// </summary>
        /// <param name="pdfDocument">The Diagnostic Imaging PDF report</param>
        /// <param name="metaData">The header metadata required for the CDA® document</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="orderDetails">The order details for the Diagnostic Imaging Order</param>
        /// <param name="imagingExamResult">A list of ImagingExamResult objects</param>
        /// <param name="requestingHospitalPatientMrn">The requesting Hospital’s Patient Mrn</param>
        /// <param name="attachments">A list of CDA® attachments, documents related to the report.</param>
        /// <param name="reportResultStatus">The Overall Report Result Status from RelatedDocumentDetails.ResultStatus</param>
        /// <param name="episode">Patient Episode details</param>
        /// <returns>
        /// The CdaDocumentDetails object.
        /// </returns>
        public CdaPathDIDocumentDetails CreateDiagnosticImagingReport(byte[] pdfDocument, CdaHeaderMetadata metaData, PatientIdentifierBase patientIdentifier,
            OrderDetails orderDetails, List<ImagingExamResult> imagingExamResult, string requestingHospitalPatientMrn, List<CdaAttachment> attachments, 
            string reportResultStatus, string patientIndigenousStatus, Episode episode)
        {
            // 1. Create a new CdaDocumentDetails and CdaPathDIDocumentDetails object.
            var cdaPathDIDocumentDetails = new CdaPathDIDocumentDetails();
            var cdaDocumentDetails = new CdaDocumentDetails();
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                try
                {
                    // 2. Assign the metaData parameter to the CdaDocumentDetails Metadata attribute.
                    cdaDocumentDetails.Metadata = metaData;

                    // 3. Assign the patientIdentifier parameter to the CdaDocumentDetails PatientIdentifier attribute
                    cdaDocumentDetails.PatientIdentifier = patientIdentifier;
                    cdaDocumentDetails.PatientIndigenousStatus = patientIndigenousStatus;

                    // 4. Using the patientIdentifier parameter lookup the Patient’s hospital details from the Pcehr Data Store and assign this to the CdaDocumentDetails HospitalPatient attribute
                    // 5. Using the patientIdentifier parameter lookup the Hospital details from the Pcehr Data Store and assign this to the CdaDocumentDetails Hospital attribute.
                    // 6. Using the patientIdentifier parameter lookup the Patient’s master record from the Pcehr Data Store and assign this to the CdaDocumentDetails PatientMaster attribute.
                    PatientAccess patientAccess = new PatientAccess(User);
                    Hospital hospital;
                    HospitalPatient hospitalPatient;
                    PatientMaster patientMaster;

                    cdaDocumentDetails.Status = patientAccess.GetHospital(patientIdentifier, out hospital);
                    if (cdaDocumentDetails.Status.Status != HipsResponseIndicator.OK)
                    {
                        throw new HipsResponseException(cdaDocumentDetails.Status);
                    }
                    cdaDocumentDetails.Status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
                    if (cdaDocumentDetails.Status.Status != HipsResponseIndicator.OK)
                    {
                        throw new HipsResponseException(cdaDocumentDetails.Status);
                    }

                    cdaDocumentDetails.HospitalPatient = hospitalPatient;
                    cdaDocumentDetails.Hospital = hospital;
                    cdaDocumentDetails.PatientMaster = patientMaster;

                    // Added episode due to SaveDocumentDetails method requirement 
                    if (episode == null)
                    {
                        string responseMessage = string.Format(ResponseStrings.UnableToGetEpisode, cdaDocumentDetails.Metadata.AdmissionDateTime, cdaDocumentDetails.PatientIdentifier, hospital.Description);
                        throw new HipsResponseException(new HipsResponse(HipsResponseIndicator.InvalidEpisode));
                    }
                    cdaDocumentDetails.Episode = episode;

                    // 7. Assign the HospitalPatient Mrn attribute retrieved in step 4 above to the CdaDocumentDetails Mrn attribute.
                    cdaDocumentDetails.Mrn = hospitalPatient.Mrn;

                    // 8. For each of the participating providers assigned to the CdaDocumentDetails Metadata attribute, these being DocumentAuthor, ResponsibleHealthProfessional (ReportingPathologist) and LegalAuthenticator; 
                    //      a. Check that the Provider’s Organisation HpiO and Organisation Name has been assigned to the EmployerHpio and EmployerName attributes, 
                    //      if not default these to the Hospital’s HpiO and Name retrieved in step 5 above.
                    List<ParticipatingProvider> providers = new List<ParticipatingProvider>()
                    {
                        cdaDocumentDetails.Metadata.DocumentAuthor,
                        cdaDocumentDetails.Metadata.ResponsibleHealthProfessional,
                        cdaDocumentDetails.Metadata.LegalAuthenticator
                    };
                    providers.ForEach(provider =>
                    {
                        if (provider != null && provider.EmployerHpio == null && provider.EmployerName == null)
                        {
                            provider.EmployerHpio = cdaDocumentDetails.Hospital.HpiO;
                            provider.EmployerName = cdaDocumentDetails.Hospital.HpioName;
                        }
                    });

                    // 9. Create a new CdaAttachment object and assign the following attributes
                    var cdaAttachment = new CdaAttachment();
                    cdaAttachment.AttachmentType = CommonSchemas.Cda.AttachmentType.DocumentBodyPdf;
                    cdaAttachment.Content = pdfDocument;
                    cdaAttachment.FileName = CommonSchemas.Cda.Constants.AttachmentFileName.UNSTRUCTURED_BODY;

                    // Assign this new CdaAttachment to the CdaDocumentDetails Attachments attribute.
                    cdaDocumentDetails.Attachments = new List<CdaAttachment>();
                    cdaDocumentDetails.Attachments.Add(cdaAttachment);

                    // 10. Add the list of CdaAttachments included in the attachments parameter to the CdaDocumentDetails Attachments attribute.
                    if (attachments != null)
                        cdaDocumentDetails.Attachments.AddRange(attachments);

                    // 11. Check if an organisational logo has been supplied as one of the CdaAttachment objects in the attachment paramters:
                    //     If no logo has been supplied and the Hospital retrieved in step 5 has a logo then create a new CdaAttachment object
                    if (attachments == null || !attachments.Any(i => i.AttachmentType == AttachmentType.DocumentBodyPdf))
                    {
                        var logoAttachment = new CdaAttachment();
                        logoAttachment.AttachmentType = CommonSchemas.Cda.AttachmentType.OrganisationalLogoPng;
                        logoAttachment.Caption = CommonSchemas.Cda.Constants.AttachmentCaption.ORGANISATIONAL_LOGO;
                        logoAttachment.Content = hospital.Logo;
                        logoAttachment.FileName = CommonSchemas.Cda.Constants.AttachmentFileName.ORGANISATIONAL_LOGO;
                    }

                    // 12. Assign the Patient’s LegalName and CurrentName to the CdaDocumentDetails PatientNames attribute.
                    var registeredName = ObjectMapper.Map<IndividualName>(cdaDocumentDetails.PatientMaster.LegalName);
                    var preferredName = ObjectMapper.Map<IndividualName>(cdaDocumentDetails.PatientMaster.CurrentName);
                    registeredName.Usage = IndividualNameUsage.RegisteredName;
                    preferredName.Usage = IndividualNameUsage.OtherName;

                    cdaDocumentDetails.PatientNames = new List<IndividualName>();
                    cdaDocumentDetails.PatientNames.Add(registeredName);
                    if (registeredName.IsMateriallyDifferentFrom(preferredName))
                    {
                        cdaDocumentDetails.PatientNames.Add(preferredName);
                    }

                    // 13. Assign the CdaDocumentDetails NarrativeText attribute to be “See attached Diagnostic Imaging Examination Report”.
                    cdaDocumentDetails.EventNarrativeText = "See attached Diagnostic Imaging Examination Report";

                    // 14. Assign the requestingHospitalPatientMrn parameter to the CdaPathDIDocumentDetails RequestHospitalPatientMrn attribute.
                    cdaPathDIDocumentDetails.RequestingHospitalPatientMrn = requestingHospitalPatientMrn;

                    // 15. Assign the orderDetails parameter to the CdaPathDIDocumentDetails OrderDetails attribute
                    cdaPathDIDocumentDetails.OrderDetails = orderDetails;

                    // 16. Assign the imagingExamResult objects to the CdaPathDIDocumentDetails attribute.
                    cdaPathDIDocumentDetails.ImagingExamResult = imagingExamResult;

                    // Assign the reportResultStatus parameter to the cdaPathDIDocumentDetails.RelatedDocumentDetails.ResultStatus
                    cdaPathDIDocumentDetails.RelatedDocumentDetails = new RelatedDocumentDetails();
                    cdaPathDIDocumentDetails.RelatedDocumentDetails.ResultStatus = reportResultStatus;

                    // 17. Set the DocumentFormatCode as Diagnostic Imaging Examination Report format code in the CdaDocumentDetails.
                    cdaDocumentDetails.DocumentFormatCode = Settings.Instance.DiagnosticImagingReportDocumentFormatCode;

                    // 18. Save the CdaDocumentDetails to the Pcehr Data Store.
                    var discharSummaryService = new DischargeSummaryService(User);
                    discharSummaryService.SaveDocumentDetails(cdaDocumentDetails, ListSingleton.Instance.AllDocumentTypes.Single(a => a.Code == DocumentTypeCodes.DiagnosticImagingReport));

                    cdaPathDIDocumentDetails.CdaDocumentDetails = cdaDocumentDetails;

                    // 19. Map the CdaDocumentDetails to the NEHTA CDA® DiagnosticImaging Report document.
                    Nehta.VendorLibrary.CDA.Common.DiagnosticImagingReport diagnosticImagingReport = PopulateDiagnosticImagingReport(cdaPathDIDocumentDetails);

                    // 20. Invoke the “GenerateDiagnosticImagingReport” passing the NEHTA CDA Diagnostic Imaging Report document created in step 17 as the parameter.
                    XmlDocument cdaDocument = CDAGenerator.GenerateDiagnosticImagingReport(diagnosticImagingReport);

                    // 21. Convert the resulting cdaDocument to Binary and assign it to the CdaDocumentDetails CdaDocumentXmlBytes attribute.
                    cdaDocumentDetails.CdaDocumentXmlBytes = ConvertToBinary(cdaDocument);
                }
                catch (HipsResponseException ex)
                {
                    // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                    throw ex;
                }
                catch (ItemNotFoundException ex)
                {
                    // This exception is designed to be thrown up to the service layer and mapped by the fault profile.
                    throw ex;
                }
                catch (ValidationException ex)
                {
                    cdaDocumentDetails.Status.Status = HipsResponseIndicator.ValidationError;
                    cdaDocumentDetails.Status.HipsErrorMessage = ex.GetMessagesString();
                    throw new HipsResponseException(cdaDocumentDetails.Status);
                }
                catch (Exception ex)
                {
                    cdaDocumentDetails.Status.Status = HipsResponseIndicator.SystemError;
                    cdaDocumentDetails.Status.HipsErrorMessage = ex.Message;
                    cdaDocumentDetails.Status.ResponseCodeDetails = ex.InnerException != null ? ex.InnerException.Message : null;
                    cdaDocumentDetails.Status.ResponseCode = ex.GetType().Name;
                    throw new HipsResponseException(cdaDocumentDetails.Status);
                }

                cdaPathDIDocumentDetails.CdaDocumentDetails = cdaDocumentDetails;
                return cdaPathDIDocumentDetails;
            }
        }
        
        #endregion Methods

        #region Private Methods        

        /// <summary>
        /// Populates the NEHTA CDA Library's PathologyResultReport model from the details.
        /// </summary>
        /// <param name="details">Document upload details.</param>
        /// <returns>NEHTA PathologyResultReport model.</returns>
        private Nehta.VendorLibrary.CDA.Common.PathologyResultReport PopulatePathologyResultReport(CdaPathDIDocumentDetails details)
        {
            Nehta.VendorLibrary.CDA.Common.PathologyResultReport model = ObjectMapper.Map<Nehta.VendorLibrary.CDA.Common.PathologyResultReport>(details);
            return model;
        }
        /// <summary>
        /// Converts the CDA document to a binary blob.
        /// </summary>
        /// <param name="cdaDocument">CDA document.</param>
        /// <returns>Binary blob.</returns>
        private static byte[] ConvertToBinary(XmlDocument cdaDocument)
        {
            byte[] cdaDocumentBytes;
            using (MemoryStream stream = new MemoryStream())
            {
                XmlWriterSettings settings = new XmlWriterSettings();
                settings.Encoding = new UTF8Encoding(encoderShouldEmitUTF8Identifier: false);
                using (XmlWriter writer = XmlWriter.Create(stream, settings))
                {
                    cdaDocument.Save(writer);
                }
                cdaDocumentBytes = stream.ToArray();
            }
            return cdaDocumentBytes;
        }

        /// <summary>
        /// Populates the NEHTA CDA Library's DiagnosticImagingReport model from the details.
        /// </summary>
        /// <param name="details">Document upload details.</param>
        /// <returns>NEHTA DiagnosticImagingReport model.</returns>
        private Nehta.VendorLibrary.CDA.Common.DiagnosticImagingReport PopulateDiagnosticImagingReport(CdaPathDIDocumentDetails details)
        {
            Nehta.VendorLibrary.CDA.Common.DiagnosticImagingReport model = ObjectMapper.Map<Nehta.VendorLibrary.CDA.Common.DiagnosticImagingReport>(details);
            return model;
        }
        #endregion Private Methods       
    }
}
