﻿using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonBusinessLogic.Cda;
using HIPS.CommonBusinessLogic.Mapping;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.CommonSchemas.Cda.ParticipatingIndividual;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.Configuration.Tracing;
using HIPS.HL7.Common.DataStructure;
using HIPS.HL7.Common.Enumerators;
using HIPS.HL7.Common.Exceptions;
using HIPS.HL7.Common.Message;
using HIPS.HL7.Common.Segment;
using HIPS.HpiiSchemas;
using HIPS.HpiiSchemas.Enumerators;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Transactions;
using Nehta.VendorLibrary.Common;
using nehta.mcaR50.ProviderSearchForProviderIndividual;

namespace HIPS.CommonBusinessLogic.HL7
{
    public class PDILoader : BaseDl
    {
        private UserDetails user;

        #region Constructor
        public PDILoader(UserDetails user) : base(user)
        {
            this.user = user;
        }
        #endregion Constructor

        #region Constants
        /// <summary>
        /// Constants used for the OBR HL7Field.
        /// </summary>
        private const string UNIVERSAL_SERVICE_ID_OBR_FIELD = "OBR-4.3";
        private const string UNIVERSAL_SERVICE_ALTERNATE_ID_OBR_FIELD = "OBR-4.6";
        #endregion Constants

        #region Public Methods
        /// <summary>
        /// Processes a HL7 Pathology or Diagnostic Imaging Message. 
        /// Creates the patient details in the database if it is a new Patient.
        /// Creates a new episode in the database if one does not exist for the FillerOrderNumber
        /// Saves the filler order number into the database against the episode if a new episode.
        /// Loads the Cda objects required to create either a Pathology or Diagnostic Imaging CDA Document
        /// </summary>
        /// <param name="documentTypeCode">Either PathologyReport or DiagnosticImagingReport</param>
        /// <param name="genericPathMessage">The HL7 message to process</param>
        /// <param name="secondaryId">Out. SecondaryID</param>
        /// <param name="patientMrn">Out. Patient MRN</param>
        /// <param name="hospitalId">Out. HospitalID</param>
        /// <param name="cdaHeaderMetadata">Out. CdaHeaderMetadata object</param>
        /// <param name="orderDetails">Out. OrderDetails object</param>
        /// <param name="imagingExamResults">Out. List of ImagingExamResults (if Diagnostic Imaging Report) </param>
        /// <param name="pathologyTestResults">Out. List PathologyTestResults (if Pathology Report)</param>
        /// <param name="relatedDocumentStatus">Out. SourceDocumentStatus</param>
        /// <param name="episode">Out. The Patients Episode details</param>
        internal void Process(string documentTypeCode, HL7GenericPathMessage genericPathMessage, out CX secondaryId, out Mrn patientMrn, out int hospitalId, 
            out CdaHeaderMetadata cdaHeaderMetadata, out OrderDetails orderDetails, out List<ImagingExamResult> imagingExamResults, 
            out List<PathologyTestResult> pathologyTestResults, out string relatedDocumentStatus, out Episode episode)
        {
            using (HIPSTraceWriter trace = new HIPSTraceWriter())
            {
                trace.WriteTrace(string.Format("IN HIPS.CommonBuisnessLogic.HL7.PDILoader.Process:: MessageControlID: {0}", genericPathMessage.MessageHeader.MessageControlID), System.Diagnostics.TraceEventType.Information);

                using (new TransactionScope(TransactionScopeOption.Suppress))
                {
                    PatientMaster patientMaster;
                    Hospital hospital;
                    bool isExistingPatientMaster, isExistingHospitalPatient, episodeCreated;
                    PatientMaster oldPatientMaster;

                    using (SqlCommand command = base.Command)
                    {
                        try
                        {
                            SqlTransaction transaction = command.Connection.BeginTransaction();
                            command.Transaction = transaction;

                            // Process the PID segment
                            trace.WriteTrace(string.Format("Call HIPS.CommonBuisnessLogic.HL7.PDILoader.Process.ProcessPatientIdentifiers:: MessageControlID: {0}", genericPathMessage.MessageHeader.MessageControlID), System.Diagnostics.TraceEventType.Information);
                            ProcessPatientIdentifiers(documentTypeCode, genericPathMessage, transaction, out patientMaster, out hospital, out episode, out patientMrn, out secondaryId, out isExistingPatientMaster, out isExistingHospitalPatient, out episodeCreated, out oldPatientMaster);
                            hospitalId = hospital.HospitalId.Value;

                            // Process the OBR segment
                            trace.WriteTrace(string.Format("Call HIPS.CommonBuisnessLogic.HL7.PDILoader.Process.ProcessOrderGroupDetails:: MessageControlID: {0}", genericPathMessage.MessageHeader.MessageControlID), System.Diagnostics.TraceEventType.Information);
                            ProcessOrderGroupDetails(documentTypeCode, genericPathMessage, transaction, hospital, episode, patientMaster, out pathologyTestResults, out imagingExamResults, out orderDetails, out cdaHeaderMetadata, out relatedDocumentStatus);

                            transaction.Commit();
                            command.Connection.Close();

                        }
                        catch (Exception ex)
                        {
                            // For any exception, HIPS will tell the message broker
                            // to retry the message, so we will rollback the transaction.
                            trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.HL7.PDILoader.Process:: MessageControlID: {0}. Exception Details: {1}", genericPathMessage.MessageHeader.MessageControlID, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    
                            if (command.Transaction != null)
                            {
                                command.Transaction.Rollback();
                            }
                            command.Connection.Close();
                            throw ex;
                        }
                    }
                    try
                    {
                        // Once we have saved the Patient details check the IHI.
                        trace.WriteTrace(string.Format("Call HIPS.CommonBuisnessLogic.HL7.PDILoader.Process.ProcessIHIDetails:: MessageControlID: {0}", genericPathMessage.MessageHeader.MessageControlID), System.Diagnostics.TraceEventType.Information);
                        ProcessIHIDetails(genericPathMessage, isExistingPatientMaster, isExistingHospitalPatient, episodeCreated, hospital, oldPatientMaster, patientMaster);
                    }
                    catch (Exception ex)
                    {
                        trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.HL7.PDILoader.Process:: MessageControlID: {0}. Exception Details: {1}", genericPathMessage.MessageHeader.MessageControlID, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    
                        throw ex;
                    }
                }

                trace.WriteTrace(string.Format("OUT HIPS.CommonBuisnessLogic.HL7.PDILoader.Process:: MessageControlID: {0}", genericPathMessage.MessageHeader.MessageControlID), System.Diagnostics.TraceEventType.Information);

            }
        }

        #endregion Public Methods

        #region Private Methods

        /// <summary>
        /// Processes an incoming Pathology or Diagnostic Imaging message. Loads patients, patient masters, episodes,
        /// episode diagnoses, episode diagnosis related groups and episode procedures.
        /// Stores the hospital ID and Mrn into the message log as soon as they are known.
        /// Throws an HL7LoaderException if the message cannot be processed.
        /// </summary>
        /// <param name="documentTypeCode">Either PathologyReport or DiagnosticImagingReport</param>
        /// <param name="message">The received parsed HL7 message</param>
        /// <param name="transaction">SQL transaction</param>
        /// <param name="patientMaster">Out. The PatientMaster record</param>
        /// <param name="hospital">Out. The hospital record</param>
        /// <param name="episode">Out. The Episode record</param>
        /// <param name="patientMrn">Out. The patient Mrn (this is the primary patient identifier)</param>
        /// <param name="secondaryId">Out. The Secondary Id CX object</param>
        /// <param name="isExistingPatientMaster">Out. True if an existing PatientMaster exists</param>
        /// <param name="isExistingHospitalPatient">Out. true if an existing HospitalPatient exists</param>
        /// <param name="episodeCreated">Out. True if a new Episode was created</param>
        /// <param name="oldPatientMaster">Out. Existing PatientMaster record if one existed</param>
        private void ProcessPatientIdentifiers(string documentTypeCode, HL7GenericPathMessage message, SqlTransaction transaction, out PatientMaster patientMaster, out Hospital hospital, out Episode episode, out Mrn patientMrn, out CX secondaryId, out bool isExistingPatientMaster, out bool isExistingHospitalPatient, out bool episodeCreated, out PatientMaster oldPatientMaster)
        {

            HospitalPatient hospitalPatient;

            CX primaryId;
            HL7GenericPasMessage genericPasMessage = new HL7GenericPasMessage();
            genericPasMessage.PatientIdentification = message.PatientIdentification;

            // Get the Patient Identifiers
            ExtractPatientIdentifiers(genericPasMessage, out primaryId, out secondaryId);

            // Process the Message and get the episode 
            episodeCreated = ProcessNormalPDIMessage(documentTypeCode, message, transaction, primaryId, out hospital, out hospitalPatient, out patientMaster, out oldPatientMaster, out isExistingPatientMaster, out isExistingHospitalPatient, out episode, out patientMrn);

        }

        /// <summary>
        /// Processes a Pathology or Diagnostic Imaging Message.
        /// Extracts the necessary infomation from the PID segment to create or update the PatientMaster, HospitalPatient, PatientMasterIHI and Episode details
        /// </summary>
        /// <param name="documentTypeCode">Either PathologyReport or DiagnosticImagingReport</param>
        /// <param name="message">HL7 Pathology or Diagnostic Imaging Message </param>
        /// <param name="transaction">SQL transaction</param>
        /// <param name="mrn">Mrn details</param>
        /// <param name="hospital">Out. Hospital Record</param>
        /// <param name="hospitalPatient">Out. HospitalPatient Record</param>
        /// <param name="patientMaster">Out. PatientMaster Record</param>
        /// <param name="oldPatientMaster">Out. Existing PatientMaster Record (if an update, null otherwise)</param>
        /// <param name="isExistingPatientMaster">Out. Flag to indicate if there is an existing PatientMaster Record</param>
        /// <param name="isExistingHospitalPatient">Out. Flag to indicate if there is an existing HospitalPatient Record</param>
        /// <param name="episode">Out. Episode loaded or created</param>
        /// <param name="patientMrn">Out. The Mrn object</param>
        /// <returns>True if a new episode was created, false otherwise</returns>
        private bool ProcessNormalPDIMessage(string documentTypeCode, HL7GenericPathMessage message, SqlTransaction transaction, CX mrn, out Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster, out PatientMaster oldPatientMaster, out bool isExistingPatientMaster, out bool isExistingHospitalPatient, out Episode episode, out Mrn patientMrn)
        {
            // Find and Load the Patient 
            PatientLoader patientLoader = new PatientLoader(user, transaction);
           
            // Create the Demographic and ValidatedIHI from the HL7 Message
            Demographic demographics;
            ValidatedIhi validatedIhi;
            string hospitalCode = message.MessageHeader.SendingFacility.universalID;
            string hospitalCodeSystem = HL7.ConstantsResource.PatientAdministrationSystemHospitalCodeSystem;
            patientLoader.CreatePatientDemographicValidatedIHI(message.PatientIdentification, hospitalCode, out demographics, out validatedIhi);

            // Find or Create the Patient
            string primaryIdentiferHospitalCode = mrn.assigningauthority.namespaceID;
            string primaryIdentifierHospitalCodeSystem = HL7.ConstantsResource.PrimaryPatientIdentifierCodeSystem;
            patientMrn = new Mrn(mrn.ID, primaryIdentiferHospitalCode, primaryIdentifierHospitalCodeSystem);
            
            patientLoader.FindOrCreatePatient(message.PatientIdentification, patientMrn, demographics, validatedIhi, out patientMaster, out hospital, out hospitalPatient, out oldPatientMaster, out isExistingPatientMaster, out isExistingHospitalPatient);
            
            bool episodeCreated = false;

            //Invoke the LoadOrCreateEpisode passing in a DocumentType passed in to retrieve a stub episode to use for the report.
            DocumentType documentType = ListSingleton.Instance.AllDocumentTypes.Single(a => a.Code == documentTypeCode);
            EpisodeLoader episodeLoader = new EpisodeLoader(user, transaction);
            episode = episodeLoader.LoadOrCreateEpisode(hospitalPatient, message.Order.ToList(), documentType, out episodeCreated);

            return episodeCreated;
        }

        /// <summary>
        /// Process the Patient IHI details. 
        /// If a new Patient master then Register the Patient and retrieve their IHI
        /// If a new Hospital Patient but not Patient Master then re-validate their IHI
        /// Otherwise existing Patient master and Hospital Patient then Update the Patient details and if demographic change then re-validate IHI
        /// Once valid IHI is retrieved invoke the CheckPcehrExists
        /// </summary>
        /// <param name="message">The HL7 Pathology/Diagnostic Imaging message</param>
        /// <param name="isExistingPatientMaster">Flag to indicate if this was an existing Patient Master or a new one was created</param>
        /// <param name="isExistingHospitalPatient">Flag to indicate if this was an existing Hospital Patient or a new one was created</param>
        /// <param name="episodeCreated">Flag to indicate if a new episode was created for the patient</param>
        /// <param name="hospital">The Hospital the Patient belongs to</param>
        /// <param name="oldPatientMaster">The old Patient Master record if a new Patient Master was created</param>
        /// <param name="patientMaster">The new Patient Master record</param>
        private void ProcessIHIDetails(HL7GenericPathMessage message, bool isExistingPatientMaster, bool isExistingHospitalPatient, bool episodeCreated, Hospital hospital, PatientMaster oldPatientMaster, PatientMaster patientMaster)
        {
            
            try
            {
                // Based on the results:
                // If the “BypassHiService” setting is false then
                if (!Settings.Instance.BypassHIService)
                {
                    // If a new patient master was created then invoke RegisterPatient on the PatientRegistration class to attempt to find the patient’s IHI.
                    if (!isExistingPatientMaster)
                    {
                        var patientRegistration = new Ihi.PatientRegistration();
                        // Check for the AUSEHR flag here, if it is in any OBRs then we shouldn't call DoecPCEHRExist when registering patient
                        patientRegistration.RegisterPatient(patientMaster, hospital, user, !CheckForAusehrFlag(message.Order));
                    }
                    else if (isExistingPatientMaster && !isExistingHospitalPatient)
                    {
                        // If there was an existing patient master but a new hospital patient record was created then invoke RevalidateIhi on PatientIhiValidation class to check for duplicate patients and confirm the IHI for this facility.
                        var patientIhiValidation = new Ihi.PatientIhiValidation();
                        patientIhiValidation.RevalidateIhi(patientMaster, hospital, user);
                    }
                    else
                    {   // Assign the new Patient Master details to see if the Demographics need to change
                        // Otherwise invoke UpdatePatient on the PatientDemographicUpdate class to update the patient’s details.
                        var patientDemographicUpdate = new Ihi.PatientDemographicUpdate();
                        patientDemographicUpdate.UpdatePatient(oldPatientMaster, patientMaster, hospital, user);
                    }
                }
                
                // If the patient has no IHI or the IHI is not currently validated then reject the message and do not attempt to upload the report.
                if (String.IsNullOrEmpty(patientMaster.Ihi) || !patientMaster.IhiLastValidated.HasValue)
                {
                    throw new IhiErrorException(ConstantsResource.NoIhiForPatient);
                }

                if (Settings.Instance.BypassHIService)
                {
                    // Check the validation Period for the IHI (if we don't do this here it will try and re-validate IHI on document upload
                    int validationPeriod;
                    int.TryParse(ConfigurationManager.AppSettings["IhiValidationPeriodDays"], out validationPeriod);
                    TimeSpan ts = DateTime.Now.Subtract(patientMaster.IhiLastValidated.Value);
                    if (ts.TotalDays >= validationPeriod)
                    {
                        throw new IhiErrorException(ConstantsResource.IhiValidationPeriodExpired);
                    }
                    // Check that after the update the IHI is still in an Active Status (may have already been in another Status before update)
                    if (patientMaster.IhiStatusId != (int)IhiStatus.Active)
                    {
                        throw new IhiErrorException(ConstantsResource.IhiNotInActiveStatus);
                    }
                }
                

                // Invoke the CheckPcehrExists to retrieve the advertised or disclosed status of the patient PCEHR.
                PatientAccess patientAccess = new PatientAccess(user);
                CheckPcehrExistsResponse pcehrResponse = patientAccess.CheckPcehrExists(message.Order.ToList(), patientMaster, hospital, episodeCreated);
                if (pcehrResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HL7MessageErrorException(pcehrResponse.HipsResponse.HipsErrorMessage);
                }
            }
            catch (HL7MessageErrorException ex)
            {
                throw new HL7MessageErrorException(ex.Message);
            }
            catch (Exception ex)
            {

                // If the exception message is one of the known data quality
                // issues, write an INFO message to the log, otherwise write an
                // ERROR message to the log for the attention of application
                // support.

                if (ex.Message == ValidIhiErrorsResource.InvalidCharactersInGivenName
                    || ex.Message == ValidIhiErrorsResource.InvalidCharactersInFamilyName
                    || ex.Message == ValidIhiErrorsResource.InvalidDvaFileNumber
                    || ex.Message == ValidIhiErrorsResource.InvalidMedicareCardNumber
                    || ex.Message == ValidIhiErrorsResource.RetiredIhiStatus)
                {
                    throw new IhiInfoException(ex.Message);
                }
                throw new IhiErrorException(ex.Message);
            }
        }

        /// <summary>
        /// Loads the Order Group details into the cda objects required for the Cda Report
        /// </summary>
        /// <param name="documentTypeCode">Either PathologyReport or DiagnosticImagingReport</param>
        /// <param name="message">Pathology or Diagnostic Imaging HL7 Message</param>
        /// <param name="transaction">SQL transaction</param>
        /// <param name="hospital">Hospital record</param>
        /// <param name="episode">Episode record</param>
        /// <param name="patientMaster">PatientMaster record</param>
        /// <param name="pathologyTestResults">Out. List of pathologyTestResults, only populated if documentTypeCode is PathologyReport</param>
        /// <param name="imagingExamResults">Out. List of imagingExamResults, only populated if documentTypeCode is DiagnosticImagingReport</param>
        /// <param name="orderDetails">Out. OrderDetails object</param>
        /// <param name="metadata">Out. CdaHeaderMetaData object</param>
        /// <param name="relatedDocumentStatus">Out. relatedDocumentStatus</param>
        private void ProcessOrderGroupDetails(string documentTypeCode, HL7GenericPathMessage message, SqlTransaction transaction, Hospital hospital, Episode episode, PatientMaster patientMaster, out List<PathologyTestResult> pathologyTestResults, out List<ImagingExamResult> imagingExamResults, out OrderDetails orderDetails, out CdaHeaderMetadata metadata, out string relatedDocumentStatus)
        {
            string pathologyRequestedTestName = string.Empty;
            
            ParticipatingProvider requester = new ParticipatingProvider();
            List<ParticipatingProvider> participatingProviders = new List<ParticipatingProvider>();
            List<string> placerOrderNumbers = new List<string>();
            SourceDocumentStatus reportStatus = SourceDocumentStatus.None;
            metadata = new CdaHeaderMetadata();
            orderDetails = new OrderDetails();
            relatedDocumentStatus = "";
            pathologyTestResults = (documentTypeCode == DocumentTypeCodes.PathologyReport ? new List<PathologyTestResult>() : null);
            imagingExamResults = (documentTypeCode == DocumentTypeCodes.DiagnosticImagingReport ? new List<ImagingExamResult>() : null);

            // Store the Filler Order Numbers so we can save later to data store
            List<string> fillerOrderNumbers = new List<string>();
            // Store the resultDates so we can get the latest one and assign to Overall Report Date
            List<DateTime> resultDates = new List<DateTime>();

            foreach (var order in message.Order)
            {
                // Report Status (Completion Code) is interim if any OBR result statuses are in a Preliminary state otherwise Final
                reportStatus = (order.Observation.Any(i => i.ObservationsReportID.ResultStatus == "P") ? SourceDocumentStatus.Interim : SourceDocumentStatus.Final);

                // Related Document Status (Report Status) if any OBR result statuses are preliminary uses preliminary, if any OBR statuses are corrected then corrected, otherwise final.
                relatedDocumentStatus = (order.Observation.Any(i => (i.ObservationsReportID.ResultStatus == "P" || i.ObservationsReportID.ResultStatus == "C")) ? (order.Observation.Any(i => i.ObservationsReportID.ResultStatus == "P") ? "P" : "C") : "F");

                // FR-PDI-15.1	The eHISC service will reject messages that contain multiple OBR results with different individuals in OBR-16 Requester (ordering provider).
                var distinctOrderingProviders = order.Observation.GroupBy(
                                                            i => string.Format("{0} {1} {2}", i.ObservationsReportID.OrderingProvider.familylastname.familyname, i.ObservationsReportID.OrderingProvider.givenname, i.ObservationsReportID.OrderingProvider.prefix),
                                                            (key, group) => group.First()
                                                            ).ToArray();
                if (distinctOrderingProviders.Count() > 1)
                    throw new HL7MessageInfoException(ResponseStrings.MultipleOrderingProvidersExist);

                foreach (var observation in order.Observation)
                {
                    var obr = observation.ObservationsReportID;

                    // Populate the Filler Order Number array
                    fillerOrderNumbers.Add(obr.FillerOrderNumber.entityidentifier);

                    // Check that we have an Ordering Provider
                    if (obr.OrderingProvider == null || obr.OrderingProvider.familylastname == null || string.IsNullOrEmpty(obr.OrderingProvider.familylastname.familyname))
                        throw new HL7MessageInfoException(ConstantsResource.NoOrderingProviderDetails);
                    // Populate the Requester object
                    requester.FamilyName = obr.OrderingProvider.familylastname.familyname;
                    requester.GivenNames = obr.OrderingProvider.givenname;
                    requester.Title = string.IsNullOrEmpty(obr.OrderingProvider.prefix) ? null : obr.OrderingProvider.prefix;
                    requester.EmployerName = (obr.OrderingProvider.assigningfacility != null && !string.IsNullOrEmpty(obr.OrderingProvider.assigningfacility.universalID) ? obr.OrderingProvider.assigningfacility.universalID : "NI");

                    // Populate the Participating Provider object (Author, Reporting Pathologist)
                    participatingProviders.Add(GetParticipatingProvider(obr, hospital, message.MessageHeader.MessageControlID, patientMaster.PatientMasterId.Value));

                    // Assign the Document Name as the requested test names and the creation date as the observation dte
                    pathologyRequestedTestName += string.Format("{0},", obr.UniversalServiceID.text);
                    // Check that there is an results date and time
                    if (obr.ResultsRptStatusChngDateTime == null || !obr.ResultsRptStatusChngDateTime.Any(r => r.TimestampValue.HasValue))
                        throw new HL7MessageInfoException(ConstantsResource.InvalidResultsDateTime);
                    resultDates.Add(obr.ResultsRptStatusChngDateTime.Where(r => r.TimestampValue.HasValue)
                                                                               .OrderBy(r => r.TimestampValue.Value)
                                                                               .Single()
                                                                               .TimestampValue.Value);

                    // Add the placeOrderNumber to the list
                    placerOrderNumbers.Add(obr.PlacerOrderNumber.entityidentifier);

                    // 12. Create a list of TestResult where each test result is created from an OBR and it’s OBX
                    if (obr.UniversalServiceID == null || string.IsNullOrEmpty(obr.UniversalServiceID.text))
                        throw new HL7MessageInfoException(ConstantsResource.NoTestResultNameSupplied);
                    var primaryName = new TestResultName()
                    {
                        Name = obr.UniversalServiceID.text,
                        Code = obr.UniversalServiceID.identifier,
                        CodeSystem = obr.UniversalServiceID.nameofcodingsystem,
                        HL7Field = UNIVERSAL_SERVICE_ID_OBR_FIELD // this should be the HL7 field of the UniversalServiceID.identfier
                    };

                    var alternateNames = new TestResultName[]
                    { 
                        new TestResultName()
                        { 
                            Name = obr.UniversalServiceID.alternatetext,
                            Code = obr.UniversalServiceID.alternateidentifier,
                            CodeSystem = obr.UniversalServiceID.nameofalternatecodingsystem,
                            HL7Field = UNIVERSAL_SERVICE_ALTERNATE_ID_OBR_FIELD // this should be the HL7 field of the UniversalServiceID.alternateidentfier
                        }
                    };

                    // Check that there is an observation date and time
                    if (obr.ObservationDateTime == null || !obr.ObservationDateTime.TimestampValue.HasValue)
                        throw new HL7MessageInfoException(ConstantsResource.InvalidObservationDateTime);
                    DateTime observationDateTime = obr.ObservationDateTime.TimestampValue.Value;

                    // Check that there is a Test Result Status
                    if (string.IsNullOrEmpty(obr.ResultStatus))
                        throw new HL7MessageInfoException(ConstantsResource.NoTestResultStatusSupplied);
                    // Check that it is a valid test result status
                    var values = Enum.GetValues(typeof(Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus))
                                 .Cast<Nehta.VendorLibrary.CDA.Common.Enums.Hl7V3ResultStatus>()
                                 .Where(v => v.GetAttributeValue<Nehta.VendorLibrary.CDA.Common.NameAttribute, string>(a => a.Code) == obr.ResultStatus);
                    if (values.Count() < 1)
                        throw new HL7MessageInfoException(ConstantsResource.IncorrectTestResultStatusSupplied);
                    var testResult = new TestResult();
                    testResult.PrimaryName = primaryName;
                    testResult.AlternateNames = alternateNames;
                    testResult.Status = obr.ResultStatus;
                    testResult.ObservationDateTime = observationDateTime;
                    // Create the narrative text from the CdaHeaderMetaData. If there are multiple DocumentAuthors then add them to the narrative text.
                    string testResultAuthor = string.Empty;
                    foreach (var authorParticipatingProviders in participatingProviders.Distinct())
                    {
                        testResultAuthor += authorParticipatingProviders.GivenNames + " " + authorParticipatingProviders.FamilyName + ", ";
                    }
                    //testResult.NarrativeText = testResultAuthor.substring(0, testResultAuthor.Length-2); TODO: Map this to another field perhaps AdminObservations?
                    if (documentTypeCode == DocumentTypeCodes.PathologyReport)
                    {
                        PathologyTestResult pathologyTestResult = new PathologyTestResult();
                        pathologyTestResult.TestResult = testResult;
                        pathologyTestResult.PathologyDiscipline = (PathologyDiscipline)Enum.Parse(typeof(PathologyDiscipline), obr.DiagnosticServSectID, true);
                        pathologyTestResult.CollectionDateTime = observationDateTime;

                        pathologyTestResults.Add(pathologyTestResult);
                    }
                    else
                    {
                        // Check that there is a Procedure set
                        if (obr.UniversalServiceID == null || string.IsNullOrEmpty(obr.UniversalServiceID.text))
                            throw new HL7MessageInfoException(ConstantsResource.NoExaminationProcedureDetails);
                        ImagingExamResult imagingExamResult = new ImagingExamResult();
                        imagingExamResult.TestResult = testResult;
                        imagingExamResult.ImageDateTime = observationDateTime;
                        imagingExamResult.Procedure = obr.UniversalServiceID.text;
                        imagingExamResult.Modality = obr.DiagnosticServSectID;
                        // FR-PDI-17.4 The eHISC service will populate Anatomic Site with a null flavour of “UNK”.
                        imagingExamResult.AnatomicalSite = new List<string>(){Nehta.HL7.CDA.NullFlavor.UNK.ToString()};

                        imagingExamResults.Add(imagingExamResult);
                    }
                }
            }
            // Populate the CdaHeaderMetaData object with the following information

            // FR-PDI.16.17	For messages that contain multiple OBR results authored by different individuals, the eHISC service will set the CDA atomic data for the individual details of the Document Author and Reporting Pathologist / Reporting Radiologist to null
            var distinctParticipatingProviders = participatingProviders.GroupBy(
                                                            i => i.Hpii,
                                                            (key, group) => group.First()
                                                            ).ToArray();
            ParticipatingProvider author;
            if (distinctParticipatingProviders.Count() > 1)
            {
                author = new ParticipatingProvider();
                author.FamilyName = Nehta.HL7.CDA.NullFlavor.NI.ToString(); // Null Flavour
                author.LocalIdentifier = Nehta.HL7.CDA.NullFlavor.NI.ToString();
            }
            else
                author = distinctParticipatingProviders.SingleOrDefault();

            metadata.AdmissionDateTime = episode.AdmissionDate;
            metadata.DischargeDateTime = (episode.DischargeDate == null) ? System.DateTime.Now : episode.DischargeDate.Value;
            metadata.DocumentAuthor = author;
            metadata.DocumentCreationDateTime = resultDates.Max();
            metadata.LegalAuthenticator = author;
            metadata.ModeOfSeparation = ModeOfSeparation.None;
            metadata.PatientAddress = ObjectMapper.Map<HIPS.CommonSchemas.Cda.Address>(patientMaster.Addresses.FirstOrDefault());
            metadata.PatientContactDetails = ObjectMapper.Map<ElectronicCommunicationDetail>(patientMaster.Contacts.FirstOrDefault(c => !string.IsNullOrEmpty(c.Detail)));
            metadata.ResponsibleHealthProfessional = author;
            metadata.SourceDocumentStatus = reportStatus;

            // Get and Save the orderIdentifier by either the existing order identifier or placeOrderNumber if unique, otherwise create a GUID
            // FR-PDI-15.3	If the OBR-2 Placer Order Number is different in each OBR segment, the eHISC service will generate a UUID for the Requester Order Identifier rather than using the OBR-2 field in any of the OBR segments. 
            string orderIdentifier = placerOrderNumbers.FirstOrDefault();
            if (string.IsNullOrEmpty(orderIdentifier) || placerOrderNumbers.GroupBy(i => i.ToString(), (key, group) => group.First()).ToArray().Count() > 1)
                orderIdentifier = Guid.NewGuid().ToString();
            // FR-PDI-12.4	The eHISC service will store the OBR-3 Filler Order Number of every OBR segment associated with the document set.
            DocumentType docType = ListSingleton.Instance.AllDocumentTypes.Single(a => a.Code == documentTypeCode);
            string docFormatCode = (documentTypeCode == DocumentTypeCodes.DiagnosticImagingReport ? Settings.Instance.DiagnosticImagingReportDocumentFormatCode : Settings.Instance.PathologyReportDocumentFormatCode);
            PatientAccess patientAccess = new PatientAccess(this.user);
            // Save the CdaSet for this Patient Episode
            CdaSetNumber cdaSetNumber;
            patientAccess.SaveCdaSetNumber(transaction, metadata, episode, docType, docFormatCode, out cdaSetNumber);
            foreach (string fillerOrderNumber in fillerOrderNumbers)
            {
                patientAccess.SaveFillerOrderNumber(transaction, orderIdentifier, fillerOrderNumber, cdaSetNumber.CdaSetNumberId.Value);
            }

            // Populate the OrderDetails object with the following information
            orderDetails.Requester = requester;
            orderDetails.OrderIdentifier = orderIdentifier;
            orderDetails.PathologyRequestedTestName = (pathologyRequestedTestName.Length > 1 ? pathologyRequestedTestName.Substring(0, pathologyRequestedTestName.Length - 1) : pathologyRequestedTestName);
            orderDetails.DiagnosticImagingAccessionNumber = (documentTypeCode == DocumentTypeCodes.DiagnosticImagingReport ? fillerOrderNumbers.FirstOrDefault() : string.Empty);
        }

        /// <summary>
        /// Determines how to get the Participating Provider (from HL7 message or local lookup) and retrieves, 
        /// and depending on HI Service Mode, validates the Provider details
        /// </summary>
        /// <param name="obr">The OBR segment of the HL7 message</param>
        /// <param name="hospital">The Hospital the Patient belongs to</param>
        /// <returns>Participating Provider</returns>
        private ParticipatingProvider GetParticipatingProvider(OBR obr, Hospital hospital, string messageControlID, int patientMasterId)
        {
            HealthProviderIndividualDl healthProviderIndividualDataAccess = new HealthProviderIndividualDl(this.user);
            HealthProviderIndividual healthProviderIndividual = null;
            ParticipatingProvider participatingProvider = new ParticipatingProvider();

            try
            {
                string pasIdentifier = obr.PrincipalResultInterpreter.name.IDnumberST;
                string assigningAuthority = ((obr.PrincipalResultInterpreter.name.assigningauthority != null) ? obr.PrincipalResultInterpreter.name.assigningauthority.universalID : string.Empty);

                // FR-PDI-7.2.1	In Mode 2 (No Connection to HI Service) the message must contain the HPI-I of the document author.
                if (Settings.Instance.BypassHIService && assigningAuthority != "AUSHIC")
                    throw new HL7MessageInfoException(ResponseStrings.HpiiNotSupplied);

                // Check that the message contains the Provider Family Name
                if (obr.PrincipalResultInterpreter.name == null || string.IsNullOrEmpty(obr.PrincipalResultInterpreter.name.familyname))
                    throw new HL7MessageInfoException(ResponseStrings.HealthProviderNameNotSupplied);
                            
                participatingProvider.FamilyName = obr.PrincipalResultInterpreter.name.familyname;
                participatingProvider.GivenNames = string.Format("{0} {1}", (string.IsNullOrEmpty(obr.PrincipalResultInterpreter.name.givenname) ? string.Empty : obr.PrincipalResultInterpreter.name.givenname.Trim()), (string.IsNullOrEmpty(obr.PrincipalResultInterpreter.name.middleinitialorname) ? string.Empty : obr.PrincipalResultInterpreter.name.middleinitialorname.Trim()));

                HpiiQueryResponse hpiiValidateResponse;
                // Check if the HPII is included in the HL7 message
                if (assigningAuthority == "AUSHIC")
                {
                    // Validate if HPI-I is in a valid format 
                    if (!Regex.IsMatch(pasIdentifier, "^800361[0-9]{10}$"))
                        throw new HL7MessageInfoException(ResponseStrings.HpiiLengthNot16);
                    // Check the Luhn Digit
                    int checkDigit;
                    int.TryParse(pasIdentifier.Substring(pasIdentifier.Length-1, 1), out checkDigit);
                    if (checkDigit != CalculateLuhnCheckDigit(pasIdentifier))
                        throw new HL7MessageInfoException(ResponseStrings.HpiiNotValid);

                    // FR-PDI-7.2.2	In Mode 2 (No Connection to HI Service) the eHISC service will trust that all IHI and HPI-I identifiers in the message have already been validated by another system.
                    participatingProvider.Hpii = pasIdentifier;

                    // Look up in Local Database
                    HospitalDl hospitalAccess = new HospitalDl();
                    Hospital facility = hospitalAccess.Get(hospital.HospitalId.GetValueOrDefault());
                    if (facility == null || !facility.HospitalId.HasValue)
                        throw new HL7MessageInfoException(ResponseStrings.HospitalNotFound);
                    HealthProviderIndividualHpiiDl hpiHpiiDataAccess = new HealthProviderIndividualHpiiDl(this.user);
                    HealthProviderIndividualHpii hpiHpii = hpiHpiiDataAccess.Get(pasIdentifier, facility.HealthProviderOrganisationNetworkId);
                    if (hpiHpii == null || !hpiHpii.HealthProviderIndividualId.HasValue) // Not found in Local DB so let's Validate with HI Service
                    {
                        // If we do then validate it (if ByPassHIService is not set - otherwise we trust what was passed in)
                        if (!Settings.Instance.BypassHIService)
                        {
                            hpiiValidateResponse = ValidateProviderHpii(hospital, healthProviderIndividual, pasIdentifier, obr.PrincipalResultInterpreter.name.familyname);
                            // Check for HipsResponse Status
                            switch (hpiiValidateResponse.HipsResponse.Status)
                            {
                                case HipsResponseIndicator.OK:
                                    participatingProvider.Hpii = hpiiValidateResponse.HpiiNumber;
                                    break;
                                case HipsResponseIndicator.HpiiWarningRaised:
                                    // HPI-I warning, this will continue processing so assign Hpii
                                    participatingProvider.Hpii = hpiiValidateResponse.HpiiNumber;
                                    throw new HpiiInfoException(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                                case HipsResponseIndicator.UnresolvedHpiiAlert:
                                    throw new HpiiErrorException(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                                case HipsResponseIndicator.HiServiceError:
                                    throw new HiServiceException(hpiiValidateResponse.HipsResponse);
                                case HipsResponseIndicator.SystemError:
                                    throw new Exception(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                            }
                            
                        }
                    }
                    else
                    {
                        // Look up the HealthProviderIndividual Record
                        healthProviderIndividualDataAccess.Get(hpiHpii.HealthProviderIndividualId.Value, hospital.HospitalId.Value, out healthProviderIndividual);
                        // Compare supplied name with stored name
                        bool providerNameChange = (healthProviderIndividual.FamilyName != obr.PrincipalResultInterpreter.name.familyname);
                        // Check LastValidatedDate against HpiiValidationPeriod
                        bool validationPeriodExpired = true;
                        if (healthProviderIndividual.HpiiLastValidated.HasValue)
                        {
                            TimeSpan duration = DateTime.Now.Subtract(healthProviderIndividual.HpiiLastValidated.Value);
                            validationPeriodExpired = (duration.TotalDays > Settings.Instance.HpiiValidationPeriodDays);
                        }
                        // If name changes or not within HpiiValidationPeriod
                        if ((providerNameChange || validationPeriodExpired) && !Settings.Instance.BypassHIService)
                        {
                            hpiiValidateResponse = ValidateProviderHpii(hospital, healthProviderIndividual, pasIdentifier, obr.PrincipalResultInterpreter.name.familyname);
                            // Check for HipsResponse Status
                            switch (hpiiValidateResponse.HipsResponse.Status)
                            {
                                case HipsResponseIndicator.OK:
                                    participatingProvider.Hpii = hpiiValidateResponse.HpiiNumber;
                                    break;
                                case HipsResponseIndicator.HpiiWarningRaised:
                                    // HPI-I warning, this will continue processing so assign Hpii
                                    participatingProvider.Hpii = hpiiValidateResponse.HpiiNumber;
                                    throw new HpiiInfoException(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                                case HipsResponseIndicator.UnresolvedHpiiAlert:
                                    throw new HpiiErrorException(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                                case HipsResponseIndicator.HiServiceError:
                                    throw new HiServiceException(hpiiValidateResponse.HipsResponse);
                                case HipsResponseIndicator.SystemError:
                                    throw new Exception(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                            }
                        }
                    }
                }
                else
                {
                    // FR-PDI-7.1.3	In Mode 1 (HI Service connection) , if the OBR-32 Principal Result Interpreter does not contain the HPI-I of the individual, then the eHISC service will look up the HPI-I in the local lookup table using the local identifier allocated by the LIS.
                    // Using Local Provider Lookup - if not using Medicare provider number then check the Hospital is valid
                    if (assigningAuthority != "AUSHICPR")
                    {
                        HospitalDl hospitalAccess = new HospitalDl();
                        Hospital facility = hospitalAccess.Get(hospital.HospitalId.GetValueOrDefault());
                        if (facility == null || !facility.HospitalId.HasValue)
                            throw new HL7MessageInfoException(ResponseStrings.HospitalNotFound);
                    }
                    // Look up the Local Provider
                    HospitalHealthProviderIndividualDl hospitalHealthProviderIndividualDataAccess = new HospitalHealthProviderIndividualDl(this.user);
                    HospitalHealthProviderIndividual hospitalHealthProviderIndividual = hospitalHealthProviderIndividualDataAccess.Get(hospital.HospitalId.Value, pasIdentifier);
                    // Check we have Hospital Health Provider using the local provider identifier from the HL7
                    if (hospitalHealthProviderIndividual == null || !hospitalHealthProviderIndividual.HealthProviderIndividualId.HasValue)
                        throw new HL7MessageInfoException(ResponseStrings.HealthProviderNotFound);

                    // Get the Health Provider Individual details including HPII
                    healthProviderIndividualDataAccess.Get(hospitalHealthProviderIndividual.HealthProviderIndividualId.Value, hospital.HospitalId.Value, out healthProviderIndividual);
                    if (healthProviderIndividual == null || string.IsNullOrEmpty(healthProviderIndividual.HpiI))
                        throw new HL7MessageInfoException(ResponseStrings.HealthProviderNotFound);

                    // Check the Provider has not been deactivated
                    if (obr.ObservationDateTime.TimestampValue > healthProviderIndividual.DeactivatedDate)
                        throw new HL7MessageInfoException(ResponseStrings.HpiiNotValid);

                    // Assign the HPII from the Local DB
                    participatingProvider.Hpii = healthProviderIndividual.HpiI;

                    // FR-PDI-7.1.4	In Mode 1 (HI Service connection), if the date/time of last validation for the IHI or HPI-I is not specified or outside the configured period, the identifier requires validation and will be validated by searching the HI Service with the identifier and key demographic information.
                    var needsRevalidation = true;
                    if (healthProviderIndividual.HpiiLastValidated.HasValue)
                    {
                        var duration = DateTime.Now.Subtract(healthProviderIndividual.HpiiLastValidated.Value);
                        needsRevalidation = duration.TotalDays > Settings.Instance.HpiiValidationPeriodDays;
                    }
                
                    if (needsRevalidation)
                    {
                        hpiiValidateResponse = ValidateProviderHpii(hospital, healthProviderIndividual, healthProviderIndividual.HpiI, obr.PrincipalResultInterpreter.name.familyname);
                        // Check for HipsResponse Status
                        switch (hpiiValidateResponse.HipsResponse.Status)
                        {
                            case HipsResponseIndicator.OK:
                                participatingProvider.Hpii = hpiiValidateResponse.HpiiNumber;
                                break;
                            case HipsResponseIndicator.HpiiWarningRaised:
                                // HPI-I warning, this will continue processing so assign Hpii
                                participatingProvider.Hpii = hpiiValidateResponse.HpiiNumber;
                                throw new HpiiInfoException(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                            case HipsResponseIndicator.UnresolvedHpiiAlert:
                                throw new HpiiErrorException(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                            case HipsResponseIndicator.HiServiceError:
                                throw new HiServiceException(hpiiValidateResponse.HipsResponse);
                            case HipsResponseIndicator.SystemError:
                                throw new Exception(hpiiValidateResponse.HipsResponse.HipsErrorMessage);
                        }
                    }
                }
            }
            catch (HiServiceException ex)
            {
                // throw an InvalidOperationException for a re-try
                throw new InvalidOperationException(ex.Message);
            }
            catch (HpiiInfoException ex)
            {
                // Simply log the warning and carry on processing
                string hpiiInfoException = ex.Message;
                hpiiInfoException += string.Format(" From Message Control ID {0} for PatientMasterID {1}", messageControlID, patientMasterId);
                UpdateEventLog(hpiiInfoException, new HpiiInfoException(hpiiInfoException), this.user, LogMessage.HIPS_MESSAGE_188);

            }
            catch (HpiiErrorException ex)
            {
                // throw this up for failure
                throw ex;
            }
            catch (HL7MessageInfoException ex)
            {
                // throw this up to be handled up further
                throw ex;
            }
            catch (Exception ex)
            {
                // throw this up to be handled up further
                throw ex;
            }
            return participatingProvider;
  
        }


        /// <summary>
        /// Checks if the AUSEHR flag in OBR-20 (Filler1 Field) is set, returns true if it is false otherwise
        /// FR-PDI-8.1	When an “AUSEHR” (Australian Electronic Health Record) flag is found in OBR-20 Utility Flags, 
        /// the eHISC service will not invoke the ‘Does PCEHR exist’ operation but rather use this flag as the sole indication of the existence of a PCEHR.
        /// </summary>
        /// <param name="genericPathMessage">The HL7 Generic Path message</param>
        /// <returns>True if any OBR segment has AUSEHR, false otherwise</returns>
        private bool CheckForAusehrFlag(HIPS.HL7.Common.SegmentGroup.OrderGroup[] orderGroup)
        {
            foreach (var order in orderGroup)
            {
                if (order.Observation.Any(i => i.ObservationsReportID.FillerField1.Contains("AUSEHR")))
                    return true;
            }
            return false;
        }

        /// <summary>
        /// Updated the SystemErrorLog table with warning details
        /// </summary>
        /// <param name="responseString">The Warning Response</param>
        /// <param name="ex">The Exception</param>
        /// <param name="user">Details about the user to log in this event.</param>
        /// <param name="logMessage">The LogMessage code</param>
        private void UpdateEventLog(string responseString, Exception ex, UserDetails user, LogMessage logMessage)
        {
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                EventLogger.WriteLog(responseString, ex, user, logMessage);
            }
        }

        /// <summary>
        /// Calls the ValidateHpii from the Patient Access.
        /// </summary>
        /// <param name="hospital">The Hospital calling the HI Service</param>
        /// <param name="healthProviderIndividual">The HealthProviderIndividual record to pass through, can be null</param>
        /// <param name="hpii">The provider's HPI-I number to validate</param>
        /// <param name="providerFamilyname">The Provider's Family Name to perform the HPI-I validation against</param>
        /// <returns>The HpiiQueryResponse object</returns>
        private HpiiQueryResponse ValidateProviderHpii(Hospital hospital, HealthProviderIndividual healthProviderIndividual, string hpii, string providerFamilyname)
        {
            PatientAccess patientAccess = new PatientAccess(this.user);
            HpiiIdentifierQuery query = new HpiiIdentifierQuery() { HpiiNumber = hpii, FamilyName = providerFamilyname };
            HpiiQueryResponse hpiiQueryResponse = patientAccess.ValidateHpii(query, healthProviderIndividual, hospital);
            return hpiiQueryResponse;
        }
        
        /// <summary>
        /// Extracts the Primary and Secondary Patient Identifiers
        /// </summary>
        /// <param name="message">Pathology or Diagnostic Imaging HL7 Message</param>
        /// <param name="primaryId">Out. The Primary Id</param>
        /// <param name="secondaryId">Out. The secondary Id</param>
        private void ExtractPatientIdentifiers(HL7GenericPasMessage message, out CX primaryId, out CX secondaryId)
        {
            // If there are multiple internal patient identifiers (PI) then throw a MulitplePrimaryIdentifierException
            int primaryIdentifierCount = (from pi in message.PatientIdentification.PatientIdentifierList where pi.identifiertypecode == IdentifierTypeCode.PI.ToString() select pi).Count();
            int mrnCount = (from mrn in message.PatientIdentification.PatientIdentifierList where mrn.identifiertypecode == IdentifierTypeCode.MR.ToString() select mrn).Count();
            primaryId = null;
            secondaryId = null;
            if (primaryIdentifierCount > 1)
            {
                throw new HL7MessageInfoException(ResponseStrings.MultiplePrimaryIdentifiersFound);
            }
            // If only 1 PI is found then return this as the primary
            if (primaryIdentifierCount == 1)
            {
                primaryId = PatientLoader.IdentifierWithType(message.PatientIdentification.PatientIdentifierList, IdentifierTypeCode.PI);
                // If an MRN is also found then retrun this as secondary
                if (mrnCount >= 1)
                    secondaryId = PatientLoader.IdentifierWithType(message.PatientIdentification.PatientIdentifierList, IdentifierTypeCode.MR);
            }
            else // No PI, then check for MR
            {
                // If there us multiple MR type identifiers then look up the assigning authority and assign the primary 
                // where the CodeSystemId is the configured CodeSystem for the primary patient identifiers
                if (mrnCount > 1)
                {
                    // Get the MR type identifiers
                    IEnumerable<CX> patientMrnList = (from mrn in message.PatientIdentification.PatientIdentifierList 
                                                      where mrn.identifiertypecode == IdentifierTypeCode.MR.ToString() select mrn);
                    // for each get the hospital Id from the Code table
                    foreach(CX mrn in patientMrnList)
                    {
                        IEnumerable<int> hospitalIdList = (from codeSystem in ListSingleton.Instance.AllCodeSystems
                                          join hospitalCodes in ListSingleton.Instance.AllHospitalCodes
                                  on codeSystem.CodeSystemId equals hospitalCodes.CodeSystemId
                                          where (codeSystem.Code == ConstantsResource.PrimaryPatientIdentifierCodeSystem) && (hospitalCodes.Code == mrn.assigningauthority.namespaceID)
                                          select hospitalCodes.HospitalId);
                        // if a single assign authority is found then this is the primary MRN
                        if (hospitalIdList.Count() == 1)
                            primaryId = mrn;
                        else if (hospitalIdList.Count() > 1) // If more than one found then reject the message as we can't determine the primary
                            throw new HL7MessageInfoException(ResponseStrings.MultiplePrimaryIdentifiersFound);
                        else // otherwise if assigning authority not found this is a secondary
                            secondaryId = mrn;
                        
                    }
                    // if no primary id is assigned (the assigning authority was not found in the hospital code table) then reject the message
                    if (primaryId == null)
                        throw new HL7MessageInfoException(ConstantsResource.NoMrnForPatient);
                    
                }
                else
                {
                    // Look for the hospital patient ID (MRN) in PID-3 Patient Identifier List with identifier MR (Medical Record Number)
                    primaryId = PatientLoader.IdentifierWithType(message.PatientIdentification.PatientIdentifierList, IdentifierTypeCode.MR);
                    if (primaryId == null)
                    {
                        // None were marked with identifier MR, try the first item in PID-3
                        primaryId = message.PatientIdentification.PatientIdentifierList[0];
                        if (primaryId == null || string.IsNullOrWhiteSpace(primaryId.ID))
                        {
                            throw new HL7MessageErrorException(ConstantsResource.NoMrnForPatient);
                        }
                        primaryId.ID = MrnPadding.Pad(primaryId.ID);
                    }
                }
                
            }   
        }
        /// <summary>
        /// Calculates the Luhn Check Digit for a health Identifier (IHI or HPI-I)
        /// </summary>
        /// <param name="healthIdentifier">HPI-I or IHI to calculate the Luhn Check Digit for</param>
        /// <returns>The Check Digit</returns>
        private int? CalculateLuhnCheckDigit(string healthIdentifier)
        {
            int[] Results = { 0, 2, 4, 6, 8, 1, 3, 5, 7, 9 };
            List<int> healthIdentifierDigits = new List<int>();
            int digit;
            for (int count = 0; count < healthIdentifier.Length - 1; count++) // add all of the digits to a list except the last check digit itself
            {
                if (int.TryParse(healthIdentifier.Substring(count, 1), out digit))
                    healthIdentifierDigits.Add(digit);
            }
            var i = 0;
            var lengthMod = healthIdentifierDigits.Count % 2;
            return (healthIdentifierDigits.Sum(d => i++ % 2 == lengthMod ? d : Results[d]) * 9) % 10;

        }
        #endregion Private Methods

    }
}
