﻿using System;
using System.Data.SqlClient;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.HL7.Common.Enumerators;
using HIPS.HL7.Common.Exceptions;
using HIPS.HL7.Common.DataStructure;
using HIPS.HL7.Common.Message;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.CommonBusinessLogic.HL7
{
    public class PasLoader : BaseDl
    {
        private UserDetails user;

        public PasLoader(UserDetails user)
        {
            this.user = user;
        }

        /// <summary>
        /// Processes an incoming PAS message. Loads patients, patient masters, episodes,
        /// episode diagnoses, episode diagnosis related groups and episode procedures.
        /// Stores the hospital ID and Mrn into the message log as soon as they are known.
        /// Throws an HL7LoaderException if the message cannot be processed.
        /// </summary>
        /// <param name="message">The received parsed HL7 message</param>
        /// <param name="messageLog">The message log entry</param>
        public void Process(HL7GenericPasMessage message, HL7MessageLog messageLog)
        {
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            PatientMaster oldPatientMaster = null;
            bool episodeCreated = false;
            bool hospitalPatientCreated = false;
            bool mergeNeedsIhiValidation = false;
            bool mergeProcessed = false;
            HL7MessageLogDl messageLogDataAccess = new HL7MessageLogDl(user);
            using (SqlCommand command = base.Command)
            {
                try
                {
                    SqlTransaction transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;
                    CX statePatientId;
                    CX mrn;
                    ExtractPatientIdentifiers(message, out statePatientId, out mrn);
                    messageLog.HospitalPatientIdentifier = mrn.ID;

                    // If this message is a merge or move, use special processing and skip the normal processing.
                    if (MergeLoader.CanProcess(message))
                    {
                        MergeLoader mergeLoader = new MergeLoader(user, transaction);
                        mergeNeedsIhiValidation = mergeLoader.Process(message, mrn, statePatientId, out hospital, out patientMaster, out oldPatientMaster);
                        mergeProcessed = true;
                    }
                    else
                    {
                        episodeCreated = ProcessNormalPasMessage(message, messageLog, transaction, statePatientId, mrn, out hospital, out hospitalPatient, out patientMaster, out oldPatientMaster, out hospitalPatientCreated);
                    }

                    if (!messageLogDataAccess.Insert(messageLog, transaction))
                    {
                        throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, messageLogDataAccess.GetType().FullName));
                    }
                    transaction.Commit();
                    command.Connection.Close();
                }
                catch (HL7MessageInfoException ex)
                {
                    // Special treatment for Info exception at this level:
                    // HIPS will rollback the transaction and skip this message,
                    // with an INFO message in the log. HIPS will tell the message
                    // broker that this message was processed successfully, so we
                    // need to keep a copy of it for data quality analysis.

                    if (command.Transaction != null)
                    {
                        command.Transaction.Rollback();
                    }
                    command.Connection.Close();

                    messageLog.FailureReason = ex.Message;
                    messageLog.QueueStatusId = (int)QueueStatus.Failure;
                    messageLogDataAccess.Insert(messageLog);
                    throw ex;
                }
                catch (Exception ex)
                {
                    // For any other exception, HIPS will tell the message broker
                    // to retry the message, so we will rollback the transaction
                    // and we won't store the message.

                    if (command.Transaction != null)
                    {
                        command.Transaction.Rollback();
                    }
                    command.Connection.Close();
                    throw ex;
                }
            }

            try
            {
                if (mergeProcessed)
                {
                    if (mergeNeedsIhiValidation)
                    {
                        PatientIhiValidation validation = new PatientIhiValidation();
                        validation.RevalidateIhi(patientMaster, hospital, user);
                        if (oldPatientMaster != null && oldPatientMaster.PatientMasterId.HasValue)
                        {
                            // The old patient master doesn't have any MRNs
                            // from this hospital any more, but its
                            // revalidation is required when there are
                            // remaining MRNs from other hospitals.
                            validation.RevalidateIhi(oldPatientMaster, hospital, user);
                        }
                    }
                }
                else if (oldPatientMaster == null)
                {
                    // A new patient master was created, attempt to get an IHI for it.
                    PatientRegistration registration = new PatientRegistration();
                    registration.RegisterPatient(patientMaster, hospital, user);
                }
                else if (hospitalPatientCreated)
                {
                    // This was a new MRN attached to an existing patient master.
                    // It could make the IHI go into a duplicate alert.
                    PatientIhiValidation validation = new PatientIhiValidation();
                    validation.RevalidateIhi(patientMaster, hospital, user);
                }
                else
                {
                    PatientDemographicUpdate update = new PatientDemographicUpdate();
                    update.UpdatePatient(oldPatientMaster, patientMaster, hospital, user);
                }

                if (episodeCreated && !string.IsNullOrEmpty(patientMaster.Ihi) && !hospitalPatientCreated && oldPatientMaster != null)
                {
                    DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
                    doesPcehrExist.PcehrExists(null, hospital, patientMaster, user);
                }
            }
            catch (Exception ex)
            {
                messageLog.FailureReason = ex.Message;
                messageLog.QueueStatusId = (int)QueueStatus.Failure;
                messageLog.RetryCount = messageLog.RetryCount + 1;
                messageLogDataAccess.Update(messageLog);

                // If the exception message is one of the known data quality
                // issues, write an INFO message to the log, otherwise write an
                // ERROR message to the log for the attention of application
                // support.

                if (ex.Message == ValidIhiErrorsResource.InvalidCharactersInGivenName
                    || ex.Message == ValidIhiErrorsResource.InvalidCharactersInFamilyName
                    || ex.Message == ValidIhiErrorsResource.InvalidDvaFileNumber
                    || ex.Message == ValidIhiErrorsResource.InvalidMedicareCardNumber
                    || ex.Message == ValidIhiErrorsResource.RetiredIhiStatus)
                {
                    throw new IhiInfoException(ex.Message);
                }
                throw new IhiErrorException(ex.Message);
            }
            messageLog.QueueStatusId = (int)QueueStatus.Success;
            if (!messageLogDataAccess.Update(messageLog))
            {
                throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, messageLogDataAccess.GetType().FullName));
            }
        }

        internal static void ExtractPatientIdentifiers(HL7GenericPasMessage message, out CX statePatientId, out CX mrn)
        {
            // Look for the state patient ID (SAUHI) and hospital patient ID (MRN) in PID-3 Patient Identifier List with identifier MR (Medical Record Number)
            // If the SAUHI was not found in the PID-3 field, fall back to the PID-2 field.
            statePatientId = PatientLoader.IdentifierWithType(message.PatientIdentification.PatientIdentifierList,
                IdentifierTypeCode.SAUHI, message.PatientIdentification.PatientID);

            // Look for the hospital patient ID (MRN) in PID-3 Patient Identifier List with identifier MR (Medical Record Number)
            mrn = PatientLoader.IdentifierWithType(message.PatientIdentification.PatientIdentifierList, IdentifierTypeCode.MR);
            if (mrn == null)
            {
                // None were marked with identifier MR, try the first item in PID-3
                mrn = message.PatientIdentification.PatientIdentifierList[0];
                if (mrn == null || string.IsNullOrWhiteSpace(mrn.ID))
                {
                    throw new HL7MessageErrorException(ConstantsResource.NoMrnForPatient);
                }
                mrn.ID = MrnPadding.Pad(mrn.ID);
            }
        }

        private bool ProcessNormalPasMessage(HL7GenericPasMessage message, HL7MessageLog messageLog, SqlTransaction transaction, CX statePatientId, CX mrn, out Hospital hospital, out HospitalPatient hospitalPatient, out PatientMaster patientMaster, out PatientMaster oldPatientMaster, out bool hospitalPatientCreated)
        {
            PatientLoader patientLoader = new PatientLoader(user, transaction);
            PatientLinker linker = new PatientLinker(user, transaction);
            linker.FindAndLinkPatient(mrn, statePatientId, out hospital, out hospitalPatient, out patientMaster);
            hospitalPatientCreated = !hospitalPatient.PatientId.HasValue;
            patientLoader.Populate(message.PatientIdentification, hospital, hospitalPatient, patientMaster, out oldPatientMaster);
            messageLog.HospitalID = hospital.HospitalId;
            bool episodeCreated = false;

            // Demographic registration or update messages don't contain a patient visit (episode).
            // For those ones, skip the episode loading.
            if (message.PatientVisit != null)
            {
                EpisodeLoader episodeLoader = new EpisodeLoader(user, transaction);
                Episode episode = episodeLoader.Load(hospitalPatient, message, out episodeCreated);
            }
            return episodeCreated;
        }
    }
}