﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.HpiiSchemas;
using HIPS.PcehrDataStore.Schemas;
using nehta.mcaR50.ProviderBatchSearchForProviderIndividual;
using nehta.mcaR50.ProviderSearchForProviderIndividual;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.HI;

namespace HIPS.CommonBusinessLogic.Hpii
{
    /// <summary>
    /// Performs the HPI-I search query operations described in the TECH.SIS.31 (single search) and TECH.SIS.33 (asynchronous batch) documents.
    /// </summary>
    public class HpiiSearch
    {
        #region Methods

        /// <summary>
        /// Retrieves the results of an HPI-I batch query.
        /// </summary>
        /// <param name="hpiiBatchAsyncRetrieveRequest">The HPI-I batch asynchronous retrieval request.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="facility">The hospital facility.</param>
        /// <returns>The HPI-I batch asynchronous retrieval response.</returns>
        public HpiiBatchAsyncRetrieveResponse HpiiBatchRetrieve(HpiiBatchAsyncRetrieveRequest hpiiBatchAsyncRetrieveRequest, UserDetails user, HospitalIdentifier facility = null)
        {
            HpiiBatchAsyncRetrieveResponse response = new HpiiBatchAsyncRetrieveResponse();

            this.ValidateQueryData(hpiiBatchAsyncRetrieveRequest, response, facility);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }
            HealthProviderOrganisation accessingOrganisation = HpiiHelper.GetHealthProviderOrganisation(hpiiBatchAsyncRetrieveRequest.HpioNumber, facility);
            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(accessingOrganisation);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(accessingOrganisation, user))
            {
                //throw new Exception(ConstantsResource.InvalidUserDetails);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.InvalidUserDetails;

                return response;
            }

            if (!User.ValidateUser(user))
            {
                //throw new Exception(ConstantsResource.MissingUserValues);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.MissingUserValues;

                return response;
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, accessingOrganisation);

            // Instantiate the client
            ProviderBatchAsyncSearchForProviderIndividualClient providerBatchAsyncIndividualClient = new ProviderBatchAsyncSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate);

            try
            {
                System.Reflection.FieldInfo field = providerBatchAsyncIndividualClient.GetType().GetField("providerBatchAsyncSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                SearchForProviderIndividualBatchAsyncPortTypeClient port = field.GetValue(providerBatchAsyncIndividualClient) as SearchForProviderIndividualBatchAsyncPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                // Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.CommonBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                retrieveSearchForProviderIndividual requestBatch = new retrieveSearchForProviderIndividual();
                requestBatch.batchIdentifier = hpiiBatchAsyncRetrieveRequest.BatchIdentifier;

                retrieveSearchForProviderIndividualResponse svcResponse = providerBatchAsyncIndividualClient.BatchRetrieveProviderIndividuals(requestBatch);

                // Extract the results.
                if (svcResponse.retrieveSearchForProviderIndividualResult != null)
                {
                    // Set the upper level batch identifier.
                    response.BatchIdentifier = svcResponse.retrieveSearchForProviderIndividualResult.batchIdentifier;
                    response.ServiceMessagesType = svcResponse.retrieveSearchForProviderIndividualResult.serviceMessages;
                    response.HpiiBatchQueryResponses = new List<HpiiBatchQueryResponse>();

                    if (svcResponse.retrieveSearchForProviderIndividualResult.batchSearchForProviderIndividualResult != null && svcResponse.retrieveSearchForProviderIndividualResult.batchSearchForProviderIndividualResult != null)
                    {
                        // Loop over all items the batchSearchForProviderIndividualResult to extract each individual result
                        foreach (BatchSearchForProviderIndividualResultType result in svcResponse.retrieveSearchForProviderIndividualResult.batchSearchForProviderIndividualResult)
                        {
                            // Map the searchForProviderIndividualResult onto HpiiQueryResponse
                            HpiiBatchQueryResponse hpiiBatchQueryResponse = new HpiiBatchQueryResponse();

                            hpiiBatchQueryResponse.RequestIdentifier = result.requestIdentifier;
                            hpiiBatchQueryResponse.HpiiQueryResponse = HpiiHelper.MapHpiiResponseData(result.searchForProviderIndividualResult, accessingOrganisation);
                            response.HpiiBatchQueryResponses.Add(hpiiBatchQueryResponse);
                        }
                    }
                }

                response.HipsResponse.Status = HipsResponseIndicator.OK;
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();
                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        response.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                        response.HipsResponse.HipsErrorMessage = error.serviceMessage[0].reason;
                        response.HipsResponse.ResponseCode = error.serviceMessage[0].code;
                        response.ServiceMessagesType = new ServiceMessagesType();
                        response.ServiceMessagesType.highestSeverity = error.highestSeverity;
                        response.ServiceMessagesType.serviceMessage = error.serviceMessage;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    // Grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;
                response.HipsResponse.HipsErrorMessage = e.Message;
                // Grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerBatchAsyncIndividualClient.SoapMessages, accessingOrganisation, hpiiBatchAsyncRetrieveResponse: response);
                providerBatchAsyncIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Submits a batch of HPI-I searches.
        /// </summary>
        /// <param name="hpiiBatchAsyncSubmitRequest">The HPI-I batch asynchronous submit request.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="facility">The hospital facility.</param>
        /// <returns>The batch submission response.</returns>
        public HpiiBatchAsyncSubmitResponse HpiiBatchSubmit(HpiiBatchAsyncSubmitRequest hpiiBatchAsyncSubmitRequest, UserDetails user, HospitalIdentifier facility = null)
        {
            HpiiBatchAsyncSubmitResponse response = new HpiiBatchAsyncSubmitResponse();

            this.ValidateQueryBatchData(hpiiBatchAsyncSubmitRequest, response, facility);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }
            HealthProviderOrganisation accessingOrganisation = HpiiHelper.GetHealthProviderOrganisation(hpiiBatchAsyncSubmitRequest.HpioNumber, facility);
            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(accessingOrganisation);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(accessingOrganisation, user))
            {
                //throw new Exception(ConstantsResource.InvalidUserDetails);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.InvalidUserDetails;

                return response;
            }

            if (!User.ValidateUser(user))
            {
                //throw new Exception(ConstantsResource.MissingUserValues);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.MissingUserValues;

                return response;
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, accessingOrganisation);

            // Instantiate the client
            ProviderBatchAsyncSearchForProviderIndividualClient providerBatchAsyncIndividualClient = new ProviderBatchAsyncSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate);

            try
            {
                System.Reflection.FieldInfo field = providerBatchAsyncIndividualClient.GetType().GetField("providerBatchAsyncSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                SearchForProviderIndividualBatchAsyncPortTypeClient port = field.GetValue(providerBatchAsyncIndividualClient) as SearchForProviderIndividualBatchAsyncPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                // Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.CommonBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                // Determine the total number of the batch items
                int batchCount = 0;
                if (hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries != null)
                {
                    batchCount = hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries.Count;
                }
                if (hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries != null)
                {
                    batchCount += hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries.Count;
                }

                if (batchCount == 0)
                {
                    response.HipsResponse.HipsErrorMessage = "No items to process within the Batch Asynchronous Submit";
                    return response;
                }

                // Initialize the batch object
                BatchSearchForProviderIndividualCriteriaType[] requestBatch = new BatchSearchForProviderIndividualCriteriaType[batchCount];

                int itemAddedCount = 0;
                // Loop over all items in the batch submission and add to the BatchSearchForProviderIndividualCriteriaType
                foreach (HpiiIdentifierQuery query in hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries)
                {
                    searchForProviderIndividual request = HpiiHelper.MapHpiiIdentifierRequestData(query);
                    requestBatch[itemAddedCount] = new BatchSearchForProviderIndividualCriteriaType();
                    requestBatch[itemAddedCount].searchForProviderIndividual = request;
                    requestBatch[itemAddedCount].requestIdentifier = (itemAddedCount + 1).ToString();
                    itemAddedCount += 1;
                }

                foreach (HpiiDemographicQuery query in hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries)
                {
                    searchForProviderIndividual request = HpiiHelper.MapHpiiDemographicRequestData(query);
                    requestBatch[itemAddedCount] = new BatchSearchForProviderIndividualCriteriaType();
                    requestBatch[itemAddedCount].searchForProviderIndividual = request;
                    requestBatch[itemAddedCount].requestIdentifier = (itemAddedCount + 1).ToString();
                    itemAddedCount += 1;
                }

                submitSearchForProviderIndividualResponse svcResponse = providerBatchAsyncIndividualClient.BatchSubmitProviderIndividuals(requestBatch);

                if (svcResponse.submitSearchForProviderIndividualResult != null)
                {
                    // Add in batch identifier and correct service messages
                    response.BatchIdentifier = svcResponse.submitSearchForProviderIndividualResult.batchIdentifier;
                    response.ServiceMessagesType = svcResponse.submitSearchForProviderIndividualResult.serviceMessages;
                    if (svcResponse.submitSearchForProviderIndividualResult.serviceMessages != null)
                    {
                        response.HipsResponse.ResponseCode = svcResponse.submitSearchForProviderIndividualResult.serviceMessages.serviceMessage[0].code;
                        response.HipsResponse.ResponseCodeDescription = svcResponse.submitSearchForProviderIndividualResult.serviceMessages.serviceMessage[0].reason;
                    }
                    response.HipsResponse.Status = HipsResponseIndicator.OK;
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = "Submit Result Missing - Batch Identifier could not be identified";
                    return response;
                }
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();
                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        response.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                        response.HipsResponse.HipsErrorMessage = error.serviceMessage[0].reason;
                        response.HipsResponse.ResponseCode = error.serviceMessage[0].code;
                        response.ServiceMessagesType = new ServiceMessagesType();
                        response.ServiceMessagesType.highestSeverity = error.highestSeverity;
                        response.ServiceMessagesType.serviceMessage = error.serviceMessage;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    // Grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;
                response.HipsResponse.HipsErrorMessage = e.Message;
                // Grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerBatchAsyncIndividualClient.SoapMessages, accessingOrganisation, hpiiBatchAsyncSubmitResponse: response);
                providerBatchAsyncIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Performs an HPII demographic search.
        /// </summary>
        /// <param name="hpiiDemographicQuery">The hpii demographic query.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>The HPI-I query response.</returns>
        public HpiiQueryResponse HpiiDemographicSearch(HpiiDemographicQuery hpiiDemographicQuery, UserDetails user, HospitalIdentifier facility = null)
        {
            HpiiQueryResponse response = new HpiiQueryResponse();
            this.ValidateQueryData(hpiiDemographicQuery, response, facility);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }
            HealthProviderOrganisation accessingOrganisation = HpiiHelper.GetHealthProviderOrganisation(hpiiDemographicQuery.HpioNumber, facility);
            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(accessingOrganisation);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(accessingOrganisation, user))
            {
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.InvalidUserDetails;

                return response;
            }

            if (!User.ValidateUser(user))
            {
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.MissingUserValues;

                return response;
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, accessingOrganisation);

            // Instantiate the client
            ProviderSearchForProviderIndividualClient providerIndividualClient = new ProviderSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate);

            try
            {
                System.Reflection.FieldInfo field = providerIndividualClient.GetType().GetField("providerSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                ProviderSearchForProviderIndividualPortTypeClient port = field.GetValue(providerIndividualClient) as ProviderSearchForProviderIndividualPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                // Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.CommonBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                nehta.mcaR50.ProviderSearchForProviderIndividual.searchForProviderIndividual request = HpiiHelper.MapHpiiDemographicRequestData(hpiiDemographicQuery);
                searchForProviderIndividualResponse svcResponse = providerIndividualClient.ProviderIndividualSearch(request);
                response = HpiiHelper.MapHpiiResponseData(svcResponse.searchForProviderIndividualResult, accessingOrganisation);
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();

                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                        // Ignore this Error Message
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        response.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                        response.HipsResponse.HipsErrorMessage = error.serviceMessage[0].reason;
                        response.HipsResponse.ResponseCode = error.serviceMessage[0].code;
                        response.ServiceMessagesType = new ServiceMessagesType();
                        response.ServiceMessagesType.highestSeverity = error.highestSeverity;
                        response.ServiceMessagesType.serviceMessage = error.serviceMessage;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    // Grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;

                // Grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerIndividualClient.SoapMessages, accessingOrganisation, hpiiDemographicQuery: hpiiDemographicQuery, hpiiQueryResponse: response);
                providerIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Performs an HPI-I validation search.
        /// </summary>
        /// <param name="hpiiIdentifierQuery">The hpii identifier query.</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>The HPI-I query response.</returns>
        public HpiiQueryResponse HpiiValidation(HpiiIdentifierQuery hpiiIdentifierQuery, UserDetails user, HospitalIdentifier facility = null)
        {
            HpiiQueryResponse response = new HpiiQueryResponse();

            this.ValidateQueryData(hpiiIdentifierQuery, response, facility);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }
            HealthProviderOrganisation accessingOrganisation = HpiiHelper.GetHealthProviderOrganisation(hpiiIdentifierQuery.HpioNumber, facility);
            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(accessingOrganisation);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(accessingOrganisation, user))
            {
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.InvalidUserDetails;

                return response;
            }

            if (!User.ValidateUser(user))
            {
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.MissingUserValues;

                return response;
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, accessingOrganisation);

            // Instantiate the client
            ProviderSearchForProviderIndividualClient providerIndividualClient = new ProviderSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate,
                Settings.Instance.MockHiServiceHpiiSearchWaitSeconds);

            try
            {
                System.Reflection.FieldInfo field = providerIndividualClient.GetType().GetField("providerSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                ProviderSearchForProviderIndividualPortTypeClient port = field.GetValue(providerIndividualClient) as ProviderSearchForProviderIndividualPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                // Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.CommonBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                searchForProviderIndividual request = HpiiHelper.MapHpiiIdentifierRequestData(hpiiIdentifierQuery);
                searchForProviderIndividualResponse svcResponse = providerIndividualClient.ProviderIndividualSearch(request);
                response = HpiiHelper.MapHpiiResponseData(svcResponse.searchForProviderIndividualResult, accessingOrganisation);
                response.HipsResponse.Status = HipsResponseIndicator.OK;
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();
                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        response.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                        response.HipsResponse.HipsErrorMessage = error.serviceMessage[0].reason;
                        response.HipsResponse.ResponseCode = error.serviceMessage[0].code;
                        response.ServiceMessagesType = new ServiceMessagesType();
                        response.ServiceMessagesType.highestSeverity = error.highestSeverity;
                        response.ServiceMessagesType.serviceMessage = error.serviceMessage;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    // Grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;
                response.HipsResponse.HipsErrorMessage = e.Message;
                // Grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerIndividualClient.SoapMessages, accessingOrganisation, hpiiIdentifierQuery: hpiiIdentifierQuery, hpiiQueryResponse: response);
                providerIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Validates the batch query data. Stores the validation result into
        /// the HipsResponse object in the response parameter.
        /// </summary>
        /// <param name="hpiiBatchAsyncSubmitRequest">The HPI-I batch asynchronous submission request.</param>
        /// <param name="response">The HPI-I batch asynchronous submission response.</param>
        private void ValidateQueryBatchData(HpiiBatchAsyncSubmitRequest hpiiBatchAsyncSubmitRequest, HpiiBatchAsyncSubmitResponse response, HospitalIdentifier facility)
        {
            // HPI-O must be provided at the root level
            if (hpiiBatchAsyncSubmitRequest.HpioNumber.IsNullOrEmptyWhitespace() && facility == null)
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                return;
            }

            // Loop over all items in the batch object
            foreach (HpiiIdentifierQuery query in hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries)
            {
                HpiiQueryResponse itemResponse = new HpiiQueryResponse();
                this.ValidateQueryData(query, itemResponse, facility);
                if (itemResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    response.HipsResponse = itemResponse.HipsResponse;
                    return;
                }
            }

            foreach (HpiiDemographicQuery query in hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries)
            {
                HpiiQueryResponse itemResponse = new HpiiQueryResponse();
                this.ValidateQueryData(query, itemResponse, facility);
                if (itemResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    response.HipsResponse = itemResponse.HipsResponse;
                    return;
                }
            }

            // No validation errors so at this time it is all ok
            response.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        /// <summary>
        /// Validates the HPI-I identifier query request. Stores the validation result into the HipsResponse object in the response parameter.
        /// </summary>
        /// <param name="hpiiIdentifierQuery">The HPI-I identifier query request.</param>
        /// <param name="response">The HPI-I query response.</param>
        private void ValidateQueryData(HpiiIdentifierQuery hpiiIdentifierQuery, HpiiQueryResponse response, HospitalIdentifier facility)
        {
            // HPI-O must always be provided
            if (hpiiIdentifierQuery.HpioNumber.IsNullOrEmptyWhitespace() && facility == null)
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                return;
            }

            // FamilyName must be provided
            if (string.IsNullOrEmpty(hpiiIdentifierQuery.FamilyName))
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.FamilyNameNotProvided;
                return;
            }

            // Either HPI-I or Registration ID must be provided
            if (string.IsNullOrEmpty(hpiiIdentifierQuery.HpiiNumber) && string.IsNullOrEmpty(hpiiIdentifierQuery.RegistrationId))
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.IdentifierNotProvided;
                return;
            }

            // Must not provide both HPI-I and Registration ID
            if (!string.IsNullOrEmpty(hpiiIdentifierQuery.HpiiNumber) && !string.IsNullOrEmpty(hpiiIdentifierQuery.RegistrationId))
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.MultipleIdentifiersProvided;
                return;
            }

            // No validation errors so at this time it is all ok
            response.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        /// <summary>
        /// Validates the HPI-I demographic query request. Stores the validation result into the HipsResponse object in the response parameter.
        /// </summary>
        /// <param name="hpiiDemographicQuery">The HPI-I demographic query request.</param>
        /// <param name="response">The HPI-I query response.</param>
        private void ValidateQueryData(HpiiDemographicQuery hpiiDemographicQuery, HpiiQueryResponse response, HospitalIdentifier facility)
        {
            // HPI-O must always be provided
            if (hpiiDemographicQuery.HpioNumber.IsNullOrEmptyWhitespace() && facility == null)
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                return;
            }

            // Address must be provided
            if (hpiiDemographicQuery.AustralianAddress == null && hpiiDemographicQuery.InternationalAddress == null)
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.AddressNotProvided;
                return;
            }

            // Must not provide both forms of Address
            if (hpiiDemographicQuery.AustralianAddress != null && hpiiDemographicQuery.InternationalAddress != null)
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.AustralianAndInternational;
                return;
            }

            // International address must have a country, must not be Australia 
            if (hpiiDemographicQuery.InternationalAddress != null)
            {
                if (!hpiiDemographicQuery.InternationalAddress.Country.HasValue)
                {
                    response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                    response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.CountryNotProvided;
                    return;
                }

                if (hpiiDemographicQuery.InternationalAddress.Country.Value == CountryType.Item1101)
                {
                    response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                    response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.CountryWasAustralia;
                    return;
                }
            }

            // FamilyName must be provided
            if (string.IsNullOrEmpty(hpiiDemographicQuery.FamilyName))
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.FamilyNameNotProvided;
                return;
            }

            // Sex must be provided
            if (!hpiiDemographicQuery.Sex.HasValue)
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.SexNotProvided;
                return;
            }

            // Date of birth must be provided
            if (!hpiiDemographicQuery.DateofBirth.HasValue)
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.DateOfBirthNotProvided;
                return;
            }

            // No validation errors so at this time it is all ok
            response.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        /// <summary>
        /// Validates the retrieve batch query data. Stores the validation result into the HipsResponse object in the response parameter.
        /// </summary>
        /// <param name="hpiiBatchAsyncRetrieveRequest">The HPI-I batch asynchronous retrieval request.</param>
        /// <param name="hpiiBatchAsyncRetrieveResponse">The HPI-I batch asynchronous retrieval response.</param>
        private void ValidateQueryData(HpiiBatchAsyncRetrieveRequest hpiiBatchAsyncRetrieveRequest, HpiiBatchAsyncRetrieveResponse hpiiBatchAsyncRetrieveResponse, HospitalIdentifier facility)
        {
            // HPI-O must always be provided
            if (hpiiBatchAsyncRetrieveRequest.HpioNumber.IsNullOrEmptyWhitespace() && facility == null)
            {
                hpiiBatchAsyncRetrieveResponse.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                hpiiBatchAsyncRetrieveResponse.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                return;
            }

            // No validation errors so at this time it is all ok
            hpiiBatchAsyncRetrieveResponse.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        #endregion Methods
    }
}