﻿using System;
using System.Configuration;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Exceptions;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas.Exceptions;
using nehta.mcaR3.ConsumerSearchIHI;
using HIPS.CommonSchemas.Cda.ParticipatingIndividual;
using System.Collections.Generic;

namespace HIPS.CommonBusinessLogic.Ihi
{
    public class PatientIhiValidation
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="PatientIhiValidation"/> class.
        /// </summary>
        /// <remarks>
        /// The value for the IHI validity period is extracted from the "IhiValidationPeriodDays" configuration setting.
        /// </remarks>
        public PatientIhiValidation()
        {
            this.IhiValidityPeriodHours = PatientIhiValidation.GetIhiValidationPeriodDays() * 24;
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="PatientIhiValidation"/> class.
        /// </summary>
        /// <param name="ihiValidityPeriodHours">The number of hours a locally stored IHI is considered valid for.</param>
        public PatientIhiValidation(int ihiValidityPeriodHours)
        {
            this.IhiValidityPeriodHours = ihiValidityPeriodHours;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the number of hours a locally stored IHI is considered valid for.
        /// </summary>
        private int IhiValidityPeriodHours { get; set; }

        #endregion
        
        #region Methods

        /// <summary>
        /// Gets the value of the "IhiValidationPeriodDays" configuration setting.
        /// </summary>
        /// <returns>Integer value containing the number of days configured.</returns>
        public static int GetIhiValidationPeriodDays()
        {
            int validationPeriod;
            int.TryParse(System.Configuration.ConfigurationManager.AppSettings["IhiValidationPeriodDays"], out validationPeriod);
            return validationPeriod;
        }

        /// <summary>
        /// Gets the value of the "UseStateIdentifierAsMrn" configuration setting.
        /// </summary>
        /// <returns>Flag indicating to Use State Identifier as MRN to create Hospital Patient.</returns>
        public bool GetUseStateIdentifierAsMrn()
        {
            bool useStateIdentifierAsMrn;
            bool.TryParse(System.Configuration.ConfigurationManager.AppSettings["UseStateIdentifierAsMrn"], out useStateIdentifierAsMrn);
            return useStateIdentifierAsMrn;
        }

        /// <summary>
        /// This is the external interface for clinical systems to request a
        /// valid IHI and registered demographics for use in a clinical
        /// document. This interface requires the date of birth to be provided
        /// as an additional identity check.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="dateOfBirth">Patient's date of birth (for identity verification)</param>
        /// <param name="user">The user responsible for this action</param>
        /// <returns>Validated IHI information</returns>
        public ValidatedIhiResponse GetValidatedIhi(PatientIdentifierBase patientIdentifier, DateTime dateOfBirth, UserDetails user)
        {
            ValidatedIhiResponse response = new ValidatedIhiResponse();
            try
            {
                PatientAccess patientAccess = new PatientAccess(user);
                Hospital hospital;
                HospitalPatient hospitalPatient;
                PatientMaster patientMaster;

                response.HipsResponse = patientAccess.GetHospital(patientIdentifier, out hospital);
                if (response.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return response;
                }
                response.HipsResponse = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
                if (response.HipsResponse.Status != HipsResponseIndicator.OK && response.HipsResponse.Status != HipsResponseIndicator.InvalidIhi)
                {
                    return response;
                }

                // Allow the input DOB to equal either the current DOB or the registered DOB
                DateTime currentDateOfBirth = patientMaster.DateOfBirth;
                if (dateOfBirth.Date == patientMaster.RegisteredDateOfBirth.Date)
                {
                    currentDateOfBirth = patientMaster.RegisteredDateOfBirth;
                }

                if (!currentDateOfBirth.Date.Equals(dateOfBirth.Date))
                {
                    response.HipsResponse.Status = HipsResponseIndicator.InvalidDateOfBirth;
                    response.HipsResponse.HipsErrorMessage = string.Format(ConstantsResource.DateOfBirthMismatch, dateOfBirth.Date, currentDateOfBirth.Date);
                    return response;
                }

                //Log to error if DOB and Registered Date of Birth not equal
                if (!ValidateInputDetails(patientMaster, patientIdentifier, hospital, dateOfBirth))
                {
                    // Incorrect date of birth could be due to unauthorised access and should be brought to attention of system administrator as well.
                    string description = string.Format(ConstantsResource.DateOfBirthRegisteredDateOfBirthMismatch, patientIdentifier, hospital, dateOfBirth, currentDateOfBirth);
                    EventLogger.WriteLog(description, null, user, LogMessage.HIPS_MESSAGE_084);
                }

                IhiSearchResponse ihiSearch = GetValidatedIhi(patientIdentifier, hospital, user, patientMaster, dateOfBirth);
                if (ihiSearch.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
                {
                    // When the HIB calls the IHI Service to retrieve the IHI number
                    // to embed into the CDA document it is important that an IHI
                    // number is returned even if the IHI is stale.  There could be
                    // a situation where the HI Service from Medicare is down where
                    // an IHI is attempted to be revalidated.  In this case the IHI
                    // may be correct, however, we can not prove this.  For this we
                    // need to ensure that we can use the stale IHI for adding to
                    // the CDA.
                    EventLogger.WriteLog(ConstantsResource.ReturningStaleIhi, new Exception(ihiSearch.HipsResponse.HipsErrorMessage), user, LogMessage.HIPS_MESSAGE_080);
                    response.HipsResponse.Status = HipsResponseIndicator.OK;
                    response.HipsResponse.HipsErrorMessage = ConstantsResource.ReturningStaleIhi;
                }
                else
                {
                    if (ihiSearch.HipsResponse.Status != HipsResponseIndicator.OK)
                    {
                        response.HipsResponse = ihiSearch.HipsResponse;
                        return response;
                    }
                    patientAccess.ValidateLocalIhiInformation(patientMaster, response.HipsResponse);
                    if (response.HipsResponse.Status != HipsResponseIndicator.OK)
                    {
                        return response;
                    }
                }

                // Log error if patient DOB and RegisteredDateOfBirth are not match
                if (!patientMaster.DateOfBirth.Date.Equals(patientMaster.RegisteredDateOfBirth.Date))
                {
                    EventLogger.WriteLog(ConstantsResource.DateOfBirthRegisteredDateOfBirthMismatch, new Exception(ihiSearch.HipsResponse.HipsErrorMessage), user, LogMessage.HIPS_MESSAGE_084);
                }

                PopulateResponse(response, patientIdentifier, hospitalPatient, patientMaster);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.IhiRequestFailed, ex, user, LogMessage.HIPS_MESSAGE_081);
                response.HipsResponse = new HipsResponse(HipsResponseIndicator.SystemError);
                response.HipsResponse.HipsErrorMessage = ex.Message;
                response.HipsResponse.ResponseCode = ex.GetType().ToString();
                response.HipsResponse.ResponseCodeDetails = ex.StackTrace;
                if (ex.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDetails = ex.InnerException.Message;
                }
                response.ValidatedIhi = null;
            }
            return response;
        }

        /// <summary>
        /// This is the internal interface for HIPS services that need to obtain
        /// a valid IHI. If the IHI is stale, i.e. it was last validated outside
        /// the configurable period, this method attempts to validate the IHI by
        /// passing the IHI and key demographic information to the Medicare HI
        /// Service.
        /// This method allows but does not require the date of birth to be
        /// provided as an additional identity check.
        /// If the patient is not found, or has a different date of birth to the
        /// one given, this method returns an empty response.
        /// </summary>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="hospital">The hospital which is requesting the validated IHI</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action</param>
        /// <param name="patientMaster">Optionally supply patient master record if already obtained</param>
        /// <param name="dateOfBirth">Optionally supply the patient's date of birth (for identity verification)</param>
        /// <returns>Validated IHI, registered Medicare first and last HL7Name, and sex code and description</returns>
        public IhiSearchResponse GetValidatedIhi(PatientIdentifierBase patientIdentifier, Hospital hospital, UserDetails user, PatientMaster patientMaster = null, DateTime? dateOfBirth = null)
        {
            IhiSearchResponse response = new IhiSearchResponse();
            response.IhiRecordStatus = IhiRecordStatus.Unknown;
            response.IhiStatus = IhiStatus.Unknown;

            if (!User.PopulateAndValidateUser(hospital, user))
            {
                // Attempted access with incorrect user details should be brought to system administrator's attention.
                EventLogger.WriteLog(ConstantsResource.InvalidUserDetails, null, user, LogMessage.HIPS_MESSAGE_082);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse.HipsErrorMessage = ConstantsResource.MissingUserValues;
                return response;
            }

            if (patientMaster == null)
            {
                // Caller has not yet obtained the patient master object, we will do so here.
                HospitalPatient hospitalPatient;
                PatientAccess patientAccess = new PatientAccess(user);
                HipsResponse getPatientResponse = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
                if (getPatientResponse.Status != HipsResponseIndicator.OK)
                {
                    // Unknown patient record should be brought to attention of system administrator.
                    string description = string.Format(ConstantsResource.CannotFindPatient, patientIdentifier, hospital.HospitalId);
                    EventLogger.WriteLog(description, null, user, LogMessage.HIPS_MESSAGE_083);
                    response.HipsResponse = getPatientResponse;
                    return response;
                }
            }

            // Log error if patient DOB and RegisteredDateOfBirth are not match
            if (!patientMaster.DateOfBirth.Date.Equals(patientMaster.RegisteredDateOfBirth.Date))
            {
                EventLogger.WriteLog(ConstantsResource.DateOfBirthRegisteredDateOfBirthMismatch, null, user, LogMessage.HIPS_MESSAGE_084);
            }

            if (!ValidateInputDetails(patientMaster, patientIdentifier, hospital, dateOfBirth))
            {
                DateTime currentDateOfBirth = patientMaster.DateOfBirth;
                if (dateOfBirth.Value.Date == patientMaster.RegisteredDateOfBirth.Date)
                {
                    currentDateOfBirth = patientMaster.RegisteredDateOfBirth;
                }

                // Incorrect date of birth could be due to unauthorised access and should be brought to attention of system administrator as well.
                string description = string.Format(ConstantsResource.InvalidIhiRequest, patientIdentifier, hospital, dateOfBirth, currentDateOfBirth);
                EventLogger.WriteLog(description, null, user, LogMessage.HIPS_MESSAGE_084);
                response.HipsResponse.Status = HipsResponseIndicator.InvalidPatient;                
                response.HipsResponse.HipsErrorMessage = string.Format(ConstantsResource.DateOfBirthMismatch, dateOfBirth.Value.Date, currentDateOfBirth.Date);
                return response;
            }

            try
            {
                if (!string.IsNullOrEmpty(patientMaster.Ihi))
                {
                    int validationPeriod;
                    int.TryParse(ConfigurationManager.AppSettings["IhiValidationPeriodDays"], out validationPeriod);
                    TimeSpan ts = DateTime.Now.Subtract(patientMaster.IhiLastValidated.Value);

                    if (ts.TotalDays >= validationPeriod || patientMaster.IhiStatusId != (int)IhiStatus.Active)
                    {
                        RevalidateIhi(patientMaster, hospital, user);
                    }
                    response.Ihi = patientMaster.Ihi;
                }
                else
                {
                    PatientRegistration registration = new PatientRegistration();
                    response.Ihi = registration.RegisterPatient(patientMaster, hospital, user);
                }
            }
            catch (IhiServiceUnavailableException ex)
            {
                response.HipsResponse.Status = HipsResponseIndicator.HiServiceError;
                response.HipsResponse.HipsErrorMessage = HIPS.CommonBusinessLogic.Pcehr.ResponseStrings.IhiRequiresValidation;
                response.HipsResponse.ResponseCodeDescription = ex.Message;
            }
            catch (PcehrServiceUnavailableException ex)
            {
                if (ex.Detail != null)
                {
                    // The details will be preserved.
                    response.HipsResponse = ex.Detail;
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.PcehrServiceError;
                    response.HipsResponse.HipsErrorMessage = HIPS.CommonBusinessLogic.Pcehr.ResponseStrings.PcehrSystemTemporarilyUnavailable;
                    response.HipsResponse.ResponseCodeDescription = ex.Message;
                }
            }
            response.IhiStatus = (IhiStatus)patientMaster.IhiStatusId;
            response.IhiRecordStatus = (IhiRecordStatus)patientMaster.IhiRecordStatusId;
            response.IhiLastValidated = patientMaster.IhiLastValidated;
            response.FirstName = patientMaster.LegalName.GivenNames;
            response.LastName = patientMaster.LegalName.FamilyName;

            ListSingleton lists = ListSingleton.Instance;
            Sex sex = lists.AllSexes.First(result => result.Id == patientMaster.RegisteredSexId);
            if (sex != null)
            {
                response.SexCode = sex.Code;
                response.SexDescription = sex.Description;
            }

            return response;
        }

        /// <summary>
        /// Sends the IHI and key demographic information to the Medicare HI
        /// Service to determine whether the IHI is still valid.
        /// </summary>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="hospital">The hospital whose HPI-O and Registered Officer is accessing the IHI</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action</param>
        public void RevalidateIhi(PatientMaster patientMaster, Hospital hospital, UserDetails user)
        {
            IhiSearchCriteria searchDetails = SearchHelper.PopulateSearchDetails(patientMaster);
            string resultMessage;
            searchIHIResponse response;
            bool success = Search.ByNumber(hospital, patientMaster, searchDetails, out resultMessage, out response, user, bool.Parse(ConfigurationManager.AppSettings["CheckDoesPcehrExist"].ToString()));
            patientMaster.Ihi = searchDetails.Ihi;
        }

        /// <summary>
        /// Obtains validated IHI information for use when receiving an e-health message.
        /// </summary>
        /// <param name="consumer">Participating Consumer</param>
        /// <param name="hpio">Health Provider Organisation Id</param>
        /// <param name="user">The user responsible for this action</param>
        /// <returns>Validated IHI information</returns>
        public ValidatedIhiResponse CheckReceivedValidatedIhi(ParticipatingConsumer consumer, string hpio, UserDetails user)
        {
            List<PatientMaster> patientMasters;
            Hospital hospital;
            PatientMasterId patientMasterId;

            var patientAccess = new PatientAccess(user);

            // Validate If HPIO exists in HIPS
            var IsHpioExistResponse = patientAccess.ValidateHpio(hpio);

            if (IsHpioExistResponse.Status != HipsResponseIndicator.OK)
            {
                // HPIO does not exists on HIPS
                return new ValidatedIhiResponse() { HipsResponse = IsHpioExistResponse };
            }

            // Check Response and Get Patient Master
            var response = patientAccess.FindLocalPatientsForReceivedMessage(hpio, consumer, out patientMasters);

            // If we didn't receive any PatientMaster objects, then there are no local patient records to validate, so return the response.
            if (patientMasters.Count == 0)
            {
                return new ValidatedIhiResponse() { HipsResponse = response };
            }

            // Determine the validity period. If the configured period is zero days, we
            // will use that value, otherwise the maximum we can use is 24 hours.
            int validityPeriodHours;
            if (response.Status == HipsResponseIndicator.OK)
            {
                // The patient records matched the incoming message in both IHI and demographic details.
                // Validation of their IHI can be skipped if they were already validated within the last 24 hours.
                var validityPeriodDays = PatientIhiValidation.GetIhiValidationPeriodDays();
                validityPeriodHours = validityPeriodDays == 0 ? 0 : 24;
            }
            else
            {
                // There is some problem with the patients returned. They don't match the incoming details, or they don't have an IHI.
                // The validation must proceed even if already validated in the configured period.
                validityPeriodHours = 0;
            }
            var validationLogic = new PatientIhiValidation(validityPeriodHours);

            // Validate the local patient records that were returned.
            List<ValidatedIhiResponse> validationResponses = new List<ValidatedIhiResponse>();
            foreach (var patientMaster in patientMasters)
            {
                // Find a hospital facility in which the patient is already registered, which belongs to the receiving healthcare provider organisation (preferably the HPI-O itself, otherwise another HPI-O in the same network).
            bool isHospitalMatched = patientAccess.GetHospitalByPatientMasterIdAndHpio(patientMaster.Id.Value, hpio, out hospital, user);

            // If hospital matched
            if (isHospitalMatched)
            {
                // Create a new patient identifier for our local data
                patientMasterId = new PatientMasterId(patientMaster.Id.Value, hospital.Codes[0].Code, hospital.Codes[0].CodeSystemCode, null);

                // Utilise PatientIhiValidation.GetValidatedIhi to perform the validation.
                    var validationResponse = validationLogic.GetValidatedIhi(patientMasterId, patientMaster.DateOfBirth, user);

                    validationResponses.Add(validationResponse);
                }
                else
                {
                    // There is no way to validate the patient for the receiving organisation.
                    validationResponses.Add(
                        new ValidatedIhiResponse
                        {
                            HipsResponse = new HipsResponse(
                                HipsResponseIndicator.InvalidPatient,
                                "A candidate patient record was found in another facility but cannot be validated for the receiving facility.")
                        });
                }
            }

            var exactMatches = validationResponses.Where(
                v => v.ValidatedIhi != null && patientAccess.IsReceivedPatientMatch(
                    consumer,
                    v.ValidatedIhi.Ihi,
                    v.ValidatedIhi.FamilyName,
                    v.ValidatedIhi.GivenName,
                    v.ValidatedIhi.Sex,
                    v.ValidatedIhi.DateOfBirth));

            if (exactMatches.Any())
            {
                // The incoming IHI and demographic data now matches the IHI and demographic data in a local patient record.
                // It may have been OK all along, or validation may have resolved the IHI mismatch or missing IHI.
                // Any of the records that matched may be returned. One is no better than any another.
                return exactMatches.First();
            }
            else if (response.Status == HipsResponseIndicator.OK)
            {
                // There was a match, but validating the local record has caused a mismatch.
                // This could happen if Medicare recently expired the IHI.
                return validationResponses.First();
            }
            else if (response.Status == HipsResponseIndicator.DemographicMismatchWarning)
            {
                // The incoming IHI still does not match any local patient record.
                // An alert will be raised and the none of the candidate records will be returned.
                return new ValidatedIhiResponse { HipsResponse = response };
            }
            else if (response.Status == HipsResponseIndicator.InvalidIhi)
            {
                // Local patient found by demographics, but has no IHI assigned in local system. 
                if (validationResponses.Any(v => v.ValidatedIhi != null && v.ValidatedIhi.Ihi != consumer.Ihi))
                {
                    // The IHI retrieved from the HI Service does not match with the incoming IHI.
                    return new ValidatedIhiResponse { HipsResponse = new HipsResponse(HipsResponseIndicator.UnresolvedIhiAlert, "The IHI retrieved from the HI Service does not match with the incoming IHI.") };
                }
                else
                    {
                    // No IHI was retrieved from the HI Service for the candidate records.
                    return new ValidatedIhiResponse { HipsResponse = response };
                }
                    }
            else
                    {
                return new ValidatedIhiResponse { HipsResponse = response };
            }
                    }

        private ValidatedIhiResponse CombineResponses(List<ValidatedIhiResponse> responses)
        {
            ValidatedIhiResponse combinedResponse = responses.First();
            foreach (var response in responses.Skip(1))
            {
                if (response.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    // Combine the string fields by appending each one.
                    combinedResponse.HipsResponse.ResponseCode += response.HipsResponse.ResponseCode;
                    combinedResponse.HipsResponse.ResponseCodeDescription += combinedResponse.HipsResponse.ResponseCodeDescription;
                    combinedResponse.HipsResponse.ResponseCodeDetails += combinedResponse.HipsResponse.ResponseCodeDetails;
                    combinedResponse.HipsResponse.HipsErrorMessage += response.HipsResponse.HipsErrorMessage;

                }

                // Allow each response to overwrite the basic status if it indicates OK, this gives at least some indication that something was not OK.
                if (combinedResponse.HipsResponse.Status == HipsResponseIndicator.OK)
                {
                    combinedResponse.HipsResponse.Status = response.HipsResponse.Status;
                }
            }

            return combinedResponse;
        }

        /// <summary>
        /// Sends the new Medicare and DVA number to the Medicare HI Service
        /// and checks whether it finds the same IHI as the current IHI for
        /// the patient. If it finds nothing, or finds a different IHI, then
        /// restores the current IHI with a status of MedicareDvaChangeMismatch.
        /// </summary>
        /// <param name="patientMaster">The updated patient master.</param>
        /// <param name="hospital">The hospital.</param>
        /// <param name="user">The person who is responsible for the action.</param>
        internal HipsResponse RevalidateMedicareDva(PatientMaster patientMaster, Hospital hospital, UserDetails user)
        {
            // Retain the current IHI information for the patient, as it may be overwritten with a different IHI when we search by Medicare or DVA.
            PatientMasterIhi currentIhi = new PatientMasterIhi();
            currentIhi.PatientMasterId = patientMaster.PatientMasterId.Value;
            currentIhi.Ihi = patientMaster.Ihi;
            currentIhi.IhiStatusId = patientMaster.IhiStatusId;
            currentIhi.IhiRecordStatusId = patientMaster.IhiRecordStatusId;
            currentIhi.RegisteredFamilyName = patientMaster.RegisteredFamilyName;
            currentIhi.RegisteredGivenName = patientMaster.RegisteredGivenName;
            currentIhi.RegisteredSexId = patientMaster.RegisteredSexId;
            currentIhi.MedicareNumber = patientMaster.MedicareNumber;
            currentIhi.MedicareNumberSequence = patientMaster.MedicareIrn;
            currentIhi.DvaNumber = patientMaster.DvaNumber;
            currentIhi.DateOfBirth = patientMaster.DateOfBirth;
            currentIhi.RegisteredDateOfBirth = (Settings.Instance.RegisteredDateOfBirthEnabled ? patientMaster.RegisteredDateOfBirth : patientMaster.DateOfBirth);
            currentIhi.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;

            // Temporarily clear the IHI and do a search on the new Medicare and DVA numbers
            patientMaster.Ihi = null;
            IhiSearchCriteria searchDetails = SearchHelper.PopulateSearchDetails(patientMaster);
            string resultMessage;
            searchIHIResponse response;
            bool success = Search.ByNumber(hospital, patientMaster, searchDetails, out resultMessage, out response, user, bool.Parse(ConfigurationManager.AppSettings["CheckDoesPcehrExist"].ToString()));

            if (!success || currentIhi.Ihi != patientMaster.Ihi)
            {
                // The search with the new details got a different IHI. Put the current IHI back and mark it as demographic mismatch.
                currentIhi.IhiStatusId = (int)IhiStatus.MedicareDvaChangeMismatch;
                currentIhi.DateLastValidated = DateTime.Now;
                PatientMasterIhiDl dataAccess = new PatientMasterIhiDl(user);
                if (!dataAccess.Update(currentIhi))
                {
                    return new HipsResponse(HipsResponseIndicator.DatabaseError);
                }
                patientMaster.Ihi = currentIhi.Ihi;
                patientMaster.IhiStatusId = currentIhi.IhiStatusId;
                patientMaster.IhiRecordStatusId = currentIhi.IhiRecordStatusId;
                patientMaster.RegisteredFamilyName = currentIhi.RegisteredFamilyName;
                patientMaster.RegisteredGivenName = currentIhi.RegisteredGivenName;
                patientMaster.RegisteredSexId = currentIhi.RegisteredSexId;
                return new HipsResponse(HipsResponseIndicator.InvalidIhi);
            }
            return new HipsResponse(HipsResponseIndicator.OK);
        }

        /// <summary>
        /// Populates the MRN, State Patient ID and Validated IHI information into the response object.
        /// </summary>
        /// <param name="response">The response object to be populated.</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="hospitalPatient">The hospital patient</param>
        /// <param name="patientMaster">The patient master.</param>
        private static void PopulateResponse(ValidatedIhiResponse response, PatientIdentifierBase patientIdentifier, HospitalPatient hospitalPatient, PatientMaster patientMaster)
        {
            response.Mrn = new Mrn(hospitalPatient.Mrn, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            response.StatePatientId = new StatePatientId(patientMaster.StatePatientId, patientIdentifier.HospitalCode, patientIdentifier.HospitalCodeSystem);
            response.ValidatedIhi = new ValidatedIhi(
                patientMaster.Ihi,
                (IhiStatus)patientMaster.IhiStatusId,
                (IhiRecordStatus)patientMaster.IhiRecordStatusId,
                patientMaster.IhiLastValidated.Value,
                patientMaster.RegisteredFamilyName,
                patientMaster.RegisteredGivenName,
                patientMaster.RegisteredDateOfBirth,
                (SexEnumerator)patientMaster.RegisteredSexId,
                patientIdentifier.HospitalCode,
                patientIdentifier.HospitalCodeSystem);
        }

        /// <summary>
        /// Compares the two string values to make sure they are the same.
        /// Strings that are both null or empty are considered the same.
        /// Strings where one is null or empty and the other is not are different.
        /// Otherwise the strings are the same if they contain the same characters
        /// after they are both converted to uppercase.
        /// </summary>
        /// <param name="value1">The first value.</param>
        /// <param name="value2">The other value.</param>
        /// <returns>True if the same, otherwise false.</returns>
        private bool CompareString(string value1, string value2)
        {
            if (string.IsNullOrEmpty(value1) && string.IsNullOrEmpty(value2))
            {
                return true;
            }
            if (string.IsNullOrEmpty(value1) && !string.IsNullOrEmpty(value2))
            {
                return false;
            }
            if (string.IsNullOrEmpty(value2) && !string.IsNullOrEmpty(value1))
            {
                return false;
            }
            if (value1.ToUpper() == value2.ToUpper())
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// Validates the input of the Service call matches the details stored against the patient's details.
        /// This will help to prevent IHIs being returned for the incorrect patient.
        /// The details required for validation are:
        ///     Date Of Birth
        /// </summary>
        /// <param name="patientMasterId">The patient master ID</param>
        /// <param name="patientIdentifier">Patient identifier (Hospital-level MRN, State Patient ID, Validated IHI or PCEHR Data Store PatientMasterId)</param>
        /// <param name="pasHospitalCode">The input hospital code</param>
        /// <param name="inDateOfBirth">The input date of birth</param>
        /// <returns>Whether validation was successful</returns>
        private bool ValidateInputDetails(PatientMaster patientMaster, PatientIdentifierBase patientIdentifier, Hospital hospital, DateTime? inDateOfBirth)
        {
            DateTime currentDateOfBirth = patientMaster.DateOfBirth;

            if (inDateOfBirth.HasValue)
            {
                if (inDateOfBirth.Value.Date != patientMaster.RegisteredDateOfBirth.Date && inDateOfBirth.Value.Date != patientMaster.DateOfBirth.Date)
                {
                    // This is potentially a data quality issue and should be brought to attention of those processing the IHI lookup alerts.
                    IhiLookupAlertBl.Insert(patientMaster.PatientMasterId.Value, string.Format(ConstantsResource.InvalidIhiRequest, patientIdentifier, hospital, inDateOfBirth, currentDateOfBirth), null, null);
                    return false;
                }
            }

            return true;
        }

        #endregion Methods
    }
}