﻿using System;
using System.Collections.Generic;

using AutoMapper;
using DatabaseSchemas = HIPS.PcehrDataStore.Schemas;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Contact entity.
    /// </summary>
    internal class ContactProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Database --> Model

            this.CreateMap<DatabaseSchemas.Contact, ModelSchemas.Cda.ElectronicCommunicationDetail>()
                .ForMember(dest => dest.Medium, opt => opt.ResolveUsing((DatabaseSchemas.Contact src) =>
                    {
                        switch (src.ContactMethodDescription)
                        {
                            case "Home Phone": return ModelSchemas.Cda.ElectronicCommunicationMedium.FixedTelephone;
                            case "Home Fax": return ModelSchemas.Cda.ElectronicCommunicationMedium.Fax;
                            case "Personal Mobile": return ModelSchemas.Cda.ElectronicCommunicationMedium.Mobile;
                            case "Personal Email": return ModelSchemas.Cda.ElectronicCommunicationMedium.Email;
                            case "Pager": return ModelSchemas.Cda.ElectronicCommunicationMedium.Pager;
                            case "Work Phone": return ModelSchemas.Cda.ElectronicCommunicationMedium.FixedTelephone;
                            case "Work Fax": return ModelSchemas.Cda.ElectronicCommunicationMedium.Fax;
                            case "Work Mobile": return ModelSchemas.Cda.ElectronicCommunicationMedium.Mobile;
                            case "Work Email": return ModelSchemas.Cda.ElectronicCommunicationMedium.Email;
                            case "home email address": return ModelSchemas.Cda.ElectronicCommunicationMedium.Email;
                        }
                        return ModelSchemas.Cda.ElectronicCommunicationMedium.None;
                    }))
                .ForMember(dest => dest.Usage, opt => opt.ResolveUsing((DatabaseSchemas.Contact src) =>
                    {
                        switch (src.ContactMethodDescription)
                        {
                            case "Home Phone": return ModelSchemas.Cda.ElectronicCommunicationUsage.Personal;
                            case "Home Fax": return ModelSchemas.Cda.ElectronicCommunicationUsage.Personal;
                            case "Personal Mobile": return ModelSchemas.Cda.ElectronicCommunicationUsage.Personal;
                            case "Personal Email": return ModelSchemas.Cda.ElectronicCommunicationUsage.Personal;
                            case "Pager": return ModelSchemas.Cda.ElectronicCommunicationUsage.Business;
                            case "Work Phone": return ModelSchemas.Cda.ElectronicCommunicationUsage.Business;
                            case "Work Fax": return ModelSchemas.Cda.ElectronicCommunicationUsage.Business;
                            case "Work Mobile": return ModelSchemas.Cda.ElectronicCommunicationUsage.Business;
                            case "Work Email": return ModelSchemas.Cda.ElectronicCommunicationUsage.Business;
                            case "home email address": return ModelSchemas.Cda.ElectronicCommunicationUsage.Personal;
                        }
                        return ModelSchemas.Cda.ElectronicCommunicationUsage.None;
                    }))
                .ForMember(dest => dest.Detail, opt => opt.MapFrom(src => src.Detail));

            // Model --> NEHTA

            this.CreateMap<ModelSchemas.Cda.ElectronicCommunicationDetail, NehtaSchemas.Common.ElectronicCommunicationDetail>()
                .ForMember(dest => dest.Address, opt => opt.MapFrom(src => src.Detail))
                .ForMember(dest => dest.Medium, opt => opt.ResolveUsing((ModelSchemas.Cda.ElectronicCommunicationDetail src) =>
                {
                    switch (src.Medium)
                    {
                        case ModelSchemas.Cda.ElectronicCommunicationMedium.Email:
                            return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Email;

                        case ModelSchemas.Cda.ElectronicCommunicationMedium.Fax:
                            return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Fax;

                        case ModelSchemas.Cda.ElectronicCommunicationMedium.FixedTelephone:
                            return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Telephone;

                        case ModelSchemas.Cda.ElectronicCommunicationMedium.Mobile:
                            return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Mobile;

                        case ModelSchemas.Cda.ElectronicCommunicationMedium.Pager:
                            return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Page;
                    }
                    Uri uri = new Uri(src.Detail);
                    switch (uri.Scheme)
                    {
                        case "ftp": return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.FTP;
                        case "http": return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.HTTP;
                        case "mllp": return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.MLLP;
                        case "modem": return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Modem;
                        case "nfs": return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.NFS;
                        case "telnet": return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Telnet;
                    }
                    return NehtaSchemas.Common.Enums.ElectronicCommunicationMedium.Undefined;
                }))
                .ForMember(dest => dest.Usage, opt => opt.ResolveUsing((ModelSchemas.Cda.ElectronicCommunicationDetail src) =>
                {
                    var list = new List<NehtaSchemas.Common.Enums.ElectronicCommunicationUsage>();
                    switch (src.Usage)
                    {
                        case ModelSchemas.Cda.ElectronicCommunicationUsage.Personal:
                            list.Add(NehtaSchemas.Common.Enums.ElectronicCommunicationUsage.Home);
                            break;

                        case ModelSchemas.Cda.ElectronicCommunicationUsage.Business:
                            list.Add(NehtaSchemas.Common.Enums.ElectronicCommunicationUsage.WorkPlace);
                            break;

                        case ModelSchemas.Cda.ElectronicCommunicationUsage.Both:
                            list.Add(NehtaSchemas.Common.Enums.ElectronicCommunicationUsage.WorkPlace);
                            list.Add(NehtaSchemas.Common.Enums.ElectronicCommunicationUsage.Home);
                            break;
                    }
                    switch (src.Medium)
                    {
                        case ModelSchemas.Cda.ElectronicCommunicationMedium.Mobile:
                            list.Add(NehtaSchemas.Common.Enums.ElectronicCommunicationUsage.MobileContact);
                            break;

                        case ModelSchemas.Cda.ElectronicCommunicationMedium.Pager:
                            list.Add(NehtaSchemas.Common.Enums.ElectronicCommunicationUsage.Pager);
                            break;
                    }
                    return list;
                }));

            // Database --> NEHTA  (using ConvertUsing to combine the two maps above)
            this.CreateMap<DatabaseSchemas.Contact, NehtaSchemas.Common.ElectronicCommunicationDetail>()
                .ConvertUsing(src => ObjectMapper.Map<NehtaSchemas.Common.ElectronicCommunicationDetail>(ObjectMapper.Map<ModelSchemas.Cda.ElectronicCommunicationDetail>(src)));
        }

        #endregion Methods
    }
}