﻿using System;
using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;
using HIPS.Configuration;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Author, Legal Authenticator and Responsible Health Professional using the ParticipatingProvider entity.
    /// </summary>
    internal class ProviderProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA

            #region Common

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.Common.ParticipationPeriod>()
                .ForMember(dest => dest.Value, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, List<ModelSchemas.Cda.EntityIdentifier>>()
                .ConvertUsing((ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider src) => new List<ModelSchemas.Cda.EntityIdentifier>()
                {
                    src.Hpii != null ? new ModelSchemas.Cda.EntityIdentifier
                    {
                        AssigningAuthorityName = "HPI-I",
                        AssigningGeographicArea = "National Identifier",
                        Root = "1.2.36.1.2001.1003.0." + src.Hpii
                    } : null,
                    src.LocalIdentifier != null ? new ModelSchemas.Cda.EntityIdentifier
                    {
                        AssigningAuthorityName = src.EmployerName,
                        Root = "1.2.36.1.2001.1005.41." + src.EmployerHpio,
                        Extension = src.LocalIdentifier,
                        Code = new ModelSchemas.Cda.CdaCode
                        {
                            Code = "EI",
                            CodeSystem = "2.16.840.1.113883.12.203",
                            CodeSystemName = "Identifier Type (HL7)"
                        }
                    } : null,
                }.Where(a => a != null).ToList());

            #endregion

            #region Discharge Summary

            // Discharge Summary - CDAContext - LegalAuthenticator
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationLegalAuthenticator>()
               .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateLegalAuthenticator())
               .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            // Discharge Summary - CDAContext - LegalAuthenticator - Participant
            // Mapping for Person: Person Profile (Discharge Summary - CDAContext - LegalAuthenticator - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.ILegalAuthenticator>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForLegalAuthenticator())
                .ForMember(dest => dest.DateTimeAuthenticated, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src.Metadata.LegalAuthenticator));

            // Discharge Summary - SCSContext - Author            
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationDocumentAuthor>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateAuthor())
                .ForMember(dest => dest.AuthorParticipationPeriodOrDateTimeAuthored, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src.Metadata.DocumentAuthor))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode
                {
                    Code = "253000",
                    CodeSystem = "2.16.840.1.113883.13.62",
                    CodeSystemName = "1220.0 - ANZSCO - Australian and New Zealand Standard Classification of Occupations, First Edition, 2006",
                    DisplayName = "Medical Practitioners nfd"
                }));

            // Discharge Summary - SCSContext - Author - Participant
            // Mapping for Person: Person Profile (Discharge Summary - SCSContext - Author - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IAuthor>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForAuthor())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.ResolveUsing((ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider src) => Guid.NewGuid()));

            // Discharge Summary - SCSContent - Event - Encounter - ResponsibleHealthProfessional
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.IParticipationResponsibleHealthProfessional>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateResponsibleHealthProfessional())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode
                {
                    Code = "253000",
                    CodeSystem = "2.16.840.1.113883.13.62",
                    CodeSystemName = "1220.0 - ANZSCO - Australian and New Zealand Standard Classification of Occupations, First Edition, 2006",
                    DisplayName = "Medical Practitioners nfd"
                }));
            
            // Discharge Summary - SCSContent - Event - Encounter - ResponsibleHealthProfessional - Participant
            // Mapping for Person: Person Profile (Discharge Summary - SCSContent - Event - Encounter - ResponsibleHealthProfessional - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.IResponsibleHealthProfessional>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForResponsibleHealthProfessional())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src));          
            
            #endregion

            #region Diagnostic Imaging

            // Diagnostic Imaging - CDAContext - LegalAuthenticator
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationLegalAuthenticator>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateLegalAuthenticator())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            // Diagnostic Imaging - CDAContext - LegalAuthenticator - Participant
            // Mapping for Person: Person Profile (Diagnostic Imaging - CDAContext - LegalAuthenticator - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.ILegalAuthenticator>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateParticipantForLegalAuthenticator())
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()))
                .ForMember(dest => dest.DateTimeAuthenticated, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src.Metadata.LegalAuthenticator));

            // Diagnostic Imaging - SCSContext - Author
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, CDA.Generator.Common.SCSModel.Interfaces.IParticipationAuthorHealthcareProvider>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateAuthorHealthcareProvider())
                .ForMember(dest => dest.AuthorParticipationPeriodOrDateTimeAuthored, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src.Metadata.DocumentAuthor))
                .ForMember(dest => dest.Role, opt => opt.ResolveUsing<RoleResolver>().FromMember(src => src));

            // Diagnostic Imaging - SCSContext - Author - Participant
            // Mapping for Person: Person profile (Diagnostic Imaging - SCSContext - Author - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IAuthor>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateParticipantForAuthor())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.ResolveUsing((ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider src) => Guid.NewGuid()));

            // Diagnostic Imaging - SCSContext - ReportingRadiologist
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationReportingRadiologist>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateReportingRadiologist())
                .ForMember(dest => dest.ParticipationEndTime, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => (src.Metadata.DocumentAuthor.FamilyName == "NI" ? NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.NullFlavour.NoInformation) : NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.Occupation.DiagnosticandInterventionalRadiologist))));


            // Diagnostic Imaging - SCSContext - ReportingRadiologist - Participant
            // Mapping for Person: Person profile (Diagnostic Imaging - SCSContext - ReportingRadiologist - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IReportingRadiologist>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.DiagnosticImagingReport.CreateParticipantForReportingRadiologist())
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()))
                .ForMember(dest => dest.Addresses, opt => opt.MapFrom(src => src.Hospital.Addresses))
                .ForMember(dest => dest.ElectronicCommunicationDetails, opt => opt.MapFrom(src => src.Hospital.Contacts))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src.Metadata.ResponsibleHealthProfessional));

            #endregion

            #region Pathology

            // Pathology - CDAContext - LegalAuthenticator
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationLegalAuthenticator>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateLegalAuthenticator())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            // Pathology - CDAContext - LegalAuthenticator - Participant
            // Mapping for Person: Person Profile (Pathology - CDAContext - LegalAuthenticator - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.ILegalAuthenticator>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateParticipantForLegalAuthenticator())
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()))
                .ForMember(dest => dest.DateTimeAuthenticated, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src.Metadata.LegalAuthenticator));

            // Pathology - SCSContext - Author
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, CDA.Generator.Common.SCSModel.Interfaces.IParticipationAuthorHealthcareProvider>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateAuthorHealthcareProvider())
               .ForMember(dest => dest.AuthorParticipationPeriodOrDateTimeAuthored, opt => opt.MapFrom(src => src))
               .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src.Metadata.DocumentAuthor))
               .ForMember(dest => dest.Role, opt => opt.ResolveUsing<RoleResolver>().FromMember(src => src));

            // Pathology - SCSContext - Author - Participant
            // Mapping for Person: Person Profile (Pathology - SCSContext - Author - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, CDA.Generator.Common.SCSModel.Interfaces.IAuthorHealthcareProvider>()
               .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateParticipantForAuthorHealthcareProvider())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.ResolveUsing((ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider src) => Guid.NewGuid()));

            // Pathology - SCSContext - ReportingPathologist
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationReportingPathologist>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateReportingPathologist())
                .ForMember(dest => dest.ParticipationEndTime, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => (src.Metadata.DocumentAuthor.FamilyName == "NI" ? NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.NullFlavour.NoInformation) : NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.Occupation.Pathologist))));


            // Pathology - SCSContext - ReportingPathologist - Participant
            // Mapping for Person: Person profile (Pathology - SCSContext - ReportingPathologist - Participant - Person)
            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IReportingPathologist>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.PathologyResultReport.CreateParticipantForReportingPathologist())
                .ForMember(dest => dest.Addresses, opt => opt.MapFrom(src => src.Hospital.Addresses))
                .ForMember(dest => dest.ElectronicCommunicationDetails, opt => opt.MapFrom(src => src.Hospital.Contacts))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src.Metadata.ResponsibleHealthProfessional))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.MapFrom(src => Guid.NewGuid()));
            
            #endregion
        }

        #endregion Methods

        /// <summary>
        /// Resolves the value of Author Roles.
        /// </summary>
        internal class RoleResolver : ValueResolver<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.ICodableText >
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">cdaPathDIDocumentDetails representing the source.</param>
            /// <returns>ICodeableText.</returns>
            protected override NehtaSchemas.SCSModel.ICodableText ResolveCore(ModelSchemas.Cda.CdaDocumentDetails source)
            {
                if (source.Metadata.DocumentAuthor.FamilyName == "NI")
                    return NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.NullFlavour.NoInformation);
                else if (source.DocumentFormatCode == Settings.Instance.PathologyReportDocumentFormatCode) // Path Report
                    return NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.Occupation.Pathologist);
                else if (source.DocumentFormatCode == Settings.Instance.DiagnosticImagingReportDocumentFormatCode) // DI Report)
                    return NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.Occupation.DiagnosticandInterventionalRadiologist);
                else // General Medical Practictioner
                    return NehtaSchemas.Common.BaseCDAModel.CreateRole(NehtaSchemas.Common.Enums.Occupation.MedicalPractitionersnfd);
            }

            #endregion Methods
        }
    }
}