﻿using HIPS.CommonBusinessLogic.Cda;
using HIPS.CommonBusinessLogic.HL7;
using HIPS.CommonBusinessLogic.Mapping;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.CommonSchemas.Exceptions;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.Configuration.Tracing;
using HIPS.HL7.Common;
using HIPS.HL7.Common.DataStructure;
using HIPS.HL7.Common.Exceptions;
using HIPS.HL7.Common.Message;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrQueueLogic;
using HIPS.PcehrSchemas;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Transactions;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class HL7ReportUpload
    {
        #region private members

        /// <summary>
        /// The queued upload operation object that was stored in the MSMQ.
        /// </summary>
        private QueuedHL7ReportOperation operation;

        #endregion private members

        #region constructor

        /// <summary>
        /// Initializes a new instance of the <see cref="HL7ReportUpload" /> class.
        /// </summary>
        /// <param name="queuedHl7ReportOperation">The queued HL7 Report operation.</param>
        public HL7ReportUpload(QueuedHL7ReportOperation queuedHl7ReportOperation)
        {
            this.operation = queuedHl7ReportOperation;
        }

        #endregion constructor

        #region Methods

        /// <summary>
        /// Validates the HL7 Pathology Message, creates the CDA document and calls the MSMQ UploadOrSupersede document method to
        /// upload the document to the PCEHR.
        /// </summary>
        /// <exception cref="System.InvalidOperationException">When the MSMQ operation should be retried</exception>
        public void UploadPathology()
        {
            DocumentQueueTransactionHandler handler = new DocumentQueueTransactionHandler(operation.User);
            // Restore the report and the pending item to the QueuedHL7ReportOperation from the data store.
            if (!handler.RestorePackageInQueuedHL7ReportOperation(operation))
            {
                return;
            }

            if (operation.PendingItem.QueueStatusId != (int)QueueStatus.Pending)
            {
                string message = string.Format(ResponseStrings.HL7ReportOperationAbortedDetail,
                   operation.PendingItem.PcehrMessageQueueId,
                   operation.PendingItem.QueueOperationName,
                   operation.HL7MessageLogId);
                handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationAborted, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_086);
                if (string.IsNullOrEmpty(operation.PendingItem.Details))
                {
                    operation.PendingItem.Details = message;
                }

                handler.PlaceAcknowledgementOnQueue(operation.PendingItem, operation.HL7MessageLogId);
                return;
            }

            // Invoke the PopulateHL7MessageLog to retrieve the message log entity and HL7 Message details.
            HL7MessageLog pathMessageLog;
            handler.GetMessageLog(operation.HL7MessageLogId, out pathMessageLog);
            HIPS.HL7.Common.HL7Message pathMessage = HL7Message.Parse(this.operation.HL7Message);

            HL7GenericPathMessage genericPathMessage = pathMessage as HL7GenericPathMessage;
            HIPS.HL7.Common.Segment.PID pid = genericPathMessage.PatientIdentification;

            HL7GenericPasMessage genericPasMessage = new HL7GenericPasMessage();
            genericPasMessage.PatientIdentification = genericPathMessage.PatientIdentification;

            // Validate the PID segment - return an error if it is not valid
            string pidValidationError = ValidatePatientIdentifiers(pid);
            if (!string.IsNullOrEmpty(pidValidationError))
            {
                handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationAborted, new Exception(pidValidationError), operation.User, LogMessage.HIPS_MESSAGE_086);
                UpdatePendingItemForFailure(pidValidationError);
                return;
            }


            try
            {
                // Process the HL7 Message
                Mrn patientMrn;
                CX secondaryId;
                OrderDetails orderDetails = new OrderDetails();
                CdaHeaderMetadata cdaHeaderMetadata = new CdaHeaderMetadata();
                string relatedDocumentStatus;
                int hospitalID;
                List<PathologyTestResult> pathologyTestResults = new List<PathologyTestResult>();
                List<ImagingExamResult> imagingExamResults = null;
                Episode episode;
                new PDILoader(operation.User).Process(DocumentTypeCodes.PathologyReport, genericPathMessage, out secondaryId, out patientMrn, out hospitalID, out cdaHeaderMetadata, out orderDetails, out imagingExamResults, out pathologyTestResults, out relatedDocumentStatus, out episode);

                // Update the Message Log with the details
                pathMessageLog.HospitalID = hospitalID;
                pathMessageLog.HospitalPatientIdentifier = patientMrn.Value;
                handler.UpdateMessageLog(pathMessageLog);

                // Now check if there is an earlier message sitting on the queue waiting to be processed for that patient
                // If yes - throw an exception but don't fail item so we can retry
                if (handler.CheckForEarlierPendingMessage(pathMessageLog.HospitalPatientIdentifier, pathMessageLog.HospitalID.Value, (Settings.Instance.UseHL7MessageDateTime && pathMessageLog.DateTimeOfMessage.HasValue ? pathMessageLog.DateTimeOfMessage.Value : pathMessageLog.DateCreated)))
                {
                    string message = string.Format(ResponseStrings.OperationOutOfOrderUploadHL7,
                        operation.PendingItem.PcehrMessageQueueId,
                        operation.PendingItem.QueueOperationName,
                        pathMessageLog.MessageControlId,
                        pathMessageLog.HospitalPatientIdentifier,
                        pathMessageLog.HospitalID);
                    handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationRetriedDueToMessagesOutofOrder, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_136);
                    throw new InvalidOperationException();
                }

                // Update pending item episode
                if (episode.EpisodeId.HasValue)
                    operation.PendingItem.EpisodeId = episode.EpisodeId.Value;

                string patientIndigenousStatus = pid.Race.FirstOrDefault().identifier.ToString();

                // Invoke the CreatePathologyResultReport passing in the report, the CdaHeaderMetaData, primaryID as the patientIdentifier, the OrderDetails, the list of pathologyTestResult’s, the secondaryID.Value as the requestingHospitalPatientMrn. This will return a CdaDocumentDetails object.
                PDIUploadService pdiReport = new PDIUploadService(this.operation.User);
                CdaPathDIDocumentDetails cdaDetails = pdiReport.CreatePathologyResultReport(this.operation.Report, cdaHeaderMetadata, patientMrn, orderDetails, pathologyTestResults, (secondaryId != null ? secondaryId.ID : string.Empty), null, relatedDocumentStatus, patientIndigenousStatus, episode);

                // Create a new DocumentUploadBeforeQueue object using the current user, primaryId as the patient identifier, AdmissionDateTime from the CdaHeaderMetaData, CdaDosumentDetails and the DocumentFormatCode.
                DocumentUploadBeforeQueue documentUploadBeforeQueue = new DocumentUploadBeforeQueue(this.operation.User, patientMrn, cdaHeaderMetadata.AdmissionDateTime, cdaDetails.CdaDocumentDetails.CdaDocumentXmlBytes, ObjectMapper.Map<PcehrSchemas.Attachment[]>(cdaDetails.CdaDocumentDetails.Attachments), HIPS.Configuration.Settings.Instance.PathologyReportDocumentFormatCode, this.operation.PendingItem);
                documentUploadBeforeQueue.SetHL7MessageLogId(this.operation.HL7MessageLogId);

                // Call UploadOrSupersedeDocument on the new DocumentUploadBeforeQueue. This will add the document to the queue for uploading to the PCEHR.
                using (new TransactionScope(TransactionScopeOption.Suppress))
                {
                    HipsResponse hipsResponse = documentUploadBeforeQueue.UploadOrSupersedeDocument();
                    if (hipsResponse.Status != HipsResponseIndicator.OK)
                    {
                        throw new HipsResponseException(hipsResponse);
                    }
                }
            }
            catch (IhiInfoException ex)
            {
                // The HL7 message was processed, but a known data quality issue was found while attempting to search for the IHI.

                // Write INFO message because Application Support do not need to know.
                string description = string.Format(ConstantsResource.InfoMessageIhiLookupFail, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_107);
                UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
            }
            catch (IhiErrorException ex)
            {
                // The HL7 message was processed but a serious failure occurred when attempting to search for the IHI.

                // Write ERROR message because Application Support need to know.
                string description = string.Format(ConstantsResource.ErrorMessageIhiLookupFail, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_108);
                UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
            }
            catch (HpiiErrorException ex)
            {
                // A Hips Response Error - the message should not be processed due to a known data quality issue
                string description = string.Format(ConstantsResource.HipsExceptionMessage, ex.Message, ex.Message);
                handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_189);
                UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
            }
            catch (HL7MessageInfoException ex)
            {
                // This message was not processed because it contains a known data quality issue.

                // Write INFO message because Application Support do not need to know.
                string description = string.Format(ConstantsResource.InfoMessageNotProcessed, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_109);
                UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
            }
            catch (HL7MessageErrorException ex)
            {
                // HIPS failed while processing this HL7 message. The message should be retried.

                // Write ERROR message because Application Support need to know.
                string description = string.Format(ConstantsResource.ErrorMessageHipsFailed, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_110);
                UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
            }
            catch (HipsResponseException ex)
            {
                string description = string.Format(ConstantsResource.HipsExceptionMessage, ex.Message, ex.Detail);
                handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_109);
                UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Detail));
            }
            catch (InvalidOperationException ex)
            {
                // this is a re-try so just throw the exeption
                throw ex;
            }
            catch (Exception nex)
            {
                // An unexpected exception occurred while HIPS was processing this HL7 message. The message should be retried.

                // Write ERROR message because Application Support need to know.
                string description = string.Format(ConstantsResource.ErrorMessageHipsFailed, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                handler.UpdateEventLog(description, nex, operation.User, LogMessage.HIPS_MESSAGE_111);
                UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, nex.Message));
            }


        }

        /// <summary>
        /// Validates the HL7 Pathology Message, creates the CDA document and calls the MSMQ UploadOrSupersede document method to
        /// upload the document to the PCEHR.
        /// </summary>
        /// <exception cref="System.InvalidOperationException">When the MSMQ operation should be retried</exception>
        public void UploadImaging()
        {
            using (HIPSTraceWriter trace = new HIPSTraceWriter())
            {
                trace.WriteTrace(string.Format("IN HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}", operation.HL7Message), System.Diagnostics.TraceEventType.Information);
                
                DocumentQueueTransactionHandler handler = new DocumentQueueTransactionHandler(operation.User);
                // Restore the report and the pending item to the QueuedHL7ReportOperation from the data store.
                if (!handler.RestorePackageInQueuedHL7ReportOperation(operation))
                {
                    trace.WriteTrace(string.Format("OUT HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}: Package could not be restored", operation.HL7Message), System.Diagnostics.TraceEventType.Information);
                    return;
                }
                if (operation.PendingItem.QueueStatusId != (int)QueueStatus.Pending)
                {
                    string message = string.Format(ResponseStrings.HL7ReportOperationAbortedDetail,
                        operation.PendingItem.PcehrMessageQueueId,
                        operation.PendingItem.QueueOperationName,
                        operation.HL7MessageLogId);
                    handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationAborted, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_086);
                    if (string.IsNullOrEmpty(operation.PendingItem.Details))
                    {
                        operation.PendingItem.Details = message;
                    }
                    handler.PlaceAcknowledgementOnQueue(operation.PendingItem, operation.HL7MessageLogId);
                    trace.WriteTrace(string.Format("OUT HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}: PcehrMessageQueue item was not in Pending state", operation.HL7Message), System.Diagnostics.TraceEventType.Information);
                    return;
                }
                
                // Invoke the PopulateHL7MessageLog to retrieve the message log entity and HL7 Message details.
                HL7MessageLog pathMessageLog;
                handler.GetMessageLog(operation.HL7MessageLogId, out pathMessageLog);
                HIPS.HL7.Common.HL7Message pathMessage = HL7Message.Parse(this.operation.HL7Message);

                HL7GenericPathMessage genericPathMessage = pathMessage as HL7GenericPathMessage;
                HIPS.HL7.Common.Segment.PID pid = genericPathMessage.PatientIdentification;

                HL7GenericPasMessage genericPasMessage = new HL7GenericPasMessage();
                genericPasMessage.PatientIdentification = genericPathMessage.PatientIdentification;

                // Validate the PID segment - return an error if it is not valid
                string pidValidationError = ValidatePatientIdentifiers(pid);
                if (!string.IsNullOrEmpty(pidValidationError))
                {
                    handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationAborted, new Exception(pidValidationError), operation.User, LogMessage.HIPS_MESSAGE_086);
                    UpdatePendingItemForFailure(pidValidationError);
                    trace.WriteTrace(string.Format("OUT HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}: Patient Identifiers Validation failed", operation.HL7Message), System.Diagnostics.TraceEventType.Information);
                    return;
                }

                try
                {
                    // Process the HL7 Message
                    Mrn patientMrn;
                    CX secondaryId;
                    OrderDetails orderDetails = new OrderDetails();
                    CdaHeaderMetadata cdaHeaderMetadata = new CdaHeaderMetadata();
                    string relatedDocumentStatus;
                    int hospitalID;
                    List<PathologyTestResult> pathologyTestResults = null;
                    List<ImagingExamResult> imagingExamResults = new List<ImagingExamResult>();
                    Episode episode;
                    new PDILoader(operation.User).Process(DocumentTypeCodes.DiagnosticImagingReport, genericPathMessage, out secondaryId, out patientMrn, out hospitalID, out cdaHeaderMetadata, out orderDetails, out imagingExamResults, out pathologyTestResults, out relatedDocumentStatus, out episode);

                    // Update the Message Log with the details
                    pathMessageLog.HospitalID = hospitalID;
                    pathMessageLog.HospitalPatientIdentifier = patientMrn.Value;
                    handler.UpdateMessageLog(pathMessageLog);

                    // Now check if there is an earlier message sitting on the queue waiting to be processed for that patient
                    // If yes - throw an exception but don't fail item so we can retry
                    if (handler.CheckForEarlierPendingMessage(pathMessageLog.HospitalPatientIdentifier, pathMessageLog.HospitalID.Value, (Settings.Instance.UseHL7MessageDateTime && pathMessageLog.DateTimeOfMessage.HasValue ? pathMessageLog.DateTimeOfMessage.Value : pathMessageLog.DateCreated)))
                    {
                        string message = string.Format(ResponseStrings.OperationOutOfOrderUploadHL7,
                            operation.PendingItem.PcehrMessageQueueId,
                            operation.PendingItem.QueueOperationName,
                            pathMessageLog.MessageControlId,
                            pathMessageLog.HospitalPatientIdentifier,
                            pathMessageLog.HospitalID);
                        handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationRetriedDueToMessagesOutofOrder, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_136);
                        throw new InvalidOperationException();
                    }

                    // Update oending item episode
                    if (episode.EpisodeId.HasValue)
                        operation.PendingItem.EpisodeId = episode.EpisodeId.Value;

                    string patientIndigenousStatus = pid.Race.FirstOrDefault().identifier.ToString();

                    // Invoke the CreateImagingExamReport passing in the report, the CdaHeaderMetaData, primaryID as the patientIdentifier, the OrderDetails, the list of pathologyTestResult’s, the secondaryID.Value as the requestingHospitalPatientMrn. This will return a CdaDocumentDetails object.
                    trace.WriteTrace(string.Format("Call HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging.CreateDiagnosticImagingReport:: hl7Message: {0}", operation.HL7Message), System.Diagnostics.TraceEventType.Information);
                    PDIUploadService pdiReport = new PDIUploadService(this.operation.User);
                    CdaPathDIDocumentDetails cdaDetails = pdiReport.CreateDiagnosticImagingReport(this.operation.Report, cdaHeaderMetadata, patientMrn, orderDetails, imagingExamResults, (secondaryId != null ? secondaryId.ID : string.Empty), null, relatedDocumentStatus, patientIndigenousStatus, episode);

                    // Create a new DocumentUploadBeforeQueue object using the current user, primaryId as the patient identifier, AdmissionDateTime from the CdaHeaderMetaData, CdaDosumentDetails and the DocumentFormatCode.
                    DocumentUploadBeforeQueue documentUploadBeforeQueue = new DocumentUploadBeforeQueue(this.operation.User, patientMrn, cdaHeaderMetadata.AdmissionDateTime, cdaDetails.CdaDocumentDetails.CdaDocumentXmlBytes, ObjectMapper.Map<PcehrSchemas.Attachment[]>(cdaDetails.CdaDocumentDetails.Attachments), HIPS.Configuration.Settings.Instance.DiagnosticImagingReportDocumentFormatCode, this.operation.PendingItem);
                    documentUploadBeforeQueue.SetHL7MessageLogId(this.operation.HL7MessageLogId);

                    // Call UploadOrSupersedeDocument on the new DocumentUploadBeforeQueue. This will add the document to the queue for uploading to the PCEHR.
                    trace.WriteTrace(string.Format("Call HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging.UploadOrSupersedeDocument:: hl7Message: {0}", operation.HL7Message), System.Diagnostics.TraceEventType.Information);
                    using (new TransactionScope(TransactionScopeOption.Suppress))
                    {
                        HipsResponse hipsResponse = documentUploadBeforeQueue.UploadOrSupersedeDocument();
                        if (hipsResponse.Status != HipsResponseIndicator.OK)
                        {
                            throw new HipsResponseException(hipsResponse);
                        }
                    }
                }
                catch (IhiInfoException ex)
                {
                    // The HL7 message was processed, but a known data quality issue was found while attempting to search for the IHI.
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. IhiInfoException Details: {1}", operation.HL7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    // Write INFO message because Application Support do not need to know.
                    string description = string.Format(ConstantsResource.InfoMessageIhiLookupFail, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                    handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_107);
                    UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));

                }
                catch (IhiErrorException ex)
                {
                    // The HL7 message was processed but a serious failure occurred when attempting to search for the IHI.
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. IhiErrorException Details: {1}", operation.HL7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    // Write ERROR message because Application Support need to know.
                    string description = string.Format(ConstantsResource.ErrorMessageIhiLookupFail, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                    handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_108);
                    UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
                }
                catch (HpiiErrorException ex)
                {
                    // A Hips Response Error - the message should not be processed due to a known data quality issue
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. HipsResponseException Details: {1}", operation.HL7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    string description = string.Format(ConstantsResource.HipsExceptionMessage, ex.Message, ex.Message);
                    handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_189);
                    UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
                }
                catch (HL7MessageInfoException ex)
                {
                    // This message was not processed because it contains a known data quality issue.
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. HL7MessageInfoException Details: {1}", operation.HL7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    // Write INFO message because Application Support do not need to know.
                    string description = string.Format(ConstantsResource.InfoMessageNotProcessed, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                    handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_109);
                    UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
                }
                catch (HL7MessageErrorException ex)
                {
                    // HIPS failed while processing this HL7 message. The message should be retried.
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. HL7MessageErrorException Details: {1}", operation.HL7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    // Write ERROR message because Application Support need to know.
                    string description = string.Format(ConstantsResource.ErrorMessageHipsFailed, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                    handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_110);
                    UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Message));
                }
                catch (HipsResponseException ex)
                {
                    // A Hips Response Error - the message should not be processed due to a known data quality issue
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. HipsResponseException Details: {1}", operation.HL7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    string description = string.Format(ConstantsResource.HipsExceptionMessage, ex.Message, ex.Detail);
                    handler.UpdateEventLog(description, ex, operation.User, LogMessage.HIPS_MESSAGE_109);
                    UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, ex.Detail));
                }
                catch (InvalidOperationException ex)
                {
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. InvalidOperationException Details: {1}", operation.HL7Message, ex.ToString()), System.Diagnostics.TraceEventType.Error);
                    // this is a re-try so just throw the exeption
                    throw ex;
                }
                catch (Exception nex)
                {
                    // An unexpected exception occurred while HIPS was processing this HL7 message. The message should be retried.
                    trace.WriteTrace(string.Format("ERROR HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}. Exception Details: {1}", operation.HL7Message, nex.ToString()), System.Diagnostics.TraceEventType.Error);
                    
                    // Write ERROR message because Application Support need to know.
                    string description = string.Format(ConstantsResource.ErrorMessageHipsFailed, pathMessageLog.MessageControlId, pathMessageLog.SendingApplication, pathMessageLog.SendingFacility);
                    handler.UpdateEventLog(description, nex, operation.User, LogMessage.HIPS_MESSAGE_111);
                    UpdatePendingItemForFailure(string.Format("{0}. {1}.", description, nex.Message));
                }
                trace.WriteTrace(string.Format("OUT HIPS.CommonBuisnessLogic.Pcehr.HL7ReportUpload.UploadImaging:: hl7Message: {0}", operation.HL7Message), System.Diagnostics.TraceEventType.Information);

            }
        }




        #endregion methods

        #region Private methods

        /// <summary>
        /// Validates the HL7 details in the PID segment
        /// </summary>
        /// <param name="pid">HL7 PID segment</param>
        /// <returns>Empty String if no errors, otherwise error message to log</returns>
        private string ValidatePatientIdentifiers(HIPS.HL7.Common.Segment.PID pid)
        {
            string errorMessage = string.Empty;
            // Patient Identifier Indigenous Status must have a value 
            if (pid.Race == null || String.IsNullOrEmpty(pid.Race.FirstOrDefault().identifier))
            {
                errorMessage = string.Format(ResponseStrings.PatientIndigenousStatusNotSet,
                    operation.PendingItem.PcehrMessageQueueId,
                    operation.PendingItem.QueueOperationName);
                return errorMessage;
            }

            return errorMessage;
        }


        /// <summary>
        /// Updates the HL7MessageLog Status to Failure and PendingItem Status to Failure and adds the error details in
        /// </summary>
        /// <param name="pathMessageLog">HL7MessageLog</param>
        /// <param name="description">The error description</param>
        private void UpdatePendingItemForFailure(string description)
        {
            DocumentQueueTransactionHandler handler = new DocumentQueueTransactionHandler(operation.User);
            try
            {
                // Update the Queue operation
                if (string.IsNullOrEmpty(operation.PendingItem.Details))
                {
                    operation.PendingItem.Details = description;
                }
                operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                handler.UpdateOperationForFailure(operation.PendingItem);
                // Operation aborted and place the error onto the Ack Queue
                string message = string.Format(ResponseStrings.HL7ReportOperationAbortedDetail,
                   operation.PendingItem.PcehrMessageQueueId,
                   operation.PendingItem.QueueOperationName,
                   operation.HL7MessageLogId);
                handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationAborted, new Exception(description), operation.User, LogMessage.HIPS_MESSAGE_086);
                operation.PendingItem.Details = string.Format("{0}. {1}", message, description);
                handler.PlaceAcknowledgementOnQueue(operation.PendingItem, operation.HL7MessageLogId);
            }
            catch (Exception ex)
            {
                string message = string.Format(ResponseStrings.HL7ReportOperationAbortedDetail,
                   operation.PendingItem.PcehrMessageQueueId,
                   operation.PendingItem.QueueOperationName,
                   operation.HL7MessageLogId);
                handler.UpdateEventLog(ResponseStrings.InfoQueuedOperationAborted, ex, operation.User, LogMessage.HIPS_MESSAGE_086);
                if (string.IsNullOrEmpty(operation.PendingItem.Details))
                {
                    operation.PendingItem.Details = message;
                }

                handler.PlaceAcknowledgementOnQueue(operation.PendingItem, operation.HL7MessageLogId);
            }

        }
        #endregion Private methods

    }
}
