﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.CommonSchemas.Exceptions;
using HIPS.Configuration;
using HIPS.HL7.Common;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.Common;

namespace HIPS.CommonBusinessLogic.Utility
{
    /// <summary>
    /// Contains extensions to assist working with CDA documents.
    /// </summary>
    public static class CdaDocumentUtility
    {
        #region Constants

        /// <summary>
        /// CDA Rendering Specification makes it a requirement to validate the logo
        /// size does not exceed 400 pixels in width or 100 pixels in height.
        /// </summary>
        private const int PCEHR_LOGO_WIDTH = 400;

        private const int PCEHR_LOGO_HEIGHT = 100;

        private const string OID_HI_REGEX_IHI = @"^(?<prefix>1.2.36.1.2001.1003.0.)(?<value>800360(\d){10})$";
        private const string OID_HI_REGEX_HPII = @"^(?<prefix>1.2.36.1.2001.1003.0.)(?<value>800361(\d){10})$";
        private const string OID_HI_REGEX_HPIO = @"^(?<prefix>1.2.36.1.2001.1003.0.)(?<value>800362(\d){10})$";

        #endregion Constants

        #region Private Fields

        /// <summary>
        /// Namespaces used in CDA documents. Each X-Path evaluation needs to reference this set of namespaces.
        /// </summary>
        private static XmlNamespaceManager cdaNamespaces;

        #endregion Private Fields

        /// <summary>
        /// Initialises the <see cref="CdaDocumentUtility"/> class. Sets the CDA namespaces.
        /// </summary>
        static CdaDocumentUtility()
        {
            cdaNamespaces = new XmlNamespaceManager(new NameTable());
            cdaNamespaces.AddNamespace(XmlStringMap.HL7V3NamespaceAbbreviation, XmlStringMap.HL7V3Namespace);
            cdaNamespaces.AddNamespace(XmlStringMap.AustralianCdaExtensionsV3NamespaceAbbreviation, XmlStringMap.AustralianCdaExtensionsV3Namespace);
        }

        /// <summary>
        /// Loads the XML document from the Content property into the Document property.
        /// </summary>
        /// <param name="document">CDA document object.</param>
        internal static void Load(this CdaDocument document)
        {
            document.Document = new XmlDocument();
            using (MemoryStream ms = new MemoryStream(document.Content))
            {
                document.Document.Load(ms);
            }
        }

        /// <summary>
        /// Adds the OID prefix to a healthcare identifier.
        /// </summary>
        /// <param name="healthcareIdentifier">Australian Healthcare Identifier (16 digits)</param>
        /// <returns>Healthcare identifier with OID prefix</returns>
        /// <exception cref="System.ArgumentException">Healthcare identifier has invalid format.</exception>
        internal static string AddOidPrefix(this string healthcareIdentifier)
        {
            if (!Regex.IsMatch(healthcareIdentifier, "^80036[012][0-9]{10}$"))
            {
                throw new ArgumentException(string.Format("Healthcare identifier '{0}' has invalid format.", healthcareIdentifier));
            }
            return string.Format("{0}{1}", XmlStringMap.NationalHealthcareIdentifierOidPrefix, healthcareIdentifier);
        }

        /// <summary>
        /// Removes the OID prefix from a healthcare identifier.
        /// </summary>
        /// <param name="healthcareIdentifierOid">Healthcare identifier with OID prefix.</param>
        /// <returns>Australian Healthcare Identifier (16 digits).</returns>
        /// <exception cref="System.ArgumentException">Healthcare identifier OID has invalid format.</exception>
        /// <exception cref="System.ArgumentException">Healthcare identifier has invalid format.</exception>
        internal static string RemoveOidPrefix(this string healthcareIdentifierOid)
        {
            if (!healthcareIdentifierOid.StartsWith(XmlStringMap.NationalHealthcareIdentifierOidPrefix))
            {
                throw new ArgumentException(string.Format("Healthcare identifier OID '{0}' has invalid format.", healthcareIdentifierOid));
            }
            string healthcareIdentifier = healthcareIdentifierOid.Substring(XmlStringMap.NationalHealthcareIdentifierOidPrefix.Length);
            if (!Regex.IsMatch(healthcareIdentifier, "^80036[012][0-9]{10}$"))
            {
                throw new ArgumentException(string.Format("Healthcare identifier '{0}' has invalid format.", healthcareIdentifier));
            }
            return healthcareIdentifier;
        }

        /// <summary>
        /// Gets the document instance identifier from a CDA document
        /// (this one is unique for every version of every document).
        /// If there is no extension then the root value is returned.
        /// If there is an extension then it is appended to the root
        /// with a caret '^' in the middle.
        /// </summary>
        /// <param name="cdaDocument">The CDA document.</param>
        /// <returns>Document ID</returns>
        internal static string GetDocumentId(this CdaDocument document)
        {
            XmlNode idNode = document.Document.SelectSingleNode(XmlStringMap.DocumentIdXPath, cdaNamespaces);
            string root = idNode.Attributes["root"].Value;
            if (idNode.Attributes["extension"] == null)
            {
                return root;
            }
            else
            {
                string extension = idNode.Attributes["extension"].Value;
                return string.Format("{0}^{1}", root, extension);
            }
        }

        /// <summary>
        /// Gets the document type code from the document.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Document type code.</returns>
        internal static CdaCode GetDocumentTypeCode(this CdaDocument document)
        {
            string code = document.Document.AttributeValueOrEmpty(XmlStringMap.DocumentTypeCodeXPath, "code");
            string displayName = document.Document.AttributeValueOrEmpty(XmlStringMap.DocumentTypeCodeXPath, "displayName");
            string codeSystemName = document.Document.AttributeValueOrEmpty(XmlStringMap.DocumentTypeCodeXPath, "codeSystemName");
            return new CdaCode
            {
                Code = code,
                DisplayName = displayName,
                CodeSystemName = codeSystemName
            };
        }

        /// <summary>
        /// If the document references an organisational logo, checks that the
        /// logo meets the PCEHR requirements (no more than 400 by 100 pixels)
        /// and adds the integrity check attributes to the logo reference. The
        /// logo can either be supplied as an attachment or pre-configured in
        /// the Logo column of the Hospital table.
        ///
        /// If using a pre-configured logo, then this method will add it to the
        /// list of attachments.
        /// </summary>
        /// <param name="metadata">CDA document packaging data.</param>
        /// <exception cref="CdaPackagingException">Thrown when logo not found or too large.</exception>
        internal static void ProcessLogo(CdaMetadata metadata)
        {
            string logoFileName = metadata.Document.GetLogoFileName();
            if (logoFileName != null)
            {
                byte[] logoBytes;
                HIPS.PcehrSchemas.Attachment logoAttachment = metadata.Document.Attachments.FirstOrDefault(a => a.FileName == logoFileName);
                if (logoAttachment != null)
                {
                    logoBytes = logoAttachment.Contents;
                }
                else if (metadata.Hospital != null && metadata.Hospital.Logo != null)
                {
                    logoBytes = metadata.Hospital.Logo;
                    metadata.Document.Attachments.Add(new HIPS.PcehrSchemas.Attachment() { Contents = logoBytes, FileName = logoFileName });
                }
                else
                {
                    string message = string.Format(ResponseStrings.LogoNotSupplied, logoFileName);
                    throw new CdaPackagingException(message);
                }
                metadata.Document.SetLogoIntegrityCheck(logoBytes);
                using (MemoryStream ms = new MemoryStream(logoBytes))
                {
                    Image image = Bitmap.FromStream(ms);
                    if (image.Height > PCEHR_LOGO_HEIGHT || image.Width > PCEHR_LOGO_WIDTH)
                    {
                        string sendingOrganisationName = metadata.Hospital != null
                            ? (metadata.Hospital.Description ?? metadata.Hospital.Name)
                            : metadata.Recipient.Organisation.Name;
                        string message = string.Format(ResponseStrings.LogoTooLarge, sendingOrganisationName, image.Width, image.Height);
                        throw new CdaPackagingException(message);
                    }
                }
            }
        }

        /// <summary>
        /// Gets the referenced file name for the organisational logo in the document.
        /// </summary>
        /// <returns>The referenced file name for the organisational logo, or null if there is no logo.</returns>
        private static string GetLogoFileName(this CdaDocument document)
        {
            XmlNode logoReferenceElement = document.Document.SelectSingleNode(XmlStringMap.LogoReferenceXPath, cdaNamespaces);
            if (logoReferenceElement != null)
            {
                XmlAttribute referenceValueAttribute = logoReferenceElement.Attributes[XmlStringMap.ValueAttributeName];
                if (referenceValueAttribute != null)
                {
                    return referenceValueAttribute.Value;
                }
            }
            return null;
        }

        /// <summary>
        /// Adds the integrity check attributes to the logo reference in the CDA document.
        /// Stores the modified document in the Content property, encoded as UTF-8
        /// without a Byte Order Mark (BOM) - this is imperative for PCEHR.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <param name="logoFile">Logo file content.</param>
        private static void SetLogoIntegrityCheck(this CdaDocument document, byte[] logoFile)
        {
            XmlNode logoValueElement = document.Document.SelectSingleNode(XmlStringMap.LogoValueXPath, cdaNamespaces);
            if (logoValueElement != null)
            {
                XmlAttribute integrityCheck = document.Document.CreateAttribute(XmlStringMap.IntegrityCheckAttributeName);
                integrityCheck.Value = Convert.ToBase64String(SHA1.Create().ComputeHash(logoFile));
                logoValueElement.Attributes.Append(integrityCheck);

                XmlAttribute integrityCheckAlgorithm = document.Document.CreateAttribute(XmlStringMap.IntegrityCheckAlgorithmAttributeName);
                integrityCheckAlgorithm.Value = XmlStringMap.IntegrityCheckAlgorithmAttributeValue;
                logoValueElement.Attributes.Append(integrityCheckAlgorithm);
            }
            document.Save();
        }

        /// <summary>
        /// Stores the modified document in the Content property, encoded as UTF-8
        /// without a Byte Order Mark (BOM) - this is imperative for PCEHR.
        /// </summary>
        /// <param name="document"></param>
        private static void Save(this CdaDocument document)
        {
            using (MemoryStream stream = new MemoryStream())
            {
                XmlWriterSettings settings = new XmlWriterSettings();
                settings.Encoding = new UTF8Encoding(encoderShouldEmitUTF8Identifier: false);
                using (XmlWriter writer = XmlWriter.Create(stream, settings))
                {
                    document.Document.Save(writer);
                }
                document.Content = stream.ToArray();
            }
        }

        /// <summary>
        /// Gets the patient's name and entity identifiers from the document.
        /// </summary>
        /// <param name="document">CDA Document.</param>
        /// <returns>Patient details.</returns>
        internal static HIPS.CommonSchemas.Participant GetPatient(this CdaDocument document)
        {
            return document.GetPerson(XmlStringMap.PatientXPath);
        }

        /// <summary>
        /// Gets the first recipient individual from the document.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Recipient individual</returns>
        internal static HIPS.CommonSchemas.Participant GetRecipientIndividual(this CdaDocument document)
        {
            return document.GetPerson(XmlStringMap.RecipientXPath);
        }

        /// <summary>
        /// Gets the author from the document.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Recipient individual</returns>
        internal static HIPS.CommonSchemas.Participant GetAuthor(this CdaDocument document)
        {
            return document.GetPerson(XmlStringMap.AuthorPersonXPath);
        }

        /// <summary>
        /// Gets a person, such as the patient, author or recipient, from the CDA document.
        /// The X-Path query should resolve to a Person element under which
        /// there exist "x:name" and "ext:asEntityIdentifier" elements.
        /// </summary>
        /// <param name="document">The CDA document.</param>
        /// <param name="xpath">The X-Path query for the Person element.</param>
        /// <returns>Person in the CDA document.</returns>
        private static HIPS.CommonSchemas.Participant GetPerson(this CdaDocument document, string xpath)
        {
            XmlNode personNode = document.Document.SelectSingleNode(xpath, cdaNamespaces);
            if (personNode == null)
            {
                return null;
            }
            HIPS.CommonSchemas.Participant person = new HIPS.CommonSchemas.Participant();
            person.FamilyName = personNode.InnerTextOrEmpty(XmlStringMap.NameFamilyXPath);
            person.GivenNames = personNode.SelectInnerText(XmlStringMap.NameGivenXPath);
            person.Titles = personNode.SelectInnerText(XmlStringMap.NamePrefixXPath);
            person.Suffixes = personNode.SelectInnerText(XmlStringMap.NameSuffixXPath);
            XmlNodeList eiNodes = personNode.SelectNodes(XmlStringMap.EntityIdentifierXPath, cdaNamespaces);
            foreach (XmlNode eiNode in eiNodes)
            {
                HIPS.CommonSchemas.Cda.EntityIdentifier id = eiNode.ToEntityIdentifier();
                person.Identifiers.Add(id);
            }
            return person;
        }

        /// <summary>
        /// Gets a participating consumer representation of the patient.
        /// </summary>
        /// <param name="document">The CDA document.</param>
        /// <returns>The participating consumer representation of the patient.</returns>
        internal static HIPS.CommonSchemas.Cda.ParticipatingIndividual.ParticipatingConsumer GetPatientConsumer(this CdaDocument document)
        {
            var patientParticipant = document.GetPatient();
            var patientSex = document.GetPatientSex();
            var patientDateOfBirth = document.GetPatientDateOfBirth();
            if (patientParticipant == null)
            {
                return null;
            }
            var result = new HIPS.CommonSchemas.Cda.ParticipatingIndividual.ParticipatingConsumer();
            result.DateOfBirth = patientDateOfBirth;
            result.FamilyName = patientParticipant.FamilyName;
            result.GivenNames = string.Join(" ", patientParticipant.GivenNames);
            if (patientParticipant.Identifiers.Any(i => RegExUtility.IsMatch(i.Root, OID_HI_REGEX_IHI)))
            {
                result.Ihi = patientParticipant.Identifiers.Where(i => RegExUtility.IsMatch(i.Root, OID_HI_REGEX_IHI)).Select(i => RegExUtility.ExtractRegExGroupValue(i.Root, OID_HI_REGEX_IHI, "value")).FirstOrDefault();
            }
            if (patientParticipant.Identifiers.Any(i => i.Code.Code == "MR"))
            {
                result.LocalIdentifier = patientParticipant.Identifiers.Where(i => i.Code.Code == "MR").Select(i => i.Extension).FirstOrDefault();
            }
            result.Sex = patientSex;
            result.Suffix = patientParticipant.Suffixes.FirstOrDefault();
            result.Title = patientParticipant.Titles.FirstOrDefault();

            return result;
        }

        /// <summary>
        /// Gets a participating provider representation of the author.
        /// </summary>
        /// <param name="document">The CDA document.</param>
        /// <returns>The participating provider representation of the author.</returns>
        internal static HIPS.CommonSchemas.Cda.ParticipatingIndividual.ParticipatingProvider GetAuthorProvider(this CdaDocument document)
        {
            var authorParticipant = document.GetAuthor();
            if (authorParticipant == null)
            {
                return null;
            }
            var result = new HIPS.CommonSchemas.Cda.ParticipatingIndividual.ParticipatingProvider();

            result.EmployerName = document.GetAuthorEmployerName();
            var employerIdentifiers = document.GetAuthorEmployerIdentifiers();
            if (employerIdentifiers.Any(i => RegExUtility.IsMatch(i.Root, OID_HI_REGEX_HPIO)))
            {
                result.EmployerHpio = employerIdentifiers.Where(i => RegExUtility.IsMatch(i.Root, OID_HI_REGEX_HPIO)).Select(i => RegExUtility.ExtractRegExGroupValue(i.Root, OID_HI_REGEX_HPIO, "value")).FirstOrDefault();
            }
            result.FamilyName = authorParticipant.FamilyName;
            result.GivenNames = string.Join(" ", authorParticipant.GivenNames);
            if (authorParticipant.Identifiers.Any(i => RegExUtility.IsMatch(i.Root, OID_HI_REGEX_HPII)))
            {
                result.Hpii = authorParticipant.Identifiers.Where(i => RegExUtility.IsMatch(i.Root, OID_HI_REGEX_HPII)).Select(i => RegExUtility.ExtractRegExGroupValue(i.Root, OID_HI_REGEX_HPII, "value")).FirstOrDefault();
            }
            result.LocalIdentifier = string.Empty; // Not currently populated.
            result.Suffix = authorParticipant.Suffixes.FirstOrDefault();
            result.Title = authorParticipant.Titles.FirstOrDefault();

            return result;
        }

        /// <summary>
        /// Gets the author's employer name, if available.
        /// </summary>
        /// <param name="document">The CDA document.</param>
        /// <returns>The author's employer name.</returns>
        internal static string GetAuthorEmployerName(this CdaDocument document)
        {
            XmlNode wholeOrganisationNode = document.Document.SelectSingleNode(XmlStringMap.AuthorOrganisationXPath, cdaNamespaces);
            if (wholeOrganisationNode == null)
            {
                return string.Empty;
            }
            return wholeOrganisationNode.InnerTextOrEmpty(XmlStringMap.OrganisationNameXPath);
        }

        /// <summary>
        /// Get's the author's employer organisation identifiers, if available.
        /// </summary>
        /// <param name="document">The CDA document.</param>
        /// <returns>The author's employer organisation identifiers.</returns>
        internal static List<HIPS.CommonSchemas.Cda.EntityIdentifier> GetAuthorEmployerIdentifiers(this CdaDocument document)
        {
            var result = new List<HIPS.CommonSchemas.Cda.EntityIdentifier>();
            XmlNode wholeOrganisationNode = document.Document.SelectSingleNode(XmlStringMap.AuthorOrganisationXPath, cdaNamespaces);
            if (wholeOrganisationNode != null)
            {
                XmlNodeList eiNodes = wholeOrganisationNode.SelectNodes(XmlStringMap.EntityIdentifierXPath, cdaNamespaces);
                foreach (XmlNode eiNode in eiNodes)
                {
                    result.Add(eiNode.ToEntityIdentifier());
                }
            }
            return result;
        }

        /// <summary>
        /// Gets the document creation time from the document.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Document creation time.</returns>
        internal static DateTime GetCreationTime(this CdaDocument document)
        {
            string timestamp = document.Document.AttributeValueOrEmpty(XmlStringMap.DocumentCreationTimeXPath, "value");
            return HL7DateTime.Parse(timestamp).Value;
        }

        /// <summary>
        /// Gets the patient date of birth from the document.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Patient date of birth.</returns>
        internal static DateTime GetPatientDateOfBirth(this CdaDocument document)
        {
            string timestamp = document.Document.AttributeValueOrEmpty(XmlStringMap.PatientDateOfBirthXPath, "value");
            return HL7DateTime.Parse(timestamp).Value;
        }

        /// <summary>
        /// Gets the patient sex from the document.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Patient sex.</returns>
        internal static SexEnumerator GetPatientSex(this CdaDocument document)
        {
            string sexCode = document.Document.AttributeValueOrEmpty(XmlStringMap.PatientSexXPath, "code");
            switch (sexCode)
            {
                case "M": return SexEnumerator.Male;
                case "F": return SexEnumerator.Female;
                case "I": return SexEnumerator.IntersexOrIndeterminate;
                case "N": return SexEnumerator.NotStatedOrInadequatelyDescribed;
                default: return SexEnumerator.NotStatedOrInadequatelyDescribed;
            }
        }

        /// <summary>
        /// Gets an entity identifier from an XML node.
        /// </summary>
        /// <param name="eiNode"></param>
        /// <returns></returns>
        private static HIPS.CommonSchemas.Cda.EntityIdentifier ToEntityIdentifier(this XmlNode eiNode)
        {
            HIPS.CommonSchemas.Cda.EntityIdentifier id = new HIPS.CommonSchemas.Cda.EntityIdentifier();
            id.Root = eiNode.AttributeValueOrEmpty(XmlStringMap.ExtIdXPath, XmlStringMap.RootAttributeName);
            id.Extension = eiNode.AttributeValueOrEmpty(XmlStringMap.ExtIdXPath, XmlStringMap.ExtensionAttributeName);
            id.AssigningAuthorityName = eiNode.AttributeValueOrEmpty(XmlStringMap.ExtIdXPath, XmlStringMap.AssigningAuthorityNameAttributeName);
            id.Code = new CdaCode();
            id.Code.Code = eiNode.AttributeValueOrEmpty(XmlStringMap.ExtCodeXPath, XmlStringMap.CodeAttributeName);
            id.Code.CodeSystem = eiNode.AttributeValueOrEmpty(XmlStringMap.ExtCodeXPath, XmlStringMap.CodeSystemAttributeName);
            id.Code.CodeSystemName = eiNode.AttributeValueOrEmpty(XmlStringMap.ExtCodeXPath, XmlStringMap.CodeSystemNameAttributeName);
            id.Code.DisplayName = eiNode.AttributeValueOrEmpty(XmlStringMap.ExtCodeXPath, XmlStringMap.DisplayNameAttributeName);
            if (id.AssigningAuthorityName == "IHI")
            {
                id.QualifiedIdentifier = new Uri(HIQualifiers.IHIQualifier + id.Root.RemoveOidPrefix());
            }
            else if (id.AssigningAuthorityName == "HPI-I")
            {
                id.QualifiedIdentifier = new Uri(HIQualifiers.HPIIQualifier + id.Root.RemoveOidPrefix());
            }
            else if (id.AssigningAuthorityName == "HPI-O")
            {
                id.QualifiedIdentifier = new Uri(HIQualifiers.HPIOQualifier + id.Root.RemoveOidPrefix());
            }
            else
            {
                // Assume the extension contains a local ID of the person within the organisation,
                // and should be qualified using the domain namespace of the organisation where
                // the ID came from.
                // See FAQ_Implementation_clarification_for_relaxation_of_HPII_in_DS_rev001.pdf
                // For example http://ns.health.domain.au/id/cda/userid/1.0/sbiber01
                string format = Settings.Instance.CdaUserIdQualifierFormat;
                id.QualifiedIdentifier = new Uri(string.Format(format, id.Extension));
            }
            return id;
        }

        /// <summary>
        /// Gets the patient address from the document.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Patient address.</returns>
        internal static Address GetPatientAddress(this CdaDocument document)
        {
            XmlNode node = document.Document.SelectSingleNode(XmlStringMap.PatientAddressXPath, cdaNamespaces);
            var address = new Address
            {                
                AddressLine1 = node.SelectInnerText(XmlStringMap.AddressStreetAddressLineXPath).FirstOrDefault(),
                AddressLine2 = node.SelectInnerText(XmlStringMap.AddressStreetAddressLineXPath).Skip(1).FirstOrDefault(),
                PlaceName = node.InnerTextOrEmpty(XmlStringMap.AddressCityXPath),                
                Postcode = node.InnerTextOrEmpty(XmlStringMap.AddressPostcodeXPath),
                CountryName = node.InnerTextOrEmpty(XmlStringMap.AddressCountryXPath)
            };
            string state = node.InnerTextOrEmpty(XmlStringMap.AddressStateXPath);
            AustralianState australianState;
            if (Enum.TryParse<AustralianState>(state, out australianState))
            {
                address.AustralianState = australianState;
            }
            else
            {
                address.InternationalStateCode = state;
            }
            return address;
        }

        /// <summary>
        /// Extracts the inner text of the XML node at the specified index within a list of nodes,
        /// or an empty string if there are not so many nodes in the list.
        /// </summary>
        /// <param name="node">XML node.</param>
        /// <param name="xpath">X-Path query.</param>
        /// <returns>Inner text of the matching element, or an empty string if there is no matching element.</returns>
        private static string InnerTextOrEmpty(this XmlNode node, string xpath, int index)
        {
            XmlNodeList list = node.SelectNodes(xpath, cdaNamespaces);

            if (list.Count > index)
            {
                return list[index].InnerText;
            }
            else
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Extracts the inner text of the XML element at the specified index within a list of elements,
        /// or an empty string if there are not so many elements in the list.
        /// </summary>
        /// <param name="element">XML element.</param>
        /// <param name="xpath">X-Path query.</param>
        /// <returns>List containing the inner text of each matching element.</returns>
        private static List<string> SelectInnerText(this XmlNode element, string xpath)
        {
            XmlNodeList nodeList = element.SelectNodes(xpath, cdaNamespaces);
            List<string> textList = new List<string>();
            foreach (XmlNode node in nodeList)
            {
                textList.Add(node.InnerText);
            }
            return textList;
        }

        /// <summary>
        /// Extracts the inner text of the XML element matching the specified X-Path query,
        /// or an empty string if there is no such element.
        /// </summary>
        /// <param name="node">XML node.</param>
        /// <param name="xpath">X-Path query.</param>
        /// <returns>Inner text of the matching element, or an empty string if there is no matching element.</returns>
        private static string InnerTextOrEmpty(this XmlNode node, string xpath)
        {
            XmlNode innerNode = node.SelectSingleNode(xpath, cdaNamespaces);

            if (innerNode != null)
            {
                return innerNode.InnerText;
            }
            else
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Extracts the value of an attribute from the XML element matching the specified X-Path query,
        /// or an empty string if there is no such element or no such attribute.
        /// </summary>
        /// <param name="node">XML node.</param>
        /// <param name="xpath">X-Path query.</param>
        /// <param name="attribute">Attribute name.</param>
        /// <returns>Value of the matching attribute, or an empty string if there is no matching attribute.</returns>
        private static string AttributeValueOrEmpty(this XmlNode node, string xpath, string attribute)
        {
            XmlNode innerNode = node.SelectSingleNode(xpath, cdaNamespaces);
            if (innerNode != null)
            {
                XmlAttribute attr = innerNode.Attributes[attribute];
                if (attr != null)
                {
                    return attr.Value;
                }
            }
            return string.Empty;
        }
    }
}
