﻿using System.Linq;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Cda;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using MDM.Model.HL7Model;

namespace HIPS.CommonBusinessLogic.Utility
{
    /// <summary>
    /// Contains extensions to assist working with NEHTA MDM model classes.
    /// </summary>
    public static class MdmModelUtility
    {
        /// <summary>
        /// Populates the HL7 Model based on the CDA document metadata.
        /// </summary>
        /// <param name="packageContent">The XDM-ZIP package content.</param>
        /// <param name="cdaDocument">The XML document for the CDA document instance.</param>
        /// <returns>Populated HL7 Model.</returns>
        internal static HL7Model ToMdmModel(this CdaMetadata metadata)
        {
            CdaDocument document = metadata.Document;

            // Create the message header (MSH)
            var messageHeader = new MSH
            {
                ReceivingApplication = new HD { NamespaceID = metadata.Recipient.Organisation.Name },
                ReceivingFacility = new HD
                {
                    NamespaceID = metadata.Recipient.Organisation.HpiO,
                    UniversalID = metadata.Recipient.Organisation.HpiO.AddOidPrefix(),
                    UniversalIDType = "ISO"
                },
                SendingApplication = new HD { NamespaceID = metadata.Sender.Organisation.Name },
                SendingFacility = new HD
                {
                    NamespaceID = metadata.Sender.Organisation.HpiO,
                    UniversalID = metadata.Sender.Organisation.HpiO.AddOidPrefix(),
                    UniversalIDType = "ISO"
                },
                MessageControlId = metadata.SmdInvocationIdentifier
            };

            // Create the event EVN
            var evn = new EVN
            {
                MessageDateTime = document.GetCreationTime()
            };

            // Create the patient information (PID)
            var patientInformation = new PID
            {
                AddressLines = document.GetPatientAddress().ToXAD(),
                BirthDateTime = document.GetPatientDateOfBirth(),
                Gender = document.GetPatientSex().ToGender(),
                PatientIdentifiers = document.GetPatient().Identifiers.Select(ei => ei.ToCX()).ToList(),
                PatientName = document.GetPatient().ToXPN()
            };


            // Create the patient visit information (PV1)
            var patientVisit = new PV1();

            // If supplied, prefer to use the recipient information from the SMD metadata.
            // Otherwise try to get recipient from the CDA document, but this may be the
            // first of many recipients and not the actual recipient of this message.
            if (metadata.Recipient.Individual != null)
            {
                patientVisit.ConsultingDoctor = metadata.Recipient.Individual.ToXCN();
            }
            else
            {
                patientVisit.ConsultingDoctor = document.GetRecipientIndividual().ToXCN();

                // If there is no recipient individual in the document, fall back to the author.
                if (patientVisit.ConsultingDoctor == null)
                {
                    patientVisit.ConsultingDoctor = document.GetAuthor().ToXCN();
                }
            }

            // Create the Transcription Document Header (TXA)
            var transcriptionDocumentHeader = new TXA
            {
                ActivityDateTime = document.GetCreationTime(),

                // This unique ID should be taken from the root element withn the CDA document
                // that is the payload for this message.
                // e.g. "8a58f026-b51a-4946-be44-ac770407448f" 
                UniqueDocumentNumber = document.GetDocumentId()
            };


            //Create the Observation / Result (OBX) without the package which will be added outside this method.
            var observation = document.GetDocumentTypeCode().ToOBX();

            //Populate the HL7Model object with the previously crated / populated objects
            var mdmModel = new HL7Model
            {
                EVN = evn,
                MSH = messageHeader,
                PID = patientInformation,
                PV1 = patientVisit,
                TXA = transcriptionDocumentHeader,
                OBX = observation
            };

            return mdmModel;
        }

        /// <summary>
        /// Converts a CDA code system name to HL7 v2.
        /// </summary>
        /// <param name="codeSystemName">CDA code system name.</param>
        /// <returns>HL7 v2 descriptor.</returns>
        private static string CodeSystemNameToHL7(string codeSystemName)
        {
            if (codeSystemName == "LOINC")
            {
                return "LN";
            }
            return codeSystemName;
        }

        /// <summary>
        /// Converts the document type code from the document to an OBX in the MDM model.
        /// The code system name is translated from LOINC to LN.
        /// </summary>
        /// <param name="documentTypeCode">CDA document type code.</param>
        /// <returns>Document type code as OBX.</returns>
        private static OBX ToOBX(this CdaCode documentTypeCode)
        {
            return new OBX
            {
                Code = documentTypeCode.Code,
                Description = documentTypeCode.DisplayName,
                Descriptor = CodeSystemNameToHL7(documentTypeCode.CodeSystemName)
            };
        }

        /// <summary>
        /// Converts a CDA document participant to an XCN in the MDM model. Checks for HPI-I first, 
        /// then Medicare Provider Number (OID 1.2.36.174030967.0.2) and lastly falls
        /// back to any local identifier that is present for the individual.
        /// </summary>
        /// <param name="document">CDA document.</param>
        /// <returns>Recipient individual</returns>
        private static XCN ToXCN(this Participant participant)
        {
            if (participant == null)
            {
                return null;
            }
            string value = string.Empty;
            string namespaceId = string.Empty;

            // Check for HPI-I first
            EntityIdentifier ei = participant.Identifiers.SingleOrDefault(i => i.AssigningAuthorityName == "HPI-I");
            if (ei != null)
            {
                value = ei.Root.RemoveOidPrefix();
                namespaceId = "AUSHIC";
            }
            else
            {
                // Check for Medicare Provider Number second
                ei = participant.Identifiers.FirstOrDefault(i => i.Root == "1.2.36.174030967.0.2");
                if (ei != null)
                {
                    value = ei.Extension;
                    namespaceId = "AUSHICPR";
                }
                else
                {
                    // Check for Local ID third
                    ei = participant.Identifiers.FirstOrDefault();
                    if (ei != null)
                    {
                        value = ei.Extension;
                        namespaceId = "L";
                    }
                }
            }

            return new XCN
            {
                ID = value,
                FamilyName = participant.FamilyName,
                GivenName = participant.GivenNames.FirstOrDefault(),
                MiddleName = string.Join(" ", participant.GivenNames.Skip(1)),
                Prefix = string.Join(" ", participant.Titles),
                Suffix = string.Join(" ", participant.Suffixes),
                AssigningAuthority = new HD
                {
                    NamespaceID = namespaceId
                },
            };
        }

        /// <summary>
        /// Converts the message addressee individual to the XCN structure in the MDM model.
        /// </summary>
        /// <param name="individual">Message addressee individual.</param>
        /// <returns>Individual in XCN structure.</returns>
        private static XCN ToXCN(this ProviderIndividualIdentity individual)
        {
            string namespaceId = individual.Identifier.Type.Code;
            if (namespaceId == "HPII")
            {
                namespaceId = "AUSHIC";
            }
            return new XCN
            {
                ID = individual.Identifier.Value,
                FamilyName = individual.Name.FamilyName,
                GivenName = individual.Name.GivenNames.FirstOrDefault(),
                MiddleName = string.Join(" ", individual.Name.GivenNames.Skip(1)),
                Prefix = string.Join(" ", individual.Name.Titles),
                Suffix = string.Join(" ", individual.Name.Suffixes),
                AssigningAuthority = new HD
                {
                    NamespaceID = namespaceId
                },
            };
        }

        /// <summary>
        /// Converts HIPS SexEnumerator value to MDM Gender value.
        /// </summary>
        /// <param name="sex">Sex value.</param>
        /// <returns>Gender value.</returns>
        private static MDM.Common.Enums.Gender ToGender(this SexEnumerator sex)
        {
            switch (sex)
            {
                case SexEnumerator.Female: return MDM.Common.Enums.Gender.Female;
                case SexEnumerator.Male: return MDM.Common.Enums.Gender.Male;
                case SexEnumerator.IntersexOrIndeterminate: return MDM.Common.Enums.Gender.Ambiguous;
                case SexEnumerator.NotStatedOrInadequatelyDescribed: return MDM.Common.Enums.Gender.Unknown;
                default: return MDM.Common.Enums.Gender.Undefined;
            }
        }

        /// <summary>
        /// Converts the patient's name to an XPN in the MDM model.
        /// </summary>
        /// <param name="patient">The patient.</param>
        /// <returns>Patient Name as XPN.</returns>
        private static XPN ToXPN(this Participant patient)
        {
            return new XPN
            {
                FamilyName = patient.FamilyName,
                GivenName = patient.GivenNames.FirstOrDefault(),
                MiddleName = string.Join(" ", patient.GivenNames.Skip(1)),
                Prefix = string.Join(" ", patient.Titles),
                Suffix = string.Join(" ", patient.Suffixes),
            };
        }

        /// <summary>
        /// Converts an address to the XAD structure from the MDM model.
        /// </summary>
        /// <param name="address">Address.</param>
        /// <returns>Address as XAD model.</returns>
        private static XAD ToXAD(this Address address)
        {
            return new XAD
            {
                StreetAddress = address.AddressLine1,
                SecondStreetAddressLine = address.AddressLine2,
                City = address.PlaceName,
                State = string.IsNullOrEmpty(address.InternationalStateCode) 
                    ? address.AustralianState.ToString() 
                    : address.InternationalStateCode,
                PostCode = address.Postcode, 
            };
        }

        /// <summary>
        /// Converts an Entity Identifier into a CX structure in the MDM model.
        /// </summary>
        /// <param name="document">Entity identifier.</param>
        /// <returns>Entity identifier in the CX structure.</returns>
        private static CX ToCX(this EntityIdentifier ei)
        {
            CX cx = new CX();
            if (ei.Root.StartsWith(XmlStringMap.NationalHealthcareIdentifierOidPrefix))
            {
                cx.ID = ei.Root.RemoveOidPrefix();
                cx.AssiginingAuthority = new HD { NamespaceID = "AUSHIC" };
                cx.IdentifierTypeCode = "NI";
            }
            else if (ei.Root == XmlStringMap.PatientMedicareWithIrnOid || ei.Root == XmlStringMap.PatientMedicareWithoutIrnOid)
            {
                cx.ID = ei.Extension;
                   cx.AssiginingAuthority = new HD { NamespaceID = "AUSHIC" };
                    cx.IdentifierTypeCode = "MC";
            }
            else if (ei.Root == XmlStringMap.PatientDvaOid1 || ei.Root == XmlStringMap.PatientDvaOid2)
            {
                cx.ID = ei.Extension;
                cx.AssiginingAuthority = new HD { NamespaceID = "AUSDVA" };
                cx.IdentifierTypeCode = "DVA";
            }
            else
            {
                cx.ID = ei.Extension;
                cx.AssiginingAuthority = new HD { NamespaceID = ei.AssigningAuthorityName };
                cx.IdentifierTypeCode = "MR";
            }
            return cx;
        }
    }
}
