﻿namespace HIPS.CommonSchemas
{
    /// <summary>
    /// The CodeSystem enumeration
    /// </summary>
    public enum CodeSystemCode
    {
        /// <summary>
        /// Used as the code system for the Hi Reference tables
        /// </summary>
        HI_ReferenceCodes = 106
    }

    /// <summary>
    /// This is used to specify the conditions when the PCEHR
    /// advertised status should be refreshed.
    /// </summary>
    public enum ForceRefreshOption
    {
        /// <summary>
        /// Indicates that a value has not been specified
        /// providing this value will cause a validation error.
        /// </summary>
        None = 0,

        /// <summary>
        /// Indicates that there is no need to refresh the PCEHR
        /// advertised status and the previously stored information will be returned.
        /// </summary>
        Never = 1,

        /// <summary>
        /// Indicates that the PCEHR advertised status should be refreshed
        /// if there is no previously stored information that indicates the patient has a PCEHR.
        /// </summary>
        WhenNotAdvertised = 2,

        /// <summary>
        /// Indicates that the PCEHR advertised status should always be refreshed.
        /// </summary>
        Always = 3
    }

    /// <summary>
    /// This enumeration indicates the success or category of failure.
    ///
    /// *******************************************************************************
    /// Use CAUTION when modifying the values in this enumeration as it will affect the
    /// calling systems that are integrated with HIPS.
    ///
    /// *******************************************************************************
    ///
    /// </summary>
    public enum HipsResponseIndicator
    {
        /// <summary>
        /// The application failed. Details recorded within the application database.
        /// </summary>
        SystemError = 0,

        /// <summary>
        /// The request executed correctly
        /// </summary>
        OK = 1,

        /// <summary>
        /// The patient was not found with the specified search data
        /// </summary>
        InvalidPatient = 2,

        /// <summary>
        /// The hospital was not found with the specified search data
        /// </summary>
        InvalidHospital = 3,

        /// <summary>
        /// The episode was not found with the specified search data
        /// </summary>
        InvalidEpisode = 4,

        /// <summary>
        /// The document was not found with the specified search data
        /// </summary>
        InvalidDocument = 5,

        /// <summary>
        /// The user was not correctly specified
        /// </summary>
        InvalidUser = 6,

        /// <summary>
        /// The HI service returned an error
        /// </summary>
        HiServiceError = 8,

        /// <summary>
        /// The PCEHR service returned an error
        /// </summary>
        PcehrServiceError = 9,

        /// <summary>
        /// The patient's IHI could not be found or validated
        /// </summary>
        InvalidIhi = 10,

        /// <summary>
        /// The document could not be uploaded because consent was withdrawn
        /// </summary>
        ConsentWithdrawn = 11,

        /// <summary>
        /// HIPS could not add the operation (upload or remove) to the MSMQ queue.
        /// This can happen if the queue is full. The calling system should retry
        /// the operation.
        /// </summary>
        CouldNotAddToQueue = 12,

        /// <summary>
        /// The Access Code is invalid.
        /// </summary>
        InvalidAccessCode = 13,

        /// <summary>
        /// Demographics in the downloaded document did not match those in the local system.
        /// This will occur when the Date of Birth, Sex or Family Name are different.
        /// </summary>
        DemographicMismatchWarning = 14,

        /// <summary>
        /// The IHI for this patient has an unresolved alert. This patient's PCEHR cannot be
        /// accessed until the alert has been resolved.
        /// </summary>
        UnresolvedIhiAlert = 15,

        /// <summary>
        /// The operation could not be completed because the local SQL Server database is
        /// unavailable. The calling system should retry the operation.
        /// </summary>
        DatabaseError = 16,

        /// <summary>
        /// The date of birth stored for the patient does not match
        /// the date of birth that was specified in the request.
        /// </summary>
        InvalidDateOfBirth = 17,

        /// <summary>
        /// The action was successful but returned one or more warnings that should be
        /// investigated as available resources permit.
        /// </summary>
        PcehrServiceWarning = 18,

        /// <summary>
        /// The PCEHR Service is temporarily unavailable. The action should be tried again later.
        /// In the case of an upload or remove operation, the queue transaction will be rolled
        /// back so that the MSMQ will handle retrying.
        /// </summary>
        PcehrServiceUnavailable = 19,

        /// <summary>
        /// The document was not uploaded because the patient was under the configured minimum
        /// age at the time of admission. This is a patient safety initiative.
        /// </summary>
        PatientUnderAge = 20,

        /// <summary>
        /// This queue item could not be found - it may have been removed from the list or the ID is invalid
        /// </summary>
        CouldNotFindQueueItem = 21,

        /// <summary>
        /// There was a validation error. Check the HIPS error message and the response code,
        /// description and detail for more information.
        /// </summary>
        ValidationError = 22,

        /// <summary>
        /// The state patient id is incorrect
        /// </summary>
        IncorrectStatePatientId = 23,

        /// <summary>
        /// The supplied MRN is incorrect
        /// </summary>
        IncorrectMrn = 24,

        /// <summary>
        /// The HPI-I for this health provider has an unresolved alert. This health provider cannot be used to access the PCEHR
        /// until the alert has been resolved.
        /// </summary>
        UnresolvedHpiiAlert = 25,

        /// <summary>
        /// The HPI-I for this health provider has a warning raised against it. This health provider can be used to access the PCEHR
        /// but the warning must be logged for this HPI_I.
        /// </summary>
        HpiiWarningRaised = 26,
    }

    /// <summary>
    /// The role of the user
    /// </summary>
    public enum UserRole
    {
        /// <summary>
        /// Individual Healthcare Provider identified by an HPI-I
        /// </summary>
        ProviderIndividual,

        /// <summary>
        /// Interactive access by clinician or patient administration clerk etc.
        /// </summary>
        InteractiveUser,

        /// <summary>
        /// Authorised employee responsible for non-interactive access by batch or background processes
        /// </summary>
        AuthorisedEmployee,
    }

    /// <summary>
    /// Represents the format of a CDA package.
    /// </summary>
    public enum CdaPackageFormat
    {
        /// <summary>
        /// No format has been defined.
        /// </summary>
        None = 0,

        /// <summary>
        /// The package format is XDM-ZIP.
        /// </summary>
        XdmZip = 1,

        /// <summary>
        /// The package format is HL7-MDM.
        /// </summary>
        Hl7Mdm = 2,
    }

    /// <summary>
    /// Pathology Discipline
    /// </summary>
    public enum PathologyDiscipline
    {
        /// <summary>
        /// Audiology
        /// </summary>
        AU,
        
        /// <summary>
        /// Bedside ICU monitoring
        /// </summary>
        ICU,
        
        /// <summary>
        /// Blood Bank
        /// </summary>
        BLB,
        
        /// <summary>
        /// Blood Gases
        /// </summary>
        BG,
        
        /// <summary>
        /// Cardiac Catheterization
        /// </summary>
        CTH,
        
        /// <summary>
        /// Cardiac Ultrasound
        /// </summary>
        CUS,
        
        /// <summary>
        /// CAT Scan
        /// </summary>
        CT,
        
        /// <summary>
        /// Chemistry
        /// </summary>
        CH,
        
        /// <summary>
        /// Cineradiograph
        /// </summary>
        XRC,
        
        /// <summary>
        /// Cytopathology
        /// </summary>
        CP,
        
        /// <summary>
        /// Electrocardiac (e.g., EKG, EEC, Holter)
        /// </summary>
        EC,
        
        /// <summary>
        /// Electroneuro (e.g., EEG, EMG, EP, PSG)
        /// </summary>
        EN,
        
        /// <summary>
        /// Hematology
        /// </summary>
        HM,
        
        /// <summary>
        /// Immunology
        /// </summary>
        IMM,
        
        /// <summary>
        /// Laboratory
        /// </summary>
        LAB,
        
        /// <summary>
        /// Microbiology
        /// </summary>
        MB,
        
        /// <summary>
        /// Mycobacteriology
        /// </summary>
        MCB,
        
        /// <summary>
        /// Mycology
        /// </summary>
        MYC,
        
        /// <summary>
        /// Nuclear Magnetic Resonance
        /// </summary>
        NMR,
        
        /// <summary>
        /// Nuclear Medicine Scan
        /// </summary>
        NMS,
        
        /// <summary>
        /// Nursing Service Measures
        /// </summary>
        NRS,
        
        /// <summary>
        /// OB Ultrasound
        /// </summary>
        OUS,
        
        /// <summary>
        /// Occupational Therapy
        /// </summary>
        OT,
        
        /// <summary>
        /// Other
        /// </summary>
        OTH,
        
        /// <summary>
        /// Outside Lab
        /// </summary>
        OSL,
        
        /// <summary>
        /// Pharmacy
        /// </summary>
        PHR,
        
        /// <summary>
        /// Physical Therapy
        /// </summary>
        PT,
        
        /// <summary>
        /// Physician (Hx. Dx, admission note, etc.)
        /// </summary>
        PHY,
        
        /// <summary>
        /// Pulmonary Function
        /// </summary>
        PF,
        
        /// <summary>
        /// Radiation Therapy
        /// </summary>
        RT,
        
        /// <summary>
        /// Radiograph
        /// </summary>
        RX,
        
        /// <summary>
        /// Radiology
        /// </summary>
        RAD,
        
        /// <summary>
        /// Radiologu Ultrasound
        /// </summary>
        RUS,
        
        /// <summary>
        /// Respiratory Care (therapy)
        /// </summary>
        RC,
        
        /// <summary>
        /// Serology
        /// </summary>
        SR,
        
        /// <summary>
        /// Surgical Pathology
        /// </summary>
        SP,
        
        /// <summary>
        /// Toxicology
        /// </summary>
        TX,
        
        /// <summary>
        /// Vascular Ultrasound
        /// </summary>
        VUS,
        
        /// <summary>
        /// Virology
        /// </summary>
        VR
    }

    // *******************************************************************************
    // Use CAUTION when modifying the values in the HipsResponseIndicator enumeration
    // as it will affect the calling systems that are integrated with HIPS.
    //
    // *******************************************************************************
}