﻿using System;
using System.Data.SqlClient;

using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the CdaSetNumber table.
    /// </summary>
    public class CdaSetNumberDl : BaseDl
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="CdaSetNumberDl" /> class.
        /// </summary>
        /// <param name="user">Identifies the person who is responsible for the action.</param>
        public CdaSetNumberDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets a CdaSetNumber by its numeric ID.
        /// </summary>
        /// <param name="cdaSetNumberId">CdaSetNumber ID.</param>
        /// <returns>CdaSetNumber object.</returns>
        public CdaSetNumber GetByCdaSetNumberId(int cdaSetNumberId)
        {
            CdaSetNumber result = new CdaSetNumber();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.CdaSetNumberGet"))
                {
                    command.Parameters.Add(new SqlParameter("@CdaSetNumberId", cdaSetNumberId));
                    PopulateBusinessObject<CdaSetNumber>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageCdaSetNumberGetById, ex, User, LogMessage.HIPS_MESSAGE_162);
            }
            return result;
        }

        /// <summary>
        /// Gets a CdaSetNumber by episode ID and document type ID.
        /// </summary>
        /// <param name="episodeId">Episode ID.</param>
        /// <param name="documentTypeId">Document Type ID.</param>
        /// <param name="transaction"></param>
        /// <returns>CdaSetNumber object.</returns>
        public CdaSetNumber GetByEpisodeAndDocumentType(int episodeId, int documentTypeId, SqlTransaction transaction = null)
        {
            CdaSetNumber result = new CdaSetNumber();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.CdaSetNumberGet", transaction))
                {
                    command.Parameters.Add(new SqlParameter("@EpisodeId", episodeId));
                    command.Parameters.Add(new SqlParameter("@DocumentTypeId", documentTypeId));
                    PopulateBusinessObject<CdaSetNumber>(command.ExecuteReader(), result);
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageCdaSetNumberGetByEpisodeAndDocumentType, ex, User, LogMessage.HIPS_MESSAGE_163);
            }
            return result;
        }

        /// <summary>
        /// Gets a SQL command for external transaction handling.
        /// </summary>
        /// <returns>SQL command.</returns>
        public SqlCommand GetCommand()
        {
            return base.Command;
        }

        /// <summary>
        /// Saves the specified CdaSetNumber.
        /// </summary>
        /// <param name="CdaSetNumber">The CdaSetNumber.</param>
        /// <returns>Whether the item was saved.</returns>
        public bool Save(CdaSetNumber item, SqlTransaction transaction)
        {
            if (item.CdaSetNumberId.HasValue)
            {
                using (SqlCommand command = GetSqlCommand("hips.CdaSetNumberUpdate", transaction))
                {
                    return base.Update<CdaSetNumber>(item, command);
                }
            }
            else
            {
                using (SqlCommand command = GetSqlCommand("hips.CdaSetNumberInsert", transaction))
                {
                    return base.Insert<CdaSetNumber>(item, command);
                }
            }
        }

        #endregion Methods
    }
}