﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the patient  table
    /// </summary>
    public class HospitalPatientDl : BaseDl
    {
        #region Constructors

        public HospitalPatientDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Deletes the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Delete(HospitalPatient item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.HospitalPatientDelete", transaction))
            {
                return base.Delete<HospitalPatient>(item, command);
            }
        }

        /// <summary>
        /// Gets the specified hospital patient by patient id.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="item">The item.</param>
        /// <returns>Whether a hospital patient record was fetched</returns>
        public bool Get(int patientId, out HospitalPatient item)
        {
            item = new HospitalPatient();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalPatientGet"))
                {
                    AddIntParameter("PatientId", patientId, command);
                    result = PopulateBusinessObject<HospitalPatient>(command.ExecuteReader(), item);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHospitalPatientGetMrnDatabaseError, ex, User, LogMessage.HIPS_MESSAGE_128);
            }
            return result;
        }

        /// <summary>
        /// Gets the specified patient by hospital id and Mrn.
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <param name="mrn">The Mrn.</param>
        /// <param name="item">The item.</param>
        /// <param name="transaction">The transaction (optional)</param>
        /// <returns>Indicator that the patient was found (OK), not found (InvalidPatient) or an error (DatabaseError).</returns>
        public HipsResponse Get(int hospitalId, string mrn, out HospitalPatient item, SqlTransaction transaction = null)
        {
            item = new HospitalPatient();
            HipsResponse result = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalPatientGet", transaction))
                {
                    command.Parameters.Add(new SqlParameter("@HospitalId", hospitalId));
                    command.Parameters.Add(new SqlParameter("@Mrn", mrn));
                    if (!PopulateBusinessObject<HospitalPatient>(command.ExecuteReader(), item))
                    {
                        string message = string.Format(ConstantsResource.ErrorMessageHospitalPatientGetMrnNotFound, mrn, hospitalId);
                        result.HipsErrorMessage = message;
                        result.Status = HipsResponseIndicator.InvalidPatient;
                    }
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessageHospitalPatientGetMrnDatabaseError, mrn, hospitalId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_129);
                result.Status = HipsResponseIndicator.DatabaseError;
                result.HipsErrorMessage = message;
                result.ResponseCodeDescription = ex.Message;
            }
            return result;
        }

        /// <summary>
        /// Gets the specified patient by Mrn only.
        /// </summary>
        /// <param name="mrn">The Mrn.</param>
        /// <param name="item">The item.</param>
        /// <param name="transaction">The transaction (optional)</param>
        /// <returns>Indicator that the patient was found (OK), not found (InvalidPatient) or an error (DatabaseError).</returns>
        public HospitalPatient GetByMrn(string mrn)
        {
            var result = new HospitalPatient();

            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalPatientByMrnGet"))
                {
                    command.Parameters.Add(new SqlParameter("@Mrn", mrn));
                    if (!PopulateBusinessObject<HospitalPatient>(command.ExecuteReader(), result))
                    {
                        result = null;
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessageHospitalPatientGetByMrnDatabaseError, mrn);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_181);
            }
            return result;
        }

        /// <summary>
        /// Gets the active MRN for a given hospital and patient master. The
        /// active MRN should be the only one with any episodes.
        ///
        /// It is a logical error for a patient to have episodes on more than
        /// one MRN from the same hospital. This could only happen if HIPS
        /// receives an enterprise merge (A34) or move (A43) that attaches
        /// both MRNs to the same SAUHI, without first receiving the expected
        /// MRN merge (A36). In this case this method will return the MRN with
        /// the most recent episode (by AdmissionDate).
        ///
        /// If the latest episode on multiple MRNs has the exact same admission
        /// date/time, the one returned is the most recently modified
        /// HospitalPatient record for the specified patient and hospital.
        ///
        /// If none of the MRNs has any episodes then this will return the most
        /// recently modified HospitalPatient record for the specified patient
        /// and hospital.
        /// </summary>
        /// <returns>Whether a record was found</returns>
        public HipsResponse GetActive(int hospitalId, int patientMasterId, out HospitalPatient item)
        {
            item = new HospitalPatient();
            HipsResponse result = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalPatientGet"))
                {
                    AddIntParameter("@HospitalId", hospitalId, command);
                    AddIntParameter("@PatientMasterId", patientMasterId, command);
                    AddBoolParameter("@ActiveOnly", true, command);
                    if (!PopulateBusinessObject<HospitalPatient>(command.ExecuteReader(), item))
                    {
                        result.Status = HipsResponseIndicator.InvalidPatient;
                    }
                    command.Connection.Close();
                }
                if (result.Status == HipsResponseIndicator.InvalidPatient)
                {
                    // No MRNs found with episodes, fall back to getting the first (most recently modified).
                    using (SqlCommand command = GetSqlCommand("hips.HospitalPatientGet"))
                    {
                        AddIntParameter("@HospitalId", hospitalId, command);
                        AddIntParameter("@PatientMasterId", patientMasterId, command);
                        if (PopulateBusinessObject<HospitalPatient>(command.ExecuteReader(), item))
                        {
                            result.Status = HipsResponseIndicator.OK;
                        }
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorHospitalPatientGetActive, patientMasterId, hospitalId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_127);
                result.Status = HipsResponseIndicator.DatabaseError;
                result.HipsErrorMessage = message;
                result.ResponseCodeDescription = ex.Message;
            }
            return result;
        }

        /// <summary>
        /// Gets all MRNs for a given hospital and patient master. Note that this may include inactive (merged) MRNs.
        /// </summary>
        /// <param name="hospitalId">The hospital ID (optional)</param>
        /// <param name="patientMasterId">The patient master ID</param>
        /// <param name="transaction">An existing transaction context in which to perform the query (optional)</param>
        /// <returns>List of hospital patients</returns>
        public List<HospitalPatient> GetAll(int? hospitalId, int patientMasterId, SqlTransaction transaction = null)
        {
            List<HospitalPatient> results = new List<HospitalPatient>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalPatientGet", transaction))
                {
                    AddIntParameter("@HospitalId", hospitalId, command);
                    AddIntParameter("@PatientMasterId", patientMasterId, command);
                    results = GetPopulatedBusinessList<HospitalPatient>(command.ExecuteReader());
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorHospitalPatientGetAll, patientMasterId, hospitalId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_050);
                if (transaction != null)
                {
                    transaction.Rollback();
                    transaction.Connection.Close();
                }
            }
            return results;
        }

        /// <summary>
        /// Gets only the MRN associated with the latest episode for the specified patient master at each hospital.
        /// In 99% of cases this will return only one MRN for the patient at each hospital.
        /// The only case where it will return more than one is if there are episodes with the
        /// exact same admission date/time on multiple MRNs. In this case they are ordered with the
        /// most recently modified HospitalPatient record first.
        /// Note: If a patient has an MRN at a hospital but no episodes under
        /// it, then that MRN will not be returned by this method.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <returns>
        /// List of active hospital patient records
        /// </returns>
        public List<HospitalPatient> GetAllActive(int patientMasterId)
        {
            List<HospitalPatient> results = new List<HospitalPatient>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalPatientGet"))
                {
                    AddIntParameter("@PatientMasterId", patientMasterId, command);
                    AddBoolParameter("@ActiveOnly", true, command);
                    results = GetPopulatedBusinessList<HospitalPatient>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorHospitalPatientGetAllActive, patientMasterId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_126);
            }
            return results;
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Insert(HospitalPatient item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.HospitalPatientInsert", transaction))
            {
                return base.Insert<HospitalPatient>(item, command);
            }
        }

        /// <summary>
        /// Moves the hospital patient, and all hospital patients from the same hospital
        /// that have previously been merged with it, from one patient master to another.
        /// Raises a merge conflict alert if the IHI's on the two patient masters differ
        /// and there was already an MRN from the same hospital on the destination PM.
        ///
        /// Populates new information into both the hospitalPatient and toPatientMaster
        /// parameters. The hospitalPatient will change PatientMasterId, and the
        /// toPatientMaster may change IhiStatusId to MergeConflict.
        /// </summary>
        /// <param name="hospitalPatient">The hospital patient (MRN) that is moving.</param>
        /// <param name="toPatientMaster">The destination patient master.</param>
        /// <param name="transaction">The SQL transaction in which to perform the action.</param>
        /// <returns>Indicator of whether the move was successful (OK) or a database error occured (DatabaseError).</returns>
        public HipsResponse Move(HospitalPatient hospitalPatient, PatientMaster toPatientMaster, SqlTransaction transaction)
        {
            HipsResponse result = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HospitalPatientMove", transaction))
                {
                    AddIntParameter("@PatientId", hospitalPatient.PatientId, command);
                    AddIntParameter("@HospitalId", hospitalPatient.HospitalId, command);
                    AddIntParameter("@FromPatientMasterId", hospitalPatient.PatientMasterId, command);
                    AddIntParameter("@ToPatientMasterId", toPatientMaster.PatientMasterId, command);
                    AddStringParameter("@UserModified", GetUserModified(), command);
                    AddDateParameter("@DateModified", hospitalPatient.DateModified, command);
                    if (!PopulateBusinessObject<HospitalPatient>(command.ExecuteReader(), hospitalPatient))
                    {
                        throw new Exception(ConstantsResource.NoRowsReturned);
                    }

                    // This doesn't call PatientMasterDl.Get because we want to repopulate the same
                    // object that was passed in, rather than getting a new object.
                    using (SqlCommand command2 = GetSqlCommand("hips.PatientMasterGet", transaction))
                    {
                        AddIntParameter("@PatientMasterId", toPatientMaster.PatientMasterId, command2);
                        AddIntParameter("@HealthProviderOrganisationNetworkId", toPatientMaster.HealthProviderOrganisationNetworkId, command2);
                        if (!PopulateBusinessObject<PatientMaster>(command2.ExecuteReader(), toPatientMaster))
                        {
                            throw new Exception(ConstantsResource.NoRowsReturned);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessageHospitalPatientMove,
                    hospitalPatient.Mrn,
                    hospitalPatient.HospitalId,
                    hospitalPatient.PatientMasterId,
                    toPatientMaster.PatientMasterId.Value);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_130);
                transaction.Rollback();
                Command.Connection.Close();
                result.Status = HipsResponseIndicator.DatabaseError;
                result.HipsErrorMessage = message;
            }
            return result;
        }

        /// <summary>
        /// Gets single patient in hospital with the latest episode details.
        /// </summary>
        /// <param name="hospitalPatient">The hospital patient.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <returns></returns>
        public PatientDisclosureDetails SinglePatientInHospital(HospitalPatient hospitalPatient, string hospitalCodeSystem, int healthProviderOrganisationNetworkId)
        {
            PatientInHospital patient = new PatientInHospital();
            PatientDisclosureDetails disclosure = new PatientDisclosureDetails();

            try
            {
                using (SqlCommand command = GetSqlCommand("hips.SinglePatientInHospitalGet"))
                {
                    AddStringParameter("@HospitalCodeSystem", hospitalCodeSystem, command);
                    AddIntParameter("@PatientId", hospitalPatient.PatientId, command);
                    AddIntParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId, command);
                    PopulateBusinessObject<PatientInHospital>(command.ExecuteReader(), patient);
                    command.Connection.Close();
                }

                disclosure.PatientInHospital = patient;

                using (SqlCommand command = GetSqlCommand("hips.PatientDisclosureDetailsGet"))
                {
                    AddStringParameter("@HospitalCodeSystem", hospitalCodeSystem, command);
                    AddIntParameter("@PatientId", hospitalPatient.PatientId, command);
                    PopulateBusinessObject<PatientDisclosureDetails>(command.ExecuteReader(), disclosure);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorSinglePatientExtendedDetailsGet, hospitalPatient.PatientId, hospitalCodeSystem), ex, User, LogMessage.HIPS_MESSAGE_159);
            }
            return disclosure;
        }

        /// <summary>
        /// Updates the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Update(HospitalPatient item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.HospitalPatientUpdate", transaction))
            {
                return base.Update<HospitalPatient>(item, command);
            }
        }

        #endregion Methods
    }
}