﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class retrieves data to show a list of patients with or without a PCEHR.
    /// </summary>
    public class PatientListDl : BaseDl
    {
        #region Constructors

        public PatientListDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Lists patients in hospital that match a set of criteria.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalId">The hospital identifier. If not supplied,
        ///     lists patients in all hospitals in the specified code system.
        ///     </param>
        /// <param name="withIhi">True to list only patients with an active IHI.
        ///     False to list only patients without an IHI. Null to list
        ///     patients regardless of whether they have an IHI.</param>
        /// <param name="withPcehr">True to list only patients with a PCEHR
        ///     (advertised or disclosed). False to list only patients without
        ///     a PCEHR. Null to list patients regardless of whether they have
        ///     a PCEHR.</param>
        /// <param name="daysDischarged">In addition to currently admitted
        ///     patients, patients who were discharged up to this many days
        ///     ago will be included in the list.</param>
        /// <param name="daysAfterService">Number of days after service date 
        ///     that non-inpatients are included in the list. Set to 0 to 
        ///     exclude non-inpatients.</param>
        /// <returns>List of patients in hospital matching the criteria given.</returns>
        public HipsResponse ListPatientsInHospital(string hospitalCodeSystem, int? hospitalId, bool? withIhi, bool? withPcehr, bool excludeMedicareExclusions, int daysDischarged, int daysAfterService, out List<PatientInHospital> results)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            results = new List<PatientInHospital>();
            try
            {
                // Populate the patient list
                using (SqlCommand command = GetSqlCommand("hips.PatientInHospitalGet"))
                {
                    AddStringParameter("@HospitalCodeSystem", hospitalCodeSystem, command);
                    AddIntParameter("@HospitalId", hospitalId, command);
                    AddBoolParameter("@WithIhi", withIhi, command);
                    AddBoolParameter("@WithPcehr", withPcehr, command);
                    AddBoolParameter("@ExcludeMedicareExclusions", excludeMedicareExclusions, command);
                    AddIntParameter("@DaysDischarged", daysDischarged, command);
                    AddIntParameter("@DaysAfterService", daysAfterService, command);
                    results = GetPopulatedBusinessList<PatientInHospital>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessagePatientInHospitalList, ex, User, LogMessage.HIPS_MESSAGE_156);
                response.Status = HipsResponseIndicator.DatabaseError;
                response.HipsErrorMessage = ConstantsResource.ErrorMessagePatientInHospitalList;
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            return response;
        }

        #endregion Methods
    }
}