﻿using System;

namespace HIPS.PcehrDataStore.Schemas.Enumerators
{
    /// <summary>
    /// This is used to match the status in the AccessCodeRequired table primary key to code.
    /// </summary>
    [Serializable]
    public enum AccessCodeRequired
    {
        /// <summary>
        /// The access status is unknown.
        /// </summary>
        Unknown = -1,

        /// <summary>
        /// Except in an emergency, the patient must provide his/her
        /// Record Access Code for the healthcare provider organisation
        /// to gain access to this PCEHR.
        /// </summary>
        WithCode = 0,

        /// <summary>
        /// No code is required for the health provider organisation
        /// to gain access to this PCEHR.
        /// </summary>
        WithoutCode = 1,

        /// <summary>
        /// The health provider organisation has already gained
        /// access to this PCEHR.
        /// </summary>
        AccessGranted = 2,
    }

    /// <summary>
    /// This enumerator is used to match the ClinicalDocumentStatus table primary key to code.
    /// </summary>
    [Serializable]
    public enum ClinicalDocumentStatus
    {
        /// <summary>
        /// The clinical document has been uploaded and has not been removed by the provider organisation that uploaded it.
        /// This does not preclude the possibility that the consumer has removed the document.
        /// </summary>
        Active = 1,

        /// <summary>
        /// The clinical document has been removed by the provider organisation that uploaded it.
        /// </summary>
        Removed = 2
    }

    /// <summary>
    /// This enumeration is used to match the death indicator table to code.
    /// </summary>
    [Serializable]
    public enum DeathIndicator
    {
        /// <summary>
        /// The death indicator is unknown.This would only ever happen if something was
        /// seriously wrong, as incorrect dates will get set as InvalidDate.
        /// </summary>
        Unknown = -1,

        /// <summary>
        /// The date of death was valid.
        /// </summary>
        ValidDate = 1,

        /// <summary>
        /// The date of death was invalid.
        /// </summary>
        InvalidDate = 2
    }

    /// <summary>
    /// This is used to match the episode lifecycle in the EpisodeLifecycle table to code.
    /// </summary>
    [Serializable]
    public enum EpisodeLifecycle
    {
        /// <summary>
        /// The appointment status is unknown.
        /// </summary>
        UnknownAppointmentStatus = -1,

        /// <summary>
        /// An outpatient visit has been booked.
        /// </summary>
        Booked = 1,

        /// <summary>
        /// The patient attended the outpatient appointment and was seen.
        /// </summary>
        AttendedSeen = 2,

        /// <summary>
        /// The outpatient appointment was deleted.
        /// </summary>
        Deleted = 3,

        /// <summary>
        /// The outpatient appointment was cancelled by the patient.
        /// </summary>
        CancelledByPatient = 4,

        /// <summary>
        /// The outpatient appointment was cancelled by the hospital.
        /// </summary>
        CancelledByHospital = 5,

        /// <summary>
        /// The outpatient appointment was cancelled for another reason.
        /// </summary>
        CancelledOther = 6,

        /// <summary>
        /// The patient failed to attend the outpatient appointment.
        /// </summary>
        FailedToAttend = 7,

        /// <summary>
        /// An inpatient visit is planned.
        /// </summary>
        PreAdmit = 9,

        /// <summary>
        /// An planned inpatient visit was cancelled.
        /// </summary>
        CancelledPreAdmit = 10,

        /// <summary>
        /// The inpatient visit is in progress.
        /// </summary>
        Admitted = 11,

        /// <summary>
        /// The inpatient visit has been cancelled.
        /// </summary>
        CancelledAdmission = 12,

        /// <summary>
        /// The inpatient visit has ended and the patient was discharged (or administratively discharged into another visit).
        /// </summary>
        Discharged = 13,

        /// <summary>
        /// This visit was merged with another visit and the other visit contains the current information.
        /// </summary>
        Merged = 14
    }

    /// <summary>
    /// This enumeration is used to match the record status in the IhiRecordStatus table to code
    /// </summary>
    [Serializable]
    public enum IhiRecordStatus
    {
        /// <summary>
        /// The IHI is a verified IHI.
        /// </summary>
        Verified = 1,

        /// <summary>
        /// The IHI is an unverified IHI.
        /// </summary>
        Unverified = 2,

        /// <summary>
        /// The IHI is a provisional IHI.
        /// </summary>
        Provisional = 3,

        /// <summary>
        /// The IHI record status is unknown or not applicable.
        /// </summary>
        Unknown = -1
    }

    /// <summary>
    /// This is used to match the status in the IhiStatus table to code
    /// </summary>
    [Serializable]
    public enum IhiStatus
    {
        /// <summary>
        /// The IHI status is Active.
        /// </summary>
        Active = 1,

        /// <summary>
        /// The IHI status is Deceased.
        /// </summary>
        Deceased = 2,

        /// <summary>
        /// The IHI status is Retired.
        /// </summary>
        Retired = 3,

        /// <summary>
        /// The IHI status is Expired.
        /// </summary>
        Expired = 4,

        /// <summary>
        /// The IHI status is Resolved. This status should never be stored for the current IHI of a patient, but will be stored in IHI history.
        /// </summary>
        Resolved = 5,

        /// <summary>
        /// The IHI status is unknown or not applicable (including when no IHI was found).
        /// </summary>
        Unknown = -1,

        /// <summary>
        /// The patient has an unresolved alert because another patient record in the same hospital had the same IHI number.
        /// </summary>
        DuplicateIhi = 100,

        /// <summary>
        /// The patient has an unresolved alert because another patient record in the same hospital had demographic information identical to that used to obtain the IHI for this patient.
        /// </summary>
        DuplicatePatient = 101,

        /// <summary>
        /// The patient has an unresolved alert because it was merged with another patient record that had a different IHI number.
        /// </summary>
        MergeConflict = 102,

        /// <summary>
        /// The IHI or PCEHR lookup failed and will be retried by the background IHI cleanup service.
        /// </summary>
        ServiceUnavailable = 103,

        /// <summary>
        /// The patient has an unresolved alert because the Medicare or DVA number was changed,
        /// resulting in an IHI search that returned no IHI or returned a different IHI to that
        /// which was assigned to the patient.
        /// </summary>
        MedicareDvaChangeMismatch = 104,

        /// <summary>
        /// The patient has an unresolved alert because no IHI was found when the IHI assigned
        /// to the patient was validated using the patient's name, sex and date of birth.
        /// </summary>
        DemographicMismatch = 105
    }

    /// <summary>
    /// This is used to match the LogMessage table to code.
    /// </summary>
    [Serializable]
    public enum LogMessage
    {
        /// <summary>
        /// BaseDl.Connection: Unable to open connection to database.
        /// </summary>
        HIPS_MESSAGE_001 = 1,

        /// <summary>
        /// BaseDl.Command: Unable to create SQL command.
        /// </summary>
        HIPS_MESSAGE_002 = 2,

        /// <summary>
        /// PatientMasterAddressDl.Get: Database error while calling PatientMasterAddressGet.
        /// </summary>
        HIPS_MESSAGE_003 = 3,

        /// <summary>
        /// AddressDl.Get: Database error while calling AddressGet.
        /// </summary>
        HIPS_MESSAGE_004 = 4,

        /// <summary>
        /// AddressDl.GetAllByPatient: Database error while calling AddressByPatientList.
        /// </summary>
        HIPS_MESSAGE_005 = 5,

        /// <summary>
        /// ContactDl.Get: Database error while calling ContactGet.
        /// </summary>
        HIPS_MESSAGE_006 = 6,

        /// <summary>
        /// ContactDl.GetAllByPatient: Database error while calling PatientMasterContactGet.
        /// </summary>
        HIPS_MESSAGE_007 = 7,

        /// <summary>
        /// HospitalCodeDl.GetAll: Database error while calling HospitalCodeGet.
        /// </summary>
        HIPS_MESSAGE_008 = 8,

        /// <summary>
        /// PatientMasterNameDl.GetAllByPatient: Database error while calling PatientMasterNameGet.
        /// </summary>
        HIPS_MESSAGE_009 = 9,

        /// <summary>
        /// HL7MessageLogDl.Get: Database error while calling HL7MessageLogGet by ID.
        /// </summary>
        HIPS_MESSAGE_010 = 10,

        /// <summary>
        /// DownloadedDocumentDl.GetAll: Database error while calling DownloadedDocumentGet by PatientMasterId.
        /// </summary>
        HIPS_MESSAGE_011 = 11,

        /// <summary>
        /// DownloadedDocumentDl.Get: Database error while calling DownloadedDocumentGet by SourceSystemDocumentId.
        /// </summary>
        HIPS_MESSAGE_012 = 12,

        /// <summary>
        /// EpisodeLifecycleDl.GetAll: Database error while calling EpisodeLifecycleGet.
        /// </summary>
        HIPS_MESSAGE_013 = 13,

        /// <summary>
        /// EpisodeLifecycleDl.Get: Database error while calling EpisodeLifecycleGet by ID.
        /// </summary>
        HIPS_MESSAGE_014 = 14,

        /// <summary>
        /// ClinicalDocumentVersionDl.GetAllVersions: Database error getting all ClinicalDocumentVersion for document.
        /// </summary>
        HIPS_MESSAGE_015 = 15,

        /// <summary>
        /// ClinicalDocumentVersionDl.GetLatestVersion: Database error getting latest ClinicalDocumentVersion for document.
        /// </summary>
        HIPS_MESSAGE_016 = 16,

        /// <summary>
        /// ClinicalDocumentVersionDl.GetVersion: Database error getting specific ClinicalDocumentVersion.
        /// </summary>
        HIPS_MESSAGE_017 = 17,

        /// <summary>
        /// DocumentFormatDl.GetAll: Database error while calling DocumentFormatGet.
        /// </summary>
        HIPS_MESSAGE_018 = 18,

        /// <summary>
        /// DocumentFormatDl.Get: Database error while calling DocumentFormatGet by ID.
        /// </summary>
        HIPS_MESSAGE_019 = 19,

        /// <summary>
        /// StateDl.GetAll: Database error while calling StateGet.
        /// </summary>
        HIPS_MESSAGE_020 = 20,

        /// <summary>
        /// StateDl.Get: Database error while calling StateGet by ID.
        /// </summary>
        HIPS_MESSAGE_021 = 21,

        /// <summary>
        /// TitleDl.GetAll: Database error while calling TitleGet.
        /// </summary>
        HIPS_MESSAGE_022 = 22,

        /// <summary>
        /// TitleDl.GetAll: Database error while calling TitleGet by ID.
        /// </summary>
        HIPS_MESSAGE_023 = 23,

        /// <summary>
        /// SuffixDl.GetAll: Database error while calling SuffixGet.
        /// </summary>
        HIPS_MESSAGE_024 = 24,

        /// <summary>
        /// SuffixDl.Get: Database error while calling SuffixGet by ID.
        /// </summary>
        HIPS_MESSAGE_025 = 25,

        /// <summary>
        /// DocumentTypeDl.GetAll: Database error while calling DocumentTypeGet.
        /// </summary>
        HIPS_MESSAGE_026 = 26,

        /// <summary>
        /// DocumentTypeDl.Get: Database error while calling DocumentTypeGet by ID.
        /// </summary>
        HIPS_MESSAGE_027 = 27,

        /// <summary>
        /// MedicareExclusionDl.GetAll: Database error while calling MedicareExclusionGet.
        /// </summary>
        HIPS_MESSAGE_028 = 28,

        /// <summary>
        /// MedicareExclusionDl.Get: Database error while calling MedicareExclusionGet by ID.
        /// </summary>
        HIPS_MESSAGE_029 = 29,

        /// <summary>
        /// SexDl.GetAll: Database error while calling SexGet.
        /// </summary>
        HIPS_MESSAGE_030 = 30,

        /// <summary>
        /// SexDl.Get: Database error while calling SexGet by ID.
        /// </summary>
        HIPS_MESSAGE_031 = 31,

        /// <summary>
        /// EpisodeTypeDl.GetAll: Database error while calling EpisodeTypeGet.
        /// </summary>
        HIPS_MESSAGE_032 = 32,

        /// <summary>
        /// EpisodeTypeDl.Get: Database error while calling EpisodeTypeGet by ID.
        /// </summary>
        HIPS_MESSAGE_033 = 33,

        /// <summary>
        /// CodeSystemDl.GetAll: Database error while calling CodeSystemGet.
        /// </summary>
        HIPS_MESSAGE_034 = 34,

        /// <summary>
        /// CodeSystemDl.Get: Database error while calling CodeSystemGet by ID.
        /// </summary>
        HIPS_MESSAGE_035 = 35,

        /// <summary>
        /// ClinicalDocumentDl.GetAll: Database error while calling ClinicalDocumentGet by EpisodeId and DocumentTypeId.
        /// </summary>
        HIPS_MESSAGE_036 = 36,

        /// <summary>
        /// ClinicalDocumentDl.Get: Database error while calling ClinicalDocumentGet by EpisodeId and SourceSystemSetId.
        /// </summary>
        HIPS_MESSAGE_037 = 37,

        /// <summary>
        /// RemovalStatusDl.GetAll: Database error while calling RemovalStatusGet.
        /// </summary>
        HIPS_MESSAGE_038 = 38,

        /// <summary>
        /// AddressTypeDl.GetAll: Database error while calling AddressTypeGet.
        /// </summary>
        HIPS_MESSAGE_039 = 39,

        /// <summary>
        /// AddressTypeDl.Get: Database error while calling AddressTypeGet by ID.
        /// </summary>
        HIPS_MESSAGE_040 = 40,

        /// <summary>
        /// CountryDl.GetAll: Database error while calling CountryGet.
        /// </summary>
        HIPS_MESSAGE_041 = 41,

        /// <summary>
        /// CountryDl.Get: Database error while calling CountryGet by ID.
        /// </summary>
        HIPS_MESSAGE_042 = 42,

        /// <summary>
        /// HealthProviderIndividualDl.GetAll: Database error while calling HealthProviderIndividualGet.
        /// </summary>
        HIPS_MESSAGE_043 = 43,

        /// <summary>
        /// HealthProviderIndividualDl.GetByPasIdentifier: Database error while calling HealthProviderIndividualGet by PAS identifier.
        /// </summary>
        HIPS_MESSAGE_044 = 44,

        /// <summary>
        /// HealthProviderIndividualDl.Get: Database error while calling HealthProviderIndividualGet by ID.
        /// </summary>
        HIPS_MESSAGE_045 = 45,

        /// <summary>
        /// PatientMasterIhirDl.Get: Database error while calling PatientMasterIhiGet by ID.
        /// </summary>
        HIPS_MESSAGE_046 = 46,

        /// <summary>
        /// PatientMasterIhirDl.GetAll: Database error while calling PatientMasterIhiGet by IhiStatusId.
        /// </summary>
        HIPS_MESSAGE_047 = 47,

        /// <summary>
        /// HealthProviderOrganisationPatientDl.Get: Database error while calling HealthProviderOrganisationPatientGet by HPI-O and PatientMasterId.
        /// </summary>
        HIPS_MESSAGE_048 = 48,

        /// <summary>
        /// HealthProviderOrganisationPatientDl.GetAll: Database error while calling HealthProviderOrganisationPatientGet by DateModified.
        /// </summary>
        HIPS_MESSAGE_049 = 49,

        /// <summary>
        /// HospitalPatientDl.GetAll: Database error while calling HospitalPatientGet by HospitalId and PatientMasterId.
        /// </summary>
        HIPS_MESSAGE_050 = 50,

        /// <summary>
        /// EpisodeDl.GetAll: Database error while calling EpisodeGet by PatientId and DateLastUpdated.
        /// </summary>
        HIPS_MESSAGE_051 = 51,

        /// <summary>
        /// PcehrMessageQueueDl.GetAll: Database error while calling PcehrMessageQueueGet by EpisodeId.
        /// </summary>
        HIPS_MESSAGE_052 = 52,

        /// <summary>
        /// PcehrMessageQueueDl.Get: Database error while calling PcehrMessageQueueGet by ID.
        /// </summary>
        HIPS_MESSAGE_053 = 53,

        /// <summary>
        /// PatientMasterDl.GetByHospitalCodeMrn: Database error while calling PatientMasterGet by MRN and CodeSystemId and HospitalCode.
        /// </summary>
        HIPS_MESSAGE_054 = 54,

        /// <summary>
        /// PatientMasterDl.GetByHospitalIdMrn: Database error while calling PatientMasterGet by HospitalId and MRN.
        /// </summary>
        HIPS_MESSAGE_055 = 55,

        /// <summary>
        /// PatientMasterDl.GetByDemographics: Database error while calling PatientMasterGet by Demographics.
        /// </summary>
        HIPS_MESSAGE_056 = 56,

        /// <summary>
        /// PatientMasterDl.GetByIhi: Database error while calling PatientMasterGet by IHI.
        /// </summary>
        HIPS_MESSAGE_057 = 57,

        /// <summary>
        /// PatientMasterDl.GetByStatePatientId: Database error while calling PatientMasterGet by StatePatientId.
        /// </summary>
        HIPS_MESSAGE_058 = 58,

        /// <summary>
        /// PatientMasterDl.Get: Database error while calling PatientMasterGet by ID.
        /// </summary>
        HIPS_MESSAGE_059 = 59,

        /// <summary>
        /// PatientMasterDl.Insert: Database error while calling PatientMasterInsert.
        /// </summary>
        HIPS_MESSAGE_060 = 60,

        /// <summary>
        /// PatientMasterDl.Update: Database error while calling PatientMasterUpdate.
        /// </summary>
        HIPS_MESSAGE_061 = 61,

        /// <summary>
        /// PatientMasterDl.Merge: Database error while calling PatientMasterMerge.
        /// </summary>
        HIPS_MESSAGE_062 = 62,

        /// <summary>
        /// HospitalDl.GetAll: Database error while calling HospitalGet.
        /// </summary>
        HIPS_MESSAGE_063 = 63,

        /// <summary>
        /// HospitalDl.Get: Database error while calling HospitalGet by HospitalId.
        /// </summary>
        HIPS_MESSAGE_064 = 64,

        /// <summary>
        /// HospitalDl.Get: Database error while calling HospitalGet by CodeSystemId and Code.
        /// </summary>
        HIPS_MESSAGE_065 = 65,

        /// <summary>
        /// PatientAccess.ReadCreateOrUpdatePatient: Unable to insert patient stub for ValidatedIhi parameter.
        /// </summary>
        HIPS_MESSAGE_066 = 66,

        /// <summary>
        /// DocumentQueueTransactionHandler.SaveRemoveOperationAndPlaceOnQueue: Remove rejected as unable to add to MSMQ queue.
        /// </summary>
        HIPS_MESSAGE_067 = 67,

        /// <summary>
        /// DocumentQueueTransactionHandler.SaveRemoveOperationAndPlaceOnQueue: Remove rejected as unable to insert pending operation in database.
        /// </summary>
        HIPS_MESSAGE_068 = 68,

        /// <summary>
        /// DocumentQueueTransactionHandler.SaveUploadOperationAndPlaceOnQueue: Upload rejected as unable to add to MSMQ queue.
        /// </summary>
        HIPS_MESSAGE_069 = 69,

        /// <summary>
        /// DocumentQueueTransactionHandler.SaveUploadOperationAndPlaceOnQueue: Upload rejected as unable to insert pending operation in database.
        /// </summary>
        HIPS_MESSAGE_070 = 70,

        /// <summary>
        /// DocumentQueueTransactionHandler.SaveAfterSuccessfulUpload: Upload retried as database access failed after successful upload.
        /// </summary>
        HIPS_MESSAGE_071 = 71,

        /// <summary>
        /// DocumentQueueTransactionHandler.SaveAfterSuccessfulRemove: Remove retried as database access failed after successful remove.
        /// </summary>
        HIPS_MESSAGE_072 = 72,

        /// <summary>
        /// DocumentQueueTransactionHandler.RestorePackageInQueuedUploadOperation: Upload failed as database access failed while reading pending operation.
        /// </summary>
        HIPS_MESSAGE_073 = 73,

        /// <summary>
        /// PatientIhiCleanup.Cleanup: Background process starting.
        /// </summary>
        HIPS_MESSAGE_074 = 74,

        /// <summary>
        /// PatientIhiCleanup.Cleanup: Background process encountered error.
        /// </summary>
        HIPS_MESSAGE_075 = 75,

        /// <summary>
        /// PatientIhiCleanup.Cleanup: Background process finished.
        /// </summary>
        HIPS_MESSAGE_076 = 76,

        /// <summary>
        /// PatientIhiCleanup.ReachedThresholdOnConsecutiveUnavailable: Background process encountered service unavailable errors 20 times in a row.
        /// </summary>
        HIPS_MESSAGE_077 = 77,

        /// <summary>
        /// DocumentRemoval.Remove: Remove aborted because the operation status was changed to Failed.
        /// </summary>
        HIPS_MESSAGE_078 = 78,

        /// <summary>
        /// DocumentRemoval.Remove: Remove retried because IHI is stale and Medicare service is unavailable.
        /// </summary>
        HIPS_MESSAGE_079 = 79,

        /// <summary>
        /// PatientIhiValidation.GetValidatedIhi: Returning stale IHI from GetValidatedIhi.
        /// </summary>
        HIPS_MESSAGE_080 = 80,

        /// <summary>
        /// PatientIhiValidation.GetValidatedIhi: Exception caught in GetValidatedIhi.
        /// </summary>
        HIPS_MESSAGE_081 = 81,

        /// <summary>
        /// PatientIhiValidation.GetValidatedIhi: Invalid user details in GetValidatedIhi.
        /// </summary>
        HIPS_MESSAGE_082 = 82,

        /// <summary>
        /// PatientIhiValidation.GetValidatedIhi: Unable to find patient in GetValidatedIhi.
        /// </summary>
        HIPS_MESSAGE_083 = 83,

        /// <summary>
        /// PatientIhiValidation.GetValidatedIhi: Date of birth mismatch in GetValidatedIhi.
        /// </summary>
        HIPS_MESSAGE_084 = 84,

        /// <summary>
        /// DocumentRemovalBeforeQueue.LoadPatientEpisode: Remove added to queue with a stale IHI.
        /// </summary>
        HIPS_MESSAGE_085 = 85,

        /// <summary>
        /// DocumentUpload.PerformUpload: Upload aborted because the operation status was changed to Failed.
        /// </summary>
        HIPS_MESSAGE_086 = 86,

        /// <summary>
        /// DocumentUpload.PerformUpload: Upload retried because IHI is stale and Medicare service is unavailable.
        /// </summary>
        HIPS_MESSAGE_087 = 87,

        /// <summary>
        /// DemographicCheck.ValidateDemographics: Demographic mismatch in downloaded document.
        /// </summary>
        HIPS_MESSAGE_088 = 88,

        /// <summary>
        /// DoesPcehrExist.PcehrExists: Internal HIPS Error while trying to invoke the DoesPCEHRExist service.
        /// </summary>
        HIPS_MESSAGE_089 = 89,

        /// <summary>
        /// DoesPcehrExist.InvokeDoesPcehrExist: Patient will be reprocessed because DoesPCEHRExist returned a SOAP fault.
        /// </summary>
        HIPS_MESSAGE_090 = 90,

        /// <summary>
        /// DoesPcehrExist.InvokeDoesPcehrExist: Patient will be reprocessed because of an error connecting to DoesPCEHRExist service.
        /// </summary>
        HIPS_MESSAGE_091 = 91,

        /// <summary>
        /// DocumentUploadBeforeQueue.UploadOrSupersedeDocument: Upload added to queue with a stale IHI.
        /// </summary>
        HIPS_MESSAGE_092 = 92,

        /// <summary>
        /// DocumentUploadBeforeQueue.UploadOrSupersedeDocument: Upload rejected as patient withdrew consent.
        /// </summary>
        HIPS_MESSAGE_093 = 93,

        /// <summary>
        /// DocumentUploadBeforeQueue.UploadOrSupersedeDocument: Upload rejected as patient under age.
        /// </summary>
        HIPS_MESSAGE_094 = 94,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateInput: Upload rejected as no CDA document received.
        /// </summary>
        HIPS_MESSAGE_095 = 95,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateInput: Upload rejected as unable to retrieve hospital.
        /// </summary>
        HIPS_MESSAGE_096 = 96,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateInput: Upload rejected as unable to retrieve patient.
        /// </summary>
        HIPS_MESSAGE_097 = 97,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateInput: Upload rejected as unable to retrieve episode.
        /// </summary>
        HIPS_MESSAGE_098 = 98,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateInput: Upload rejected as unable to find document format code.
        /// </summary>
        HIPS_MESSAGE_099 = 99,

        /// <summary>
        /// DocumentUploadBeforeQueue.PackageDocument: Upload rejected due to CDA packaging exception.
        /// </summary>
        HIPS_MESSAGE_100 = 100,

        /// <summary>
        /// DocumentUploadBeforeQueue.PackageDocument: Upload rejected due to other exception in CDA packaging.
        /// </summary>
        HIPS_MESSAGE_101 = 101,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateAttachments: Upload rejected as attachment too large.
        /// </summary>
        HIPS_MESSAGE_102 = 102,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateAttachments: Upload rejected as attachment filename invalid.
        /// </summary>
        HIPS_MESSAGE_103 = 103,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateAttachments: Upload rejected as attachment type disallowed.
        /// </summary>
        HIPS_MESSAGE_104 = 104,

        /// <summary>
        /// DocumentRemovalInvoker.HandlePcehrResponse: Remove retried because PCEHR system temporarily unavailable.
        /// </summary>
        HIPS_MESSAGE_105 = 105,

        /// <summary>
        /// DocumentUploadInvoker.HandlePcehrResponse: Upload retried because PCEHR system temporarily unavailable.
        /// </summary>
        HIPS_MESSAGE_106 = 106,

        /// <summary>
        /// MessageLoader.Process: The HL7 message was processed, but a known data quality issue was found while attempting to search for the IHI..
        /// </summary>
        HIPS_MESSAGE_107 = 107,

        /// <summary>
        /// MessageLoader.Process: The HL7 message was processed but a serious failure occurred when attempting to search for the IHI..
        /// </summary>
        HIPS_MESSAGE_108 = 108,

        /// <summary>
        /// MessageLoader.Process: This message was not processed because it contains a known data quality issue..
        /// </summary>
        HIPS_MESSAGE_109 = 109,

        /// <summary>
        /// MessageLoader.Process: HIPS failed while processing this HL7 message. The message should be retried..
        /// </summary>
        HIPS_MESSAGE_110 = 110,

        /// <summary>
        /// MessageLoader.Process: An unexpected exception occurred while HIPS was processing this HL7 message. The message should be retried..
        /// </summary>
        HIPS_MESSAGE_111 = 111,

        /// <summary>
        /// PatientLoader.PopulateSecondaryDemographics: Flagged the patient record as having an invalid date of death.
        /// </summary>
        HIPS_MESSAGE_112 = 112,

        /// <summary>
        /// CheckPatientParticipation.GetRecentPatientParticipationStatus: Client system checked patient participation.
        /// </summary>
        HIPS_MESSAGE_113 = 113,

        /// <summary>
        /// DatabaseLoaderService.NotifyPasEvent: Caught HL7ParseException while parsing HL7 message.
        /// </summary>
        HIPS_MESSAGE_114 = 114,

        /// <summary>
        /// DatabaseLoaderService.NotifyPasEvent: Caught other exception while parsing HL7 message.
        /// </summary>
        HIPS_MESSAGE_115 = 115,

        /// <summary>
        /// IhiCleanUpServiceHost.InitializeRuntime: Background thread timer started.
        /// </summary>
        HIPS_MESSAGE_116 = 116,

        /// <summary>
        /// IhiCleanUpServiceHost.RunThread: Background thread timer interval set.
        /// </summary>
        HIPS_MESSAGE_117 = 117,

        /// <summary>
        /// IhiCleanUpServiceHost.IhiCleanUpServiceHost_IhiCleanUpCallback: Background thread timer reset exception.
        /// </summary>
        HIPS_MESSAGE_118 = 118,

        /// <summary>
        /// IhiCleanUpServiceHost.IhiCleanUpServiceHost_IhiCleanUpCallback: Background thread timer stopped.
        /// </summary>
        HIPS_MESSAGE_119 = 119,

        /// <summary>
        /// ReferenceService.ReloadReferenceData: Reference data was reloaded.
        /// </summary>
        HIPS_MESSAGE_120 = 120,

        /// <summary>
        /// ReferenceService.ReloadReferenceData: Unable to reload reference data.
        /// </summary>
        HIPS_MESSAGE_121 = 121,

        /// <summary>
        /// PatientMasterIhiDl.Update: Database error while calling PatientMasterIhiUpdate.
        /// </summary>
        HIPS_MESSAGE_122 = 122,

        /// <summary>
        /// PatientMasterIhiDl.GetTimesIhiUsed: Database error while calling IhiUsageCount.
        /// </summary>
        HIPS_MESSAGE_123 = 123,

        /// <summary>
        /// PatientMasterIhiDl.GetTimesIhiUsed: Duplicate IHI check not performed.
        /// </summary>
        HIPS_MESSAGE_124 = 124,

        /// <summary>
        /// PatientMasterIhiDl.GetPossibleDuplicatePatients: Database error while calling IhiDuplicatePatientCount.
        /// </summary>
        HIPS_MESSAGE_125 = 125,

        /// <summary>
        /// HospitalPatientDl.GetAllActive: Database error while calling HospitalPatientGet for active MRNs for PatientMasterId.
        /// </summary>
        HIPS_MESSAGE_126 = 126,

        /// <summary>
        /// HospitalPatientDl.GetActive: Database error while calling HospitalPatientGet for active MRNs for PatientMasterId and HospitalId.
        /// </summary>
        HIPS_MESSAGE_127 = 127,

        /// <summary>
        /// HospitalPatientDl.Get: Database error while calling HospitalPatientGet by ID.
        /// </summary>
        HIPS_MESSAGE_128 = 128,

        /// <summary>
        /// HospitalPatientDl.Get: Database error while calling HospitalPatientGet by HospitalId and MRN.
        /// </summary>
        HIPS_MESSAGE_129 = 129,

        /// <summary>
        /// HospitalPatientDl.Move: Database error while calling HospitalPatientMove.
        /// </summary>
        HIPS_MESSAGE_130 = 130,

        /// <summary>
        /// HL7MessageLogDl.Get: Database error while calling HL7MessageLogGet by header fields.
        /// </summary>
        HIPS_MESSAGE_131 = 131,

        /// <summary>
        /// ContactDl.GetAllByHospital: Database error while getting contacts for hospital.
        /// </summary>
        HIPS_MESSAGE_132 = 132,

        /// <summary>
        /// HospitalAddressDl.GetAll: Database error while getting addresses for hospital.
        /// </summary>
        HIPS_MESSAGE_133 = 133,

        /// <summary>
        /// DocumentUploadBeforeQueue.ValidateInput: Unable to extract information from CDA document.
        /// </summary>
        HIPS_MESSAGE_134 = 134,

        /// <summary>
        /// PcehrMessageQueueDl.CountEarlierPendingMessages: Database error while calling CountEarlierPendingMessages by SourceSystemSetId and CreateDateTime.
        /// </summary>
        HIPS_MESSAGE_135 = 135,

        /// <summary>
        /// DocumentUpload.PerformUpload: PcehrQueueMessage Upload attempted an out of sequential order message queue process.
        /// </summary>
        HIPS_MESSAGE_136 = 136,

        /// <summary>
        /// DocumentRemoval.Remove: PcehrQueueMessage Remove attempted an out of sequential order message queue process.
        /// </summary>
        HIPS_MESSAGE_137 = 137,

        /// <summary>
        /// HealthProviderOrganisationDl.GetAll: Database error while calling HealthProviderOrganisationGetAll.
        /// </summary>
        HIPS_MESSAGE_138 = 138,

        /// <summary>
        /// HealthProviderOrganisationDl.GetAll: Database error while calling HealthProviderOrganisationGet by ID.
        /// </summary>
        HIPS_MESSAGE_139 = 139,

        /// <summary>
        /// StreetSuffixTypeCodeDl.GetAll: Database error while calling StreetSuffixTypeCodeGet .
        /// </summary>
        HIPS_MESSAGE_140 = 140,

        /// <summary>
        /// StreetSuffixTypeCodeDl.Get: Database error while calling StreetSuffixTypeCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_141 = 141,

        /// <summary>
        /// StreetTypeCodeDl.GetAll: Database error while calling StreetTypeCodeGet .
        /// </summary>
        HIPS_MESSAGE_142 = 142,

        /// <summary>
        /// StreetTypeCodeDl.Get: Database error while calling StreetTypeCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_143 = 143,

        /// <summary>
        /// UnitTypeCodeDl.GetAll: Database error while calling UnitTypeCodeGet .
        /// </summary>
        HIPS_MESSAGE_144 = 144,

        /// <summary>
        /// UnitTypeCodeDl.Get: Database error while calling UnitTypeCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_145 = 145,

        /// <summary>
        /// LevelTypeCodeDl.GetAll: Database error while calling LevelTypeCodeGet .
        /// </summary>
        HIPS_MESSAGE_146 = 146,

        /// <summary>
        /// LevelTypeCodeDl.Get: Database error while calling LevelTypeCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_147 = 147,

        /// <summary>
        /// PostalDeliveryTypeCodeDl.GetAll: Database error while calling PostalDeliveryTypeCodeGet .
        /// </summary>
        HIPS_MESSAGE_148 = 148,

        /// <summary>
        /// PostalDeliveryTypeCodeDl.Get: Database error while calling PostalDeliveryTypeCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_149 = 149,

        /// <summary>
        /// StateCodeDl.GetAll: Database error while calling StateCodeGet .
        /// </summary>
        HIPS_MESSAGE_150 = 150,

        /// <summary>
        /// StateCodeDl.Get: Database error while calling StateCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_151 = 151,

        /// <summary>
        /// CountryCodeDl.GetAll: Database error while calling CountryCodeGet .
        /// </summary>
        HIPS_MESSAGE_152 = 152,

        /// <summary>
        /// CountryCodeDl.Get: Database error while calling CountryCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_153 = 153,

        /// <summary>
        /// SexCodeDl.GetAll: Database error while calling SexCodeGet .
        /// </summary>
        HIPS_MESSAGE_154 = 154,

        /// <summary>
        /// SexCodeDl.Get: Database error while calling SexCodeGet By ID.
        /// </summary>
        HIPS_MESSAGE_155 = 155,

        /// <summary>
        /// PatientListDl.ListPatientsInHospital: Database error while calling PatientInHospitalGet.
        /// </summary>
        HIPS_MESSAGE_156 = 156,

        /// <summary>
        /// EpisodeListDl.ListPatientInHospitalEpisodes: Database error while calling PatientInHospitalEpisodesGet.
        /// </summary>
        HIPS_MESSAGE_157 = 157,

        /// <summary>
        /// EpisodeDl.GetEpisodePatientExtendedDetails: Database error while calling EpisodePatientExtendedDetailsGet.
        /// </summary>
        HIPS_MESSAGE_158 = 158,

        /// <summary>
        /// EpisodeDl.GetEpisodePatientExtendedDetails: Database error while calling SinglePatientInHospitalGet.
        /// </summary>
        HIPS_MESSAGE_159 = 159,

        /// <summary>
        /// ClinicalDocumentDl.ListLocalUploadedDocuments: Database error while calling ClinicalDocumentUploadList.
        /// </summary>
        HIPS_MESSAGE_160 = 160,

        /// <summary>
        /// ClinicalDocumentDl.GetLocalUploadedDocument: Database error while calling ClinicalDocumentVersionLocalGet.
        /// </summary>
        HIPS_MESSAGE_161 = 161,

        /// <summary>
        /// CdaSetNumberDl.GetByCdaSetNumberId: Database error while calling CdaSetNumberGet by ID.
        /// </summary>
        HIPS_MESSAGE_162 = 162,

        /// <summary>
        /// CdaSetNumberDl.GetByEpisodeAndDocumentType: Database error while calling CdaSetNumberGet by Episode and Document Type.
        /// </summary>
        HIPS_MESSAGE_163 = 163,

        /// <summary>
        /// CdaService.PackageCdaDocument: Exception while packaging CDA document.
        /// </summary>
        HIPS_MESSAGE_164 = 164,

        /// <summary>
        /// CdaService.UnpackageCdaDocument: InvalidCertificateException while unpackaging CDA document.
        /// </summary>
        HIPS_MESSAGE_165 = 165,

        /// <summary>
        /// CdaService.UnpackageCdaDocument: CdaPackagingException while unpackaging CDA document.
        /// </summary>
        HIPS_MESSAGE_166 = 166,

        /// <summary>
        /// CdaService.UnpackageCdaDocument: Exception while unpackaging CDA document.
        /// </summary>
        HIPS_MESSAGE_167 = 167,

        /// <summary>
        /// CdaService.GetCdaContent: Exception while getting CDA document content..
        /// </summary>
        HIPS_MESSAGE_168 = 168,

        /// <summary>
        /// CdaService.CreateDischargeSummaryLevel1A: Error creating Discharge Summary Level 1A.
        /// </summary>
        HIPS_MESSAGE_169 = 169,

        /// <summary>
        /// PCEHRService.UploadOrSupersedeDischargeSummaryLevel1A: Error creating and uploading Discharge Summary Level 1A.
        /// </summary>
        HIPS_MESSAGE_170 = 170,

        /// <summary>
        /// PendingIhiPcehrLookupDl.GetAll: Error getting the pending IHI and PCEHR lookups.
        /// </summary>
        HIPS_MESSAGE_171 = 171,

        /// <summary>
        /// PatientAccess.GetPatient: Patient is registered in hospital with a different patientmasterid.
        /// </summary>
        HIPS_MESSAGE_172 = 172,

        /// <summary>
        /// PatientAccess.GetPatient: Patient is already registered in hospital with a different MRN.
        /// </summary>
        HIPS_MESSAGE_173 = 173,

        /// <summary>
        /// PatientAccess.GetPatient: Database access failed when creating patient record.
        /// </summary>
        HIPS_MESSAGE_174 = 174,

        /// <summary>
        /// PcehrServiceV2.RefreshParticipationStatus: Error refreshing and retrieving the patient participation status.
        /// </summary>
        HIPS_MESSAGE_175 = 175,

        /// <summary>
        /// HpiiService.HpiiBatchRetrieve: Error retrieving the results of the HPI-I search batch.
        /// </summary>
        HIPS_MESSAGE_176 = 176,

        /// <summary>
        /// HpiiService.HpiiBatchSubmit: Error submitting the HPI-I search batch.
        /// </summary>
        HIPS_MESSAGE_177 = 177,

        /// <summary>
        /// HpiiService.HpiiDemographicSearch: Error performing the HPI-I demographic search.
        /// </summary>
        HIPS_MESSAGE_178 = 178,

        /// <summary>
        /// HpiiService.HpiiValidation: Error performing the HPI-I validation search.
        /// </summary>
        HIPS_MESSAGE_179 = 179,

        /// <summary>
        /// PcehrServiceV2.IsPcehrAdvertisedLocal: Error retrieving the is PCEHR advertised status.
        /// </summary>
        HIPS_MESSAGE_180 = 180,
        
        /// <summary>
        /// HospitalPatientDl.GetByMrn: Database error while calling HospitalPatientByMrnGet.
        /// </summary>
        HIPS_MESSAGE_181 = 181,

        /// <summary>
        /// EpisodeDl.GetAllEpisodeByFillerOrderNumber: Database error while calling EpisodeByFillerOrderNumberGet by Order Identifier.
        /// </summary>
        HIPS_MESSAGE_182 = 182,

        /// <summary>
        /// HealthProviderOrganisationDl.GetByHpio: Database error while calling HealthProviderOrganisationByHpioGet by Hpio.
        /// </summary>
        HIPS_MESSAGE_183 = 183,

        /// <summary>
        /// HealthProviderIndividualHpiiDl.Get: Database error while calling HealthProviderIndividualHpiiGet by Health Provider Individual Id.
        /// </summary>
        HIPS_MESSAGE_184 = 184,

        /// <summary>
        /// HospitalHealthProviderIndividualDl.Get: Database error while calling HospitalHealthProviderIndividualGet by Hospital Id and Identifier.
        /// </summary>
        HIPS_MESSAGE_185 = 185,

        /// <summary>
        /// HealthProviderIndividualDl.Get: Database error while calling HealthProviderIndividualGet by Local Provider Code or Iam Provider Identifier.
        /// </summary>
        HIPS_MESSAGE_186 = 186,

        /// <summary>
        /// AckQueue.SendAck: Logging acknowledgement.
        /// </summary>
        HIPS_MESSAGE_187 = 187,

        /// <summary>
        /// PDILoader.GetParticipatingProvider: HPI-I Validation Warning.
        /// </summary>
        HIPS_MESSAGE_188 = 188,

        /// <summary>
        /// PDILoader.GetParticipatingProvider: HPI-I Validation Alert.
        /// </summary>
        HIPS_MESSAGE_189 = 189,

        /// <summary>
        /// IHIService.ValidateReceivingPatient: Error validating the patient IHI for a received message.
        /// </summary>
        HIPS_MESSAGE_190 = 190,

        /// <summary>
        /// IHIService.ValidateSendingPatient: Error validating the patient IHI for a message to be sent.
        /// </summary>
        HIPS_MESSAGE_191 = 191,

        /// <summary>
        /// HpiiServiceV2.ListLocalProviders: Error retrieving local providers list.
        /// </summary>
        HIPS_MESSAGE_192 = 192,

        /// <summary>
        /// HpiiServiceV2.AddEditLocalProvider: Error adding or editing local provider.
        /// </summary>
        HIPS_MESSAGE_193 = 193,

        /// <summary>
        /// HpiiServiceV2.AddReplaceLocalProviderIdentifier: Error adding or replacing local provider identifier.
        /// </summary>
        HIPS_MESSAGE_194 = 194,

        /// <summary>
        /// HpiiServiceV2.DeactivateLocalProvider: Error deactivating local provider.
        /// </summary>
        HIPS_MESSAGE_195 = 195,

        /// <summary>
        /// ComplicationDl.GetAll: Database error while getting complication.
        /// </summary>
        PUMA_MESSAGE_1001 = 1001,

        /// <summary>
        /// ConsentAuditDl.GetCurrentConsentAudit: Database error while getting consent audit.
        /// </summary>
        PUMA_MESSAGE_1002 = 1002,

        /// <summary>
        /// DiagnosisDl.GetAll: Database error while getting diagnoses.
        /// </summary>
        PUMA_MESSAGE_1003 = 1003,

        /// <summary>
        /// DiagnosisDl.Get: Database error while getting diagnosis.
        /// </summary>
        PUMA_MESSAGE_1004 = 1004,

        /// <summary>
        /// DisclosureAuditDl.GetCurrentDisclosureAudit: Database error while getting disclosure audit.
        /// </summary>
        PUMA_MESSAGE_1005 = 1005,

        /// <summary>
        /// EpisodeDl.Get: Database error while getting episode.
        /// </summary>
        PUMA_MESSAGE_1006 = 1006,

        /// <summary>
        /// EpisodeDl.GetEpisodesWithUploadedDocuments: Database error while getting episodes with uploaded documents.
        /// </summary>
        PUMA_MESSAGE_1007 = 1007,

        /// <summary>
        /// EpisodeDl.GetEpisodesWithUploadedDocumentCount: Database error while getting count of episodes with uploaded documents.
        /// </summary>
        PUMA_MESSAGE_1008 = 1008,

        /// <summary>
        /// EpisodeDl.GetEpisodes: Database error while getting episode.
        /// </summary>
        PUMA_MESSAGE_1009 = 1009,

        /// <summary>
        /// EpisodeDl.GetEpisodesCount: Database error while getting count of episodes.
        /// </summary>
        PUMA_MESSAGE_1010 = 1010,

        /// <summary>
        /// EpisodeDl.Update: Database error while updating episode.
        /// </summary>
        PUMA_MESSAGE_1011 = 1011,

        /// <summary>
        /// EpisodeIhiCheckListDl.GetAll: Database error while getting episode IHI checklist.
        /// </summary>
        PUMA_MESSAGE_1012 = 1012,

        /// <summary>
        /// HelpPageDl.List: Database error while getting help page.
        /// </summary>
        PUMA_MESSAGE_1013 = 1013,

        /// <summary>
        /// HospitalDl.GetActive: Database error while getting hospital.
        /// </summary>
        PUMA_MESSAGE_1014 = 1014,

        /// <summary>
        /// HospitalWardDl.GetAll: Database error while getting hospital ward.
        /// </summary>
        PUMA_MESSAGE_1015 = 1015,

        /// <summary>
        /// MenuDl.GetAll: Database error while getting menu.
        /// </summary>
        PUMA_MESSAGE_1016 = 1016,

        /// <summary>
        /// PatientDl.Get: Database error while getting patient.
        /// </summary>
        PUMA_MESSAGE_1017 = 1017,

        /// <summary>
        /// PatientDl.PatientsWithoutIhi: Database error while getting patients without IHI.
        /// </summary>
        PUMA_MESSAGE_1018 = 1018,

        /// <summary>
        /// PatientDl.PatientsWithIhi: Database error while getting patients with IHI.
        /// </summary>
        PUMA_MESSAGE_1019 = 1019,

        /// <summary>
        /// PatientDl.PatientsWithOverridesReport: Database error while getting patients with overrides.
        /// </summary>
        PUMA_MESSAGE_1020 = 1020,

        /// <summary>
        /// PatientMasterDl.Get: Database error while getting patient master.
        /// </summary>
        PUMA_MESSAGE_1021 = 1021,

        /// <summary>
        /// PatientMasterDl.Insert: Database error while inserting patient master.
        /// </summary>
        PUMA_MESSAGE_1022 = 1022,

        /// <summary>
        /// PatientMasterDl.Update: Database error while updating patient master.
        /// </summary>
        PUMA_MESSAGE_1023 = 1023,

        /// <summary>
        /// PatientTileDl.GetAll: Database error while getting patient tiles.
        /// </summary>
        PUMA_MESSAGE_1024 = 1024,

        /// <summary>
        /// PatientTileDl.GetByMrn: Database error while getting patient tile.
        /// </summary>
        PUMA_MESSAGE_1025 = 1025,

        /// <summary>
        /// RemoveAuditDl.GetCurrentRemoveAudit: Database error while getting remove audit.
        /// </summary>
        PUMA_MESSAGE_1026 = 1026,

        /// <summary>
        /// SecurityDl.GetUserPermission: Database error while getting user permission.
        /// </summary>
        PUMA_MESSAGE_1027 = 1027,

        /// <summary>
        /// SecurityDl.CreateUserPermission: Database error while creating user permission.
        /// </summary>
        PUMA_MESSAGE_1028 = 1028,

        /// <summary>
        /// SecurityDl.GetHadGroups: Error while getting HAD groups with Local security.
        /// </summary>
        PUMA_MESSAGE_1029 = 1029,

        /// <summary>
        /// SecurityDl.GetHadGroups: Error while getting HAD groups with Active Directory security.
        /// </summary>
        PUMA_MESSAGE_1030 = 1030,

        /// <summary>
        /// SeparationLocationDl.GetAll: Database error while getting separation location.
        /// </summary>
        PUMA_MESSAGE_1031 = 1031,

        /// <summary>
        /// SeparationLocationDl.Get: Database error while getting separation location.
        /// </summary>
        PUMA_MESSAGE_1032 = 1032,

        /// <summary>
        /// SpecialityDl.GetAll: Database error while getting speciality.
        /// </summary>
        PUMA_MESSAGE_1033 = 1033,

        /// <summary>
        /// SpecialityDl.Get: Database error while getting speciality.
        /// </summary>
        PUMA_MESSAGE_1034 = 1034,
    }

    /// <summary>
    /// This enumeration is no longer used in HIPS.
    /// </summary>
    [Serializable]
    public enum Option
    {
        /// <summary>
        /// Information.
        /// </summary>
        Information,

        /// <summary>
        /// Episodes.
        /// </summary>
        Episodes,

        /// <summary>
        /// Pathology.
        /// </summary>
        Pathology,

        /// <summary>
        /// Radiology.
        /// </summary>
        Radiology,

        /// <summary>
        /// Discharge.
        /// </summary>
        Discharge,

        /// <summary>
        /// PCEHR.
        /// </summary>
        Pcehr
    }

    /// <summary>
    /// This is used to match the QueueOperation table to code.
    /// </summary>
    [Serializable]
    public enum QueueOperation
    {
        /// <summary>
        /// The queued operation is to upload a document to a PCEHR, possibly superseding an older version of the same document.
        /// </summary>
        UploadOrSupersede = 1,

        /// <summary>
        /// The queued operation is to remove a document from a PCEHR.
        /// </summary>
        Remove = 2
    }

    /// <summary>
    /// This is used to match the QueueStatus table to code.
    /// </summary>
    [Serializable]
    public enum QueueStatus
    {
        /// <summary>
        /// The queued operation is currently in progress or waiting to be processed.
        /// </summary>
        Pending = 1,

        /// <summary>
        /// The queued operation was successful. This status is not typically found
        /// because the default behaviour is to remove queued items after successful
        /// processing.
        /// </summary>
        Success = 2,

        /// <summary>
        /// The queued operation was not successful. The system administrator should
        /// review and correct the reason for failure and resubmit the request.
        /// </summary>
        Failure = 3
    }

    /// <summary>
    /// This enum is used to match the removal reason id in the RemovalReason table to code.
    /// The values of the IDs should also be kept in sync with the NEHTA enum
    /// Nehta.VendorLibrary.PCEHR.RemoveDocument.removeDocumentReasonForRemoval.
    /// Note: The NEHTA enum has an extra item "ElectToRemove", which is omitted from this
    /// enum because that is defined for consumer use only.
    /// </summary>
    [Serializable]
    public enum RemovalReason
    {
        /// <summary>
        /// The provider organisation withdraws the document from the PCEHR.
        /// Withdrawn documents will be reinstated if a new version of the
        /// document is uploaded.
        /// </summary>
        Withdrawn = 0,

        /// <summary>
        /// The document is removed because it relates to the wrong consumer.
        /// Documents removed with this reason cannot be reinstated by the
        /// provider or the consumer.
        /// </summary>
        IncorrectIdentity = 2
    }

    /// <summary>
    /// This enum is used to match the RepositoryId in the DocumentType table to code.
    /// </summary>
    [Serializable]
    public enum Repository
    {
        /// <summary>
        /// The main PCEHR repository.
        /// </summary>
        PCEHR = 1,

        /// <summary>
        /// The National Prescription and Dispense Repository.
        /// </summary>
        NPDR = 2
    }

    /// <summary>
    /// This enumeration represents a person's sex, and is used to match the Sex table to code.
    /// </summary>
    [Serializable]
    public enum SexEnumerator
    {
        /// <summary>
        /// Not stated or inadequately described.
        /// </summary>
        NotStatedOrInadequatelyDescribed = -1,

        /// <summary>
        /// The male sex.
        /// </summary>
        Male = 1,

        /// <summary>
        /// The female sex.
        /// </summary>
        Female = 2,

        /// <summary>
        /// Intersex or indeterminate.
        /// </summary>
        IntersexOrIndeterminate = 3,
    }
}