﻿using System;
using System.Runtime.Serialization;
using HIPS.Base.Schemas;
using BaseEnumerators = HIPS.Base.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.Schemas
{
    /// <summary>
    /// This class represents a contact method and value.
    /// </summary>
    [KnownType(typeof(Contact))]
    [Serializable]
    [DataContract]
    public class Contact : BaseSchema
    {
        #region Private Members

        private int? hospitalId;
        private string detail;
        private int contactMethodId;
        private string contactMethodDescription;
        private string telecommunicationType;
        private string cdaType;

        #endregion Private Members

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="Contact"/> class.
        /// </summary>
        public Contact()
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="Contact"/> class.
        /// </summary>
        /// <param name="detail">The detail.</param>
        /// <param name="contactMethodId">The contact method id.</param>
        /// <param name="contactMethodDescription">The contact method description.</param>
        public Contact(string detail, int contactMethodId, string contactMethodDescription)
        {
            this.detail = detail;
            this.contactMethodId = contactMethodId;
            this.contactMethodDescription = contactMethodDescription;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets the contact id.
        /// </summary>
        /// <value>
        /// The contact id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.AutoGeneratedKey)]
        public int? ContactId
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }

        /// <summary>
        /// Gets or sets the hospital ID. This will only be valued if this contact belongs to a hospital.
        /// </summary>
        /// <value>
        /// The hospital id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public int? HospitalId
        {
            get
            {
                return hospitalId;
            }
            set
            {
                hospitalId = value;
            }
        }

        /// <summary>
        /// Gets or sets the contact detail.
        /// </summary>
        /// <value>
        /// The detail.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string Detail
        {
            get
            {
                return detail;
            }
            set
            {
                detail = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => Detail);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the contact method id.
        /// </summary>
        /// <value>
        /// The contact method id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int ContactMethodId
        {
            get
            {
                return contactMethodId;
            }
            set
            {
                contactMethodId = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => ContactMethodId);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the contact method description.
        /// </summary>
        /// <value>
        /// The contact method description.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string ContactMethodDescription
        {
            get
            {
                return contactMethodDescription;
            }
            set
            {
                contactMethodDescription = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => ContactMethodDescription);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the type of the telecommunication.
        /// </summary>
        /// <value>
        /// The type of the telecommunication.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string TelecommunicationType
        {
            get
            {
                return telecommunicationType;
            }
            set
            {
                telecommunicationType = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => TelecommunicationType);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the type of the cda.
        /// </summary>
        /// <value>
        /// The type of the cda.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string CdaType
        {
            get
            {
                return cdaType;
            }
            set
            {
                cdaType = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => CdaType);
#endif
            }
        }

        /// <summary>
        /// Gets the compare key, for comparing two contacts to see if they are the same.
        /// </summary>
        public string CompareKey
        {
            get
            {
                string detail = (Detail ?? string.Empty).Trim().ToUpper();
                return string.Format("{0}{1}", detail, ContactMethodId);
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Creates a copy of this Contact.
        /// </summary>
        /// <returns>A copy of this Contact.</returns>
        internal Contact CloneForNewPatientMaster()
        {
            Contact clone = this.MemberwiseClone() as Contact;
            clone.ContactId = null;
            return clone;
        }

        #endregion Methods
    }
}