﻿using System;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using HIPS.CommonBusinessLogic;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrHiBusinessLogic.Ihi;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.GetChangeHistoryView;

using ChangeHistoryViewRegistry = Nehta.VendorLibrary.PCEHR.GetChangeHistoryView;

namespace HIPS.PcehrHiBusinessLogic.Pcehr
{
    public class ChangeHistoryView
    {
        /// <summary>
        /// Return a list of uploaded and changed documents from a single document set
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <param name="documentUniqueId"></param>
        /// <returns>ChangeHistoryViewResponse</returns>
        public ChangeHistoryViewResponse<PatientIdentifierBase> GetChangeHistoryView(PatientIdentifierBase patientIdentifier, UserDetails user, String documentUniqueId)
        {
            ChangeHistoryViewResponse<PatientIdentifierBase> changeHistoryResponse = new ChangeHistoryViewResponse<PatientIdentifierBase>();
            changeHistoryResponse.PatientIdentifier = patientIdentifier;

            PatientAccess dataAccessHelper = new PatientAccess(user);
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;

            // Get the hospital
            HipsResponse status = dataAccessHelper.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                changeHistoryResponse.HipsResponse = status;
                return changeHistoryResponse;
            }

            // Get the patient
            status = dataAccessHelper.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi)
            {
                changeHistoryResponse.HipsResponse = status;
                return changeHistoryResponse;
            }

            // Validate the IHI if stale
            IhiSearchResponse ihiResponse = new PatientIhiValidation().GetValidatedIhi(patientIdentifier, hospital, user, patientMaster);
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                changeHistoryResponse.HipsResponse = ihiResponse.HipsResponse;
                return changeHistoryResponse;
            }

            // Check that we have a valid IHI
            dataAccessHelper.ValidateLocalIhiInformation(patientMaster, changeHistoryResponse.HipsResponse);
            if (changeHistoryResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return changeHistoryResponse;
            }

            // Populate user
            if (!User.PopulateAndValidateUser(hospital, user))
            {
                changeHistoryResponse.HipsResponse.Status = HipsResponseIndicator.InvalidUser;
                return changeHistoryResponse;
            }

            //set IHI found
            changeHistoryResponse.IhiNumber = patientMaster.Ihi;

            X509Certificate2 certificate = Helpers.GetHpioCertificate(hospital);
            Uri uri = Helpers.GetChangeHistoryViewUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(patientIdentifier, patientMaster.Ihi, user, hospital);

            // Instantiate the client
            GetChangeHistoryViewClient changeHistoryViewClient = new GetChangeHistoryViewClient(uri, certificate, certificate);

            try
            {
                System.Reflection.FieldInfo clientField = changeHistoryViewClient.GetType().GetField("changeHistoryClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                GetChangeHistoryViewPortTypeClient ptc = clientField.GetValue(changeHistoryViewClient) as GetChangeHistoryViewPortTypeClient;
                CustomBinding binding = ptc.Endpoint.Binding as CustomBinding;
                HttpsTransportBindingElement https = binding.Elements.Where(a => a.GetType() == typeof(HttpsTransportBindingElement)).First() as HttpsTransportBindingElement;

                //Set the connection timeout for the GetChangeHistoryView service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);

                //If avoiding the proxy then set the UseDefaultWebProxy to false
                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

                getChangeHistoryView request = new getChangeHistoryView();
                request.documentID = documentUniqueId;

                // Invoke the service
                getChangeHistoryViewResponse changeHistoryViewResponse = changeHistoryViewClient.GetChangeHistoryView(header, request);
                getChangeHistoryViewResponseAdhocQueryResponse queryResponse = changeHistoryViewResponse.AdhocQueryResponse;

                //check if there are no errors returned from the service
                if (queryResponse.RegistryErrorList == null)
                {
                    HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);

                    if (queryResponse.RegistryObjectList.ExtrinsicObject != null)
                    {
                        //loop over all Extrinsic Objects and extract data to create DocumentMetaDataItems
                        foreach (ChangeHistoryViewRegistry.ExtrinsicObjectType extrinsicObj in queryResponse.RegistryObjectList.ExtrinsicObject)
                        {
                            string IhiNumber = "";
                            changeHistoryResponse.DocumentList.Add(DocumentHelper.ExtractMetaData(extrinsicObj, out IhiNumber));
                            changeHistoryResponse.IhiNumber = IhiNumber;
                        }
                    }
                    else
                    {
                        // Let the caller know that the list was empty.
                        response.HipsErrorMessage = ResponseStrings.ChangeHistoryListEmpty;
                    }

                    // All has succeeded so add in the HIPSResponseObject
                    changeHistoryResponse.HipsResponse = response;
                }
                else //parse for the errors
                {
                    //fail has occurred so add in the HIPSResponseObject
                    HipsResponse response = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                    String errorCodeContext = queryResponse.RegistryErrorList.RegistryError[0].codeContext;
                    if (!errorCodeContext.IsNullOrEmptyWhitespace())
                    {
                        string[] errorCodeDelimiter = new string[] { " - " };
                        string[] errors = errorCodeContext.Split(errorCodeDelimiter, StringSplitOptions.None);
                        response.ResponseCode = errors[0];
                        response.ResponseCodeDescription = errors[1];
                    }
                    else
                    {
                        response.ResponseCode = "Unknown PCEHR Error Code";
                        response.ResponseCodeDescription = queryResponse.RegistryErrorList.RegistryError[0].errorCode;
                    }
                    changeHistoryResponse.HipsResponse = response;
                }
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.GetChangeHistoryView.StandardErrorType> fe)
            {
                HipsResponse response = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                MessageFault fault = fe.CreateMessageFault();
                if (fault.HasDetail)
                {
                    Nehta.VendorLibrary.PCEHR.GetChangeHistoryView.StandardErrorType error = fault.GetDetail<Nehta.VendorLibrary.PCEHR.GetChangeHistoryView.StandardErrorType>();
                    response.ResponseCode = error.errorCode.ToString();
                    response.ResponseCodeDescription = error.message;
                    response.ResponseCodeDetails = fe.StackTrace;
                }
                else
                {
                    response.ResponseCode = fe.Reason.ToString();
                    response.ResponseCodeDescription = fe.Message;
                    response.ResponseCodeDetails = fe.StackTrace;
                }
                changeHistoryResponse.HipsResponse = response;
            }
            catch (Exception e)
            {
                HipsResponse response = new HipsResponse(HipsResponseIndicator.SystemError);
                response.HipsErrorMessage = e.Message;
                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.ResponseCodeDescription = e.InnerException.Message;
                }
                response.ResponseCodeDetails = e.StackTrace;
                changeHistoryResponse.HipsResponse = response;
            }
            finally
            {
                changeHistoryViewClient.Close();
            }

            Helpers.InsertAudit(patientMaster, user, hospital, AuditOperationNames.GetChangeHistoryView,
                changeHistoryResponse.HipsResponse,
                changeHistoryViewClient.SoapMessages);

            return changeHistoryResponse;
        }

        /// <summary>
        /// Validator for the Service Certificate
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="certificate"></param>
        /// <param name="chain"></param>
        /// <param name="sslPolicyErrors"></param>
        /// <returns></returns>
        private bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return DocumentHelper.ValidateServiceCertificate(sender, certificate, chain, sslPolicyErrors);
        }
    }
}