﻿using System;
using System.Transactions;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrHiBusinessLogic.Ihi;
using HIPS.PcehrQueueLogic;
using HIPS.PcehrSchemas;
using Enums = HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrHiBusinessLogic.Pcehr
{
    /// <summary>
    /// This class contains the business logic to handle a queued operation to mark a clinical document as removed in the PCEHR repository.
    /// </summary>
    public class DocumentRemoval
    {
        /// <summary>
        /// The queued remove operation.
        /// </summary>
        private QueuedRemoveOperation operation;

        /// <summary>
        /// The data access helper for patient and episode records
        /// </summary>
        private PatientAccess patientAccess;

        /// <summary>
        /// Creates an instance of DocumentRemoval to handle a queued remove operation.
        /// </summary>
        /// <param name="operation">The queued remove operation.</param>
        public DocumentRemoval(QueuedRemoveOperation operation)
        {
            this.operation = operation;
            this.patientAccess = new PatientAccess(operation.User);
        }

        /// <summary>
        /// Processes the queued remove operation, which if successful will remove a clinical document from the PCEHR repository.
        /// </summary>
        /// <exception cref="System.InvalidOperationException">When the MSMQ operation should be retried</exception>
        public void Remove()
        {
            DocumentQueueTransactionHandler handler = new DocumentQueueTransactionHandler(operation.User);
            ClinicalDocument document;
            if (!IsRemovalOperationStillPending())
            {
                string message = string.Format(ResponseStrings.OperationAbortedDetail,
                    operation.PendingItem.PcehrMessageQueueId,
                    operation.PendingItem.QueueOperationName,
                    operation.SourceSystemSetId,
                    operation.PatientIdentifier,
                    operation.Hospital.Description);
                EventLogger.WriteLog(ResponseStrings.InfoQueuedOperationAborted, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_078);
                return;
            }

            //ALL Documents in the same Document Set must be performed in order. This is determined from the SourceSystemSetId Id in the
            //PcehrMessageQueue table. If there are any PENDING messages that are from the same DocumentSet that are earlier than this
            //message then this message must log that there are earlier messages still to upload and this must wait.
            PcehrMessageQueueDl dataAcces = new PcehrMessageQueueDl(operation.User);
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                if (dataAcces.CountEarlierPendingMessages(operation.SourceSystemSetId, operation.PendingItem.DateCreated, operation.PendingItem.Id) > 0)
                {
                    string message = string.Format(ResponseStrings.OperationOutOfOrderRemove,
                        operation.PendingItem.PcehrMessageQueueId,
                        operation.PendingItem.QueueOperationName,
                        operation.SourceSystemSetId,
                        operation.PatientIdentifier,
                        operation.Hospital.Description);
                    EventLogger.WriteLog(ResponseStrings.InfoQueuedOperationRetriedDueToMessagesOutofOrder, new Exception(message), operation.User, LogMessage.HIPS_MESSAGE_137);
                    throw new InvalidOperationException();
                }
            }

            // Ensure the IHI has been validated within the configured period.
            // This is always done before the removal operation is placed on the
            // queue, however if the HI Service was unavailable at the time then
            // it must be done now. If the HI Service is still unavailable now
            // then the queue transaction will be rolled back and the removal
            // will stay on the queue, being retried, until the HI service is
            // once more available.
            PatientIhiValidation patientIhiValidation = new PatientIhiValidation();
            IhiSearchResponse ihiResponse;
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                ihiResponse = patientIhiValidation.GetValidatedIhi(operation.PatientIdentifier, operation.Hospital, operation.User, operation.PatientMaster);
            }
            if (ihiResponse.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
            {
                // If the IHI cannot be validated because the HI service is unavailable, HIPS will
                // roll back the queue transaction, and the operation will be tried again.
                using (new TransactionScope(TransactionScopeOption.Suppress))
                {
                    EventLogger.WriteLog(ResponseStrings.InfoRemovalWithStaleIhiBeingRetried, new Exception(ihiResponse.HipsResponse.HipsErrorMessage), operation.User, LogMessage.HIPS_MESSAGE_079);
                }
                throw new InvalidOperationException(ResponseStrings.InfoRemovalWithStaleIhiBeingRetried);
            }
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                operation.PendingItem.Details = ihiResponse.HipsResponse.HipsErrorMessage;
                handler.UpdateOperationForFailure(operation.PendingItem);
                return;
            }
            this.patientAccess.ValidateLocalIhiInformation(operation.PatientMaster, ihiResponse.HipsResponse);
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                operation.PendingItem.Details = ihiResponse.HipsResponse.HipsErrorMessage;
                handler.UpdateOperationForFailure(operation.PendingItem);
                return;
            }

            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                document = this.patientAccess.DocumentDataAccess.Get(this.operation.Episode.EpisodeId.Value, this.operation.SourceSystemSetId);
            }
            if (document == null || !document.ClinicalDocumentId.HasValue)
            {
                string errorDetails = string.Format(ResponseStrings.DocumentNotFoundFormat, this.operation.SourceSystemSetId);
                this.operation.PendingItem.Details = errorDetails;
                this.operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                handler.UpdateOperationForFailure(operation.PendingItem);
            }

            // Resolve the document type using the ID stored in the ClinicalDocument table.
            // The invoker needs the document type to determine whether to call PCEHR or NPDR.
            operation.DocumentType = ListSingleton.Instance.AllDocumentTypes.FirstOrDefault(a => a.DocumentTypeId == document.DocumentTypeId);

            ClinicalDocumentVersion version;
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                this.patientAccess.VersionDataAccess.GetLatestVersion(document.ClinicalDocumentId.Value, out version);
            }
            string documentId = version.SourceSystemDocumentId;

            if (DocumentRemovalInvoker.Remove(this.operation, documentId))
            {
                document.ClinicalDocumentStatusId = (int)Enums.ClinicalDocumentStatus.Removed;
                document.RemovalReasonId = (int)this.operation.RemovalReason;
                document.RemovedDate = DateTime.Now;
                handler.SaveAfterSuccessfulRemove(document, this.operation);
            }
            else
            {
                handler.UpdateOperationForFailure(operation.PendingItem);
            }
        }

        /// <summary>
        /// Checks whether the PcehrMessageQueue item is still pending. If the
        /// status has been changed to Success or Failure then the queued
        /// operation will be aborted.
        /// </summary>
        /// <returns>True if the queue status is still Pending.</returns>
        /// <exception cref="System.InvalidOperationException">If access to the database fails.</exception>
        private bool IsRemovalOperationStillPending()
        {
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                PcehrMessageQueueDl queueDataAccess = new PcehrMessageQueueDl(operation.User);
                PcehrMessageQueue storedItem;
                if (queueDataAccess.Get(operation.PendingItem.PcehrMessageQueueId.Value, out storedItem))
                {
                    return storedItem.QueueStatusId == (int)QueueStatus.Pending;
                }
            }

            // Error getting to the database - retry the queued operation, don't abort it.
            throw new InvalidOperationException();
        }
    }
}