﻿using System;
using System.Collections.Generic;
using System.Linq;
using HIPS.AppServer.ServiceHost;
using HIPS.BusinessEntities.ResponseEntities;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.Reference.ServiceContracts;

namespace HIPS.Reference.ServiceHost
{
    public class ReferenceService : BaseService, IHIPSReferenceService
    {
        /// <summary>
        /// Find all known information about a hospital, such as:
        /// <list identifier="bullet">
        /// <item>Description and Name</item>
        /// <item>Address</item>
        /// <item>Fax Number</item>
        /// <item>List of Identifiers (HPI-O, OID for MRNs, OID for Provider Identifers, ISAAC)</item>
        /// </list>
        /// </summary>
        /// <param name="identifier">An identifier for the hospital</param>
        /// <param name="codeSystemName">The namespace of the identifier supplied</param>
        /// <returns>The hospital details</returns>
        public HospitalDetailResponse GetHospitalDetails(string hospitalCode, string hospitalCodeSystem)
        {
            ListSingleton lists = ListSingleton.Instance;
            HospitalSingleton hospitals = HospitalSingleton.Value;
            HospitalDetailResponse response = new HospitalDetailResponse();
            Hospital hospital = hospitals.Find(hospitalCode, hospitalCodeSystem);
            if (hospital == null)
            {
                return response;
            }
            CodeSystem isaacNamespace = lists.AllCodeSystems.Where(a => a.Code == "isaacFacCd").FirstOrDefault();
            response.ShortDescription = hospital.Description;
            response.FullName = hospital.Name;
            response.Contacts = hospital.Contacts;
            response.Addresses = hospital.Addresses;
            response.HpiO = hospital.HpiO;
            response.ProviderIdentifierRoot = hospital.GetCode("doctorOid");
            response.PatientIdentifierRoot = hospital.GetCode("mrnOid");
            response.HospitalIdentifierExtension = hospital.GetCode("isaacFacCd");
            response.HospitalIdentifierRoot = isaacNamespace.Oid;
            return response;
        }

        /// <summary>
        /// Lists the hospitals.
        /// </summary>
        /// <returns>A list of all the hospitals.</returns>
        public List<Hospital> ListHospitals()
        {
            List<Hospital> hospitals = new List<Hospital>();

            foreach (Hospital hospital in ListSingleton.Instance.AllHospitals)
            {
                hospitals.Add(hospital);
            }

            return hospitals;
        }

        /// <summary>
        /// Reloads all the reference data from the database.
        /// See <see cref="HIPS.CommonBusinessLogic.Singleton.ListSingleton"/> for
        /// further details of which reference data is loaded.
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action.</param>
        /// <returns>Indicator of success or failure</returns>
        public HipsResponse ReloadReferenceData(UserDetails user)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                ListSingleton.Instance.Refresh(true);
                Settings.Instance.RefreshAppConfigSettings();
                EventLogger.WriteLog("(INFO) Reference data was reloaded", null, user, LogMessage.HIPS_MESSAGE_120);
            }
            catch (Exception ex)
            {
                response.Status = HipsResponseIndicator.DatabaseError;
                response.HipsErrorMessage = ex.Message;
                EventLogger.WriteLog("(ERROR) Unable to reload reference data", ex, user, LogMessage.HIPS_MESSAGE_121);
            }
            return response;
        }
    }
}