﻿using System;
using System.Runtime.Serialization;

using HIPS.ServiceContracts.Common.DTO.ParticipatingIndividual;

using EntLibVal = Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.ServiceContracts.Common.DTO
{
    /// <summary>
    /// CDA Header Metadata DTO. This object contains the metadata that is
    /// required to be used within a generated CDA Document, such as the CDA
    /// Header and other data within the CDA Document that is not gathered
    /// from HL7 messages passed into HIPS from the DatabaseLoader.
    /// Some data may be available from an HL7 feed, however it may not be
    /// sent to HIPS before it is actually required when generating the CDA
    /// document and thus this information is also
    /// required in this metadata.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    [EntLibVal.Validators.HasSelfValidation]
    public class CdaHeaderMetadata
    {
        #region Properties

        /// <summary>
        /// Gets or sets the admission date and time. This is when the patient visit started.
        /// This date and time should match an existing episode supplied in the HL7 feed.
        /// If there is no matching episode, and the patient identifier is a ValidatedIhi,
        /// an episode stub will be created. If there is no matching episode, and the patient
        /// identifier is a different type, an error will be returned.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DateTimeRangeValidator(Constants.SupportedDates.MINIMUM, Constants.SupportedDates.MAXIMUM)]
        public DateTime AdmissionDateTime { get; set; }

        /// <summary>
        /// Gets or sets the discharge date and time. This is when the patient visit ended or is planned to end.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DateTimeRangeValidator(Constants.SupportedDates.MINIMUM, Constants.SupportedDates.MAXIMUM)]
        public DateTime DischargeDateTime { get; set; }

        /// <summary>
        /// Gets or sets an <see cref="ParticipatingIndividual" /> object that represents
        /// the person who is the 'Author' for the document.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.ObjectValidator]
        [EntLibVal.Validators.NotNullValidator]
        public ParticipatingProvider DocumentAuthor { get; set; }

        /// <summary>
        /// Gets or sets the document creation date and time. This should be the
        /// date and time when this version of the document was completed, not
        /// when work on it started.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DateTimeRangeValidator(Constants.SupportedDates.MINIMUM, Constants.SupportedDates.MAXIMUM)]
        public DateTime DocumentCreationDateTime { get; set; }

        /// <summary>
        /// Gets or sets an <see cref="ParticipatingIndividual" /> object that represents
        /// the person who is the 'Legal Authenticator' for the document.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.ObjectValidator]
        public ParticipatingProvider LegalAuthenticator { get; set; }

        /// <summary>
        /// Gets or sets a code that represents the status of the patient at the time of discharge and the place to which the person was or will be released.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DomainValidator(ModeOfSeparation.None, Negated = true)]
        public ModeOfSeparation ModeOfSeparation { get; set; }

        /// <summary>
        /// Gets or sets the address of the patient. An address is mandatory for the CDA document. If this data member is not supplied, the address must have been supplied in the HL7 feed.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.ObjectValidator]
        public Address PatientAddress { get; set; }

        /// <summary>
        /// Gets or sets the contact details of the patient. Contact details
        /// are not mandatory for the CDA document. If this data member is
        /// not supplied, the contact details supplied in the HL7 feed may be used.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.ObjectValidator]
        public ElectronicCommunicationDetail PatientContactDetails { get; set; }

        /// <summary>
        /// Gets or sets an <see cref="ParticipatingIndividual" /> object that represents
        /// the person who is the 'Responsible Health Professional at Time of
        /// Discharge' for the document.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.ObjectValidator]
        [EntLibVal.Validators.NotNullValidator]
        public ParticipatingProvider ResponsibleHealthProfessional { get; set; }

        /// <summary>
        /// Gets or sets the status of this version of the document, which may
        /// be either Interim (draft), Final (complete) or Withdrawn (recalled
        /// by the authoring organisation).
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.DomainValidator(SourceDocumentStatus.None, Negated = true)]
        public SourceDocumentStatus SourceDocumentStatus { get; set; }

        /// <summary>
        /// Gets or sets the clinical specialty or hospital department in which the patient was treated before discharge.
        /// </summary>
        [DataMember(IsRequired = true)]
        [EntLibVal.Validators.StringLengthValidator(1, 1024)]
        public string Specialty { get; set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Validate the instance is valid.
        /// </summary>
        /// <param name="results">Validation results to add results of validation to.</param>
        [EntLibVal.Validators.SelfValidation]
        protected void Validate(EntLibVal.ValidationResults results)
        {
            if (this.AdmissionDateTime > this.DischargeDateTime)
            {
                results.AddResult(new EntLibVal.ValidationResult("The value of AdmissionDateTime must be prior to the value of DischargeDateTime.", this, "CdaHeaderMetadata", string.Empty, null));
            }
        }

        #endregion Methods
    }
}