﻿using System.Runtime.Serialization;

using EntLibVal = Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.ServiceContracts.Common.DTO.ParticipatingIndividual
{
    /// <summary>
    /// Participating Provider DTO. This object contains information that identifies the
    /// person who is the 'Author', 'Legal Authenticator' and/or 'Responsible Health
    /// Professional at time of discharge' for the document.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    [KnownType(typeof(ParticipatingIndividual))]
    [EntLibVal.Validators.HasSelfValidation]
    public class ParticipatingProvider : ParticipatingIndividual
    {
        #region Properties

        /// <summary>
        /// Gets or sets the HPI-O of the organisation that employs the participant.
        /// If the employer HPI-O is specified then the employer name must also.
        /// If neither are specified, the HPI-O and name of the hospital will be used.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(16, 16)]
        [EntLibVal.Validators.IgnoreNulls]
        public string EmployerHpio { get; set; }

        /// <summary>
        /// Gets or sets the name of the organisation that employs the participant.
        /// If the employer HPI-O is specified then the employer name must also.
        /// If neither are specified, the HPI-O and name of the hospital will be used.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(1, 80)]
        [EntLibVal.Validators.IgnoreNulls]
        public string EmployerName { get; set; }

        /// <summary>
        /// Gets or sets the validated HPI-I of the participating provider. For
        /// the document author and responsible health professional, this is
        /// conditionally mandatory with the Local Identifier as at least one
        /// must be specified. If both are specified then the HPI-I will take
        /// precedence over the Local Identifier.
        /// </summary>
        [DataMember(IsRequired = false)]
        [EntLibVal.Validators.StringLengthValidator(16, 16)]
        [EntLibVal.Validators.IgnoreNulls]
        public string Hpii { get; set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Validate the instance is valid.
        /// </summary>
        /// <param name="results">Validation results to add results of validation to.</param>
        [EntLibVal.Validators.SelfValidation]
        protected void Validate(EntLibVal.ValidationResults results)
        {
            if ((this.EmployerHpio == null && this.EmployerName != null)
                || (this.EmployerHpio != null && this.EmployerName == null))
            {
                results.AddResult(new EntLibVal.ValidationResult("The EmployerHpio and EmployerName must both be specified or neither may be specified.", this, "ParticipatingProvider", string.Empty, null));
            }

            if (this.Hpii == null && this.LocalIdentifier == null)
            {
                results.AddResult(new EntLibVal.ValidationResult("At least one of the Hpii and LocalIdentifier must be specified.", this, "ParticipatingProvider", string.Empty, null));
            }
        }

        #endregion Methods
    }
}