﻿using System.Collections.Generic;
using System.ServiceModel;
using AutoMapper;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.CommonSchemas.Exceptions;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Common.DTO;
using HIPS.ServiceContracts.Common.Fault;
using Microsoft.Practices.EnterpriseLibrary.Validation;

namespace HIPS.AppServer.HIPSServiceHost.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for faults.
    /// Every type of business exception should be mapped to a separate fault type.
    /// </summary>
    internal class FaultProfile : Profile
    {
        #region Properties

        /// <summary>
        /// Gets the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            /*
            Common --> Service
            */

            // System.Exception
            this.CreateMap<System.Exception, FaultException<ServiceOperationFault>>()
                // Need to use this approach because FaultException:
                // (a) Doesn't have a default constructor, and
                // (b) The majority of its properties are read-only.
                .ConstructUsing(
                    (System.Func<System.Exception, FaultException<ServiceOperationFault>>)
                    (src => new FaultException<ServiceOperationFault>(
                        ObjectMapper.Map<ServiceOperationFault>(src),
                        HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_SERVICE_OPERATION,
                        new FaultCode(
                            typeof(ServiceOperationFault).Name,
                            HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1))));

            // System.Exception
            this.CreateMap<System.Exception, ServiceOperationFault>()
                .ForMember(dest => dest.Message, opt => opt.UseValue(HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_SERVICE_OPERATION))
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => src.InnerException.Message))
                .ForMember(dest => dest.FaultIdentifier, opt => opt.ResolveUsing((System.Exception val) => System.Guid.NewGuid().ToString()));

            // ValidationResults
            this.CreateMap<ValidationResults, FaultException<InvalidRequestFault>>()
                // Need to use this approach because FaultException:
                // (a) Doesn't have a default constructor, and
                // (b) The majority of its properties are read-only.
                .ConstructUsing(
                    (System.Func<ValidationResults, FaultException<InvalidRequestFault>>)
                    (src => new FaultException<InvalidRequestFault>(
                        ObjectMapper.Map<InvalidRequestFault>(src),
                        HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_INVALID_REQUEST,
                        new FaultCode(
                            typeof(InvalidRequestFault).Name,
                            HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1))));

            // ValidationResults
            this.CreateMap<ValidationResults, InvalidRequestFault>()
                .ForMember(dest => dest.Message, opt => opt.UseValue(HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_INVALID_REQUEST))
                .ForMember(dest => dest.FaultIdentifier, opt => opt.ResolveUsing((ValidationResults val) => System.Guid.NewGuid().ToString()))
                .ForMember(dest => dest.Messages, opt => opt.MapFrom(s => ObjectMapper.Map<List<Message>>(s)));

            // ValidationResult
            this.CreateMap<ValidationResult, Message>()
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => string.Format("{0}: {1}", src.Key, src.Message)))
                .ForMember(dest => dest.Level, opt => opt.UseValue(MessageLevel.Warning))
                .ForMember(dest => dest.Origin, opt => opt.UseValue(HIPS.CommonSchemas.Constants.Origin.HIPS));

            // ItemNotFoundException
            this.CreateMap<ItemNotFoundException, FaultException<ItemNotFoundFault>>()
                // Need to use this approach because FaultException:
                // (a) Doesn't have a default constructor, and
                // (b) The majority of its properties are read-only.
                .ConstructUsing(
                    (System.Func<ItemNotFoundException, FaultException<ItemNotFoundFault>>)
                    (src => new FaultException<ItemNotFoundFault>(
                        ObjectMapper.Map<ItemNotFoundFault>(src),
                        HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_ITEM_NOT_FOUND,
                        new FaultCode(
                            typeof(ItemNotFoundFault).Name,
                            HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1))));

            // ItemNotFoundException
            this.CreateMap<ItemNotFoundException, ItemNotFoundFault>()
                .ForMember(dest => dest.FaultIdentifier, opt => opt.ResolveUsing((ItemNotFoundException val) => System.Guid.NewGuid().ToString()));

            // InvalidUserException
            this.CreateMap<InvalidUserException, FaultException<InvalidUserFault>>()
                // Need to use this approach because FaultException:
                // (a) Doesn't have a default constructor, and
                // (b) The majority of its properties are read-only.
                .ConstructUsing(
                    (System.Func<InvalidUserException, FaultException<InvalidUserFault>>)
                    (src => new FaultException<InvalidUserFault>(
                        ObjectMapper.Map<InvalidUserFault>(src),
                        HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_INVALID_USER,
                        new FaultCode(
                            typeof(InvalidUserFault).Name,
                            HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1))));

            // InvalidUserException
            this.CreateMap<InvalidUserException, InvalidUserFault>()
                .ForMember(dest => dest.FaultIdentifier, opt => opt.ResolveUsing((InvalidUserException val) => System.Guid.NewGuid().ToString()));

            // HiServiceException
            this.CreateMap<HiServiceException, FaultException<HiServiceFault>>()
                // Need to use this approach because FaultException:
                // (a) Doesn't have a default constructor, and
                // (b) The majority of its properties are read-only.
                .ConstructUsing(
                    (System.Func<HiServiceException, FaultException<HiServiceFault>>)
                    (src => new FaultException<HiServiceFault>(
                        ObjectMapper.Map<HiServiceFault>(src),
                        HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_HI_SERVICE,
                        new FaultCode(
                            typeof(HiServiceFault).Name,
                            HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1,
                            new FaultCode(
                                src.ResponseCode,
                                HIPS.ServiceContracts.Common.Constants.ServiceNamespace.IHI_V1)))));

            // HiServiceException
            this.CreateMap<HiServiceException, HiServiceFault>()
                .ForMember(dest => dest.FaultIdentifier, opt => opt.ResolveUsing((HiServiceException val) => System.Guid.NewGuid().ToString()));

            // PcehrServiceException
            this.CreateMap<PcehrServiceException, FaultException<PcehrServiceFault>>()
                // Need to use this approach because FaultException:
                // (a) Doesn't have a default constructor, and
                // (b) The majority of its properties are read-only.
                .ConstructUsing(
                    (System.Func<PcehrServiceException, FaultException<PcehrServiceFault>>)
                    (src => new FaultException<PcehrServiceFault>(
                        ObjectMapper.Map<PcehrServiceFault>(src),
                        HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_PCEHR_SERVICE,
                        new FaultCode(
                            typeof(PcehrServiceFault).Name,
                            HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1,
                            new FaultCode(
                                src.ResponseCode,
                                HIPS.ServiceContracts.Common.Constants.ServiceNamespace.PCEHR_V2)))));

            // PcehrServiceException
            this.CreateMap<PcehrServiceException, PcehrServiceFault>()
                .ForMember(dest => dest.FaultIdentifier, opt => opt.ResolveUsing((PcehrServiceException val) => System.Guid.NewGuid().ToString()));

            // HipsResponseException - unhandled HipsResponse types are mapped to ServiceOperationFault.
            this.CreateMap<HipsResponseException, FaultException<ServiceOperationFault>>()
                .ConstructUsing(
                    (System.Func<HipsResponseException, FaultException<ServiceOperationFault>>)
                    (src => new FaultException<ServiceOperationFault>(
                        ObjectMapper.Map<ServiceOperationFault>(src),
                        HIPS.AppServer.HIPSServiceHost.Properties.Resources.MESSAGE_ERROR_SERVICE_OPERATION,
                        new FaultCode(
                            typeof(ServiceOperationFault).Name,
                            HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1,
                            new FaultCode(
                                src.Detail.Status.ToString(),
                                HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)))));

            // HipsResponseException
            this.CreateMap<HipsResponseException, ServiceOperationFault>()
                .ForMember(dest => dest.Type, opt => opt.MapFrom(src => src.Detail.Status))
                .ForMember(dest => dest.Code, opt => opt.MapFrom(src => src.Detail.ResponseCode))
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => src.Detail.ResponseCodeDescription))
                .ForMember(dest => dest.Details, opt => opt.MapFrom(src => src.Detail.ResponseCodeDetails))
                .ForMember(dest => dest.Message, opt => opt.MapFrom(src => src.Detail.HipsErrorMessage))
                .ForMember(dest => dest.FaultIdentifier, opt => opt.ResolveUsing((HipsResponseException val) => System.Guid.NewGuid().ToString()));
        }

        #endregion Methods
    }
}