﻿using System;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.CommonBusinessLogic.Ihi
{
    public class PatientDemographicUpdate
    {
        #region Methods

        /// <summary>
        /// Performs necessary IHI processing for an updated patient record.
        /// If we do not have an IHI, but the necessary information is available, performs an IHI search.
        /// If we do have an IHI, but the key fields have changed, revalidates the IHI.
        /// </summary>
        /// <param name="oldPatientMaster">A clone of the patient master record before the update occurred</param>
        /// <param name="patientMaster">The newly updated patient master record</param>
        /// <param name="hospital">The hospital</param>
        /// <param name="user">The person responsible for the IHI searches triggered by this demographic update</param>
        public void UpdatePatient(PatientMaster oldPatientMaster, PatientMaster patientMaster, Hospital hospital, UserDetails user)
        {
            if (!User.PopulateAndValidateUser(hospital, user))
            {
                throw new Exception(ConstantsResource.InvalidUserDetails);
            }
            if (!string.IsNullOrEmpty(patientMaster.Ihi))
            {
                bool sameFirst;
                bool sameLast;
                if (string.IsNullOrEmpty(oldPatientMaster.RegisteredFamilyName) || string.IsNullOrEmpty(oldPatientMaster.RegisteredGivenName))
                {
                    // This should not happen - indicates that an IHI was recorded for the patient but we didn't set the registered name!
                    sameFirst = false;
                    sameLast = false;
                }
                else
                {
                    sameFirst = (oldPatientMaster.CurrentName.GivenNames.ToUpper() == patientMaster.CurrentName.GivenNames.ToUpper());
                    sameLast = (oldPatientMaster.CurrentName.FamilyName.ToUpper() == patientMaster.CurrentName.FamilyName.ToUpper());
                }
                bool sameDob = oldPatientMaster.DateOfBirth.Equals(patientMaster.DateOfBirth);
                bool sameSex = oldPatientMaster.CurrentSexId == patientMaster.CurrentSexId;
                bool medicareNumberChange = Medicare.ChangeNeedsIhiValidation(oldPatientMaster.MedicareNumber, patientMaster.MedicareNumber);
                bool sameIrn = oldPatientMaster.MedicareIrn == patientMaster.MedicareIrn;
                bool sameDvaNumber = oldPatientMaster.DvaNumber == patientMaster.DvaNumber;

                if (medicareNumberChange || !sameIrn || !sameDvaNumber)
                {
                    // If you change the first 9 digits, or the IRN, or the DVA number, it will trigger revalidation.
                    // This revalidation must do a search using the current Medicare and DVA number, instead of the IHI.
                    PatientIhiValidation validation = new PatientIhiValidation();
                    validation.RevalidateMedicareDva(patientMaster, hospital, user);
                }
                else if (!sameFirst || !sameLast || !sameDob || !sameSex || patientMaster.IhiStatusId == (int)IhiStatus.Unknown)
                {
                    // If the key demographic information changed, we must revalidate the IHI
                    PatientIhiValidation validation = new PatientIhiValidation();
                    validation.RevalidateIhi(patientMaster, hospital, user);
                }
            }
            else
            {
                // The system does not already have an IHI for this patient. Attempt a lookup with the updated information.
                PatientRegistration registration = new PatientRegistration();
                registration.RegisterPatient(patientMaster, hospital, user);
            }
        }

        #endregion Methods
    }
}