﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.CommonBusinessLogic.Ihi
{
    public static class SearchHelper
    {
        #region Static Fields

        /// <summary>
        /// The apostrophe.
        /// </summary>
        private static readonly string Apostrophe = "'";

        /// <summary>
        /// Precompiled regular expression that finds all characters other than spaces, letters, apostrophes and hyphens.
        /// </summary>
        private static readonly Regex DisallowedCharacterPattern = new Regex("[^ A-Za-z'-]");

        /// <summary>
        /// The hyphen.
        /// </summary>
        private static readonly string Hyphen = "-";

        /// <summary>
        /// Precompiled regular expression that finds apostrophes and all spaces next to them.
        /// </summary>
        private static readonly Regex SpacesAroundApostrophePattern = new Regex(" *' *");

        /// <summary>
        /// Precompiled regular expression that finds hyphens and all spaces next to them.
        /// </summary>
        private static readonly Regex SpacesAroundHyphenPattern = new Regex(" *- *");

        #endregion Static Fields

        #region Public Methods

        /// <summary>
        /// Populates the IHI search criteria from the patient master object.
        /// If an IHI is supplied in the master, creates the criteria for
        /// validation. HIPS will attempt to validate with the new current name
        /// first, and if not found under that name, then try validating with
        /// the previous registered name. If no IHI is supplied in the master,
        /// then creates the criteria for a new search by Medicare Card Number
        /// or DVA File Number.
        /// </summary>
        /// <param name="patientMaster">The patient master (required)</param>
        /// <returns>Populated IHI search criteria object</returns>
        public static IhiSearchCriteria PopulateSearchDetails(PatientMaster patientMaster)
        {
            IhiSearchCriteria searchDetails = new IhiSearchCriteria();
            searchDetails.PatientMasterId = patientMaster.PatientMasterId.Value;
            SearchHelper.AddSearchName(searchDetails, patientMaster.CurrentName, patientMaster.CurrentSexId);
            SearchHelper.AddSearchName(searchDetails, patientMaster.LegalName, patientMaster.RegisteredSexId);
            searchDetails.DateOfBirth = patientMaster.DateOfBirth;

            if (!patientMaster.IsDOBandRDOBEqual() && patientMaster.RegisteredDateOfBirth != DateTime.MinValue)
            {
                AddSearchName(searchDetails, patientMaster.CurrentName, patientMaster.CurrentSexId, patientMaster.RegisteredDateOfBirth);

                AddSearchName(searchDetails, patientMaster.LegalName, patientMaster.RegisteredSexId, patientMaster.RegisteredDateOfBirth);
            }

            if (string.IsNullOrEmpty(patientMaster.Ihi))
            {
                searchDetails.DvaNumber = patientMaster.DvaNumber;
                searchDetails.MedicareNumber = patientMaster.MedicareNumber;
                if (!string.IsNullOrEmpty(patientMaster.MedicareIrn) && patientMaster.MedicareIrn != "0")
                {
                    searchDetails.MedicareIrn = patientMaster.MedicareIrn;
                }
                searchDetails.IsMedicareNumberValid = patientMaster.IsMedicareNumberValid;
            }
            else
            {
                searchDetails.Ihi = patientMaster.Ihi;
            }
            return searchDetails;
        }

        #endregion Public Methods

        #region Private Methods

        /// <summary>
        /// Adds each given name along with the family name and sex to the search criteria.
        /// </summary>
        /// <param name="search">The search criteria.</param>
        /// <param name="searchName">The name to add to the search criteria.</param>
        /// <param name="sex">The sex to add to the search criteria</param>
        private static void AddSearchName(IhiSearchCriteria search, PatientMasterName searchName, int sexId, DateTime? dateOfBirth = null)
        {
            if (searchName == null)
            {
                return;
            }
            search.Names = search.Names ?? new List<IhiSearchCriteriaName>();

            if (searchName.GivenNames == null || searchName.FamilyName == null)
            {
                return;
            }
            searchName.GivenNames = SanitiseName(searchName.GivenNames);
            searchName.FamilyName = SanitiseName(searchName.FamilyName);
            List<string> firstNames = searchName.GivenNames.Trim(' ').Split(' ').ToList();
            foreach (string firstName in firstNames)
            {
                if (string.IsNullOrEmpty(firstName))
                {
                    continue;
                }
                IhiSearchCriteriaName name = new IhiSearchCriteriaName();
                name.FamilyName = searchName.FamilyName;
                name.GivenName = firstName;
                if (Enum.IsDefined(typeof(SexEnumerator), sexId))
                {
                    name.Sex = (SexEnumerator)sexId;
                }
                else
                {
                    name.Sex = SexEnumerator.NotStatedOrInadequatelyDescribed;
                }

                name.SuffixId = searchName.SuffixId;
                name.TitleId = searchName.TitleId;
                name.DateOfBirth = dateOfBirth;
                if (search.Names.Count(result =>
                       result.FamilyName == name.FamilyName
                    && result.GivenName == name.GivenName
                    && result.Sex == name.Sex
                    && CompareDateOfBirth(result.DateOfBirth, name.DateOfBirth)) == 0)
                {
                    search.Names.Add(name);
                }
            }
        }

        /// <summary>
        /// Compares two nullable date of births to see if the dates are the same.
        /// </summary>
        /// <param name="dateOfBirth1">The date of birth1.</param>
        /// <param name="dateOfBirth2">The date of birth2.</param>
        /// <returns></returns>
        private static bool CompareDateOfBirth(DateTime? dateOfBirth1, DateTime? dateOfBirth2)
        {
            bool result = false;

            if (dateOfBirth1.HasValue && dateOfBirth2.HasValue)
            {
                result = dateOfBirth1.Value.Date == dateOfBirth2.Value.Date;
            }
            else
            {
                result = dateOfBirth1 == dateOfBirth2;
            }

            return result;
        }

        /// <summary>
        /// Removes spaces around hyphens and apostrophes, and removes all
        /// characters except for spaces, letters, hyphens and apostrophes.
        /// </summary>
        /// <param name="name">The given names or family name.</param>
        /// <returns>The sanitised name.</returns>
        private static string SanitiseName(string name)
        {
            name = SpacesAroundHyphenPattern.Replace(name, Hyphen);
            name = SpacesAroundApostrophePattern.Replace(name, Apostrophe);
            name = DisallowedCharacterPattern.Replace(name, string.Empty);
            return name;
        }

        #endregion Private Methods
    }
}