﻿using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using HIPS.CommonSchemas;
using HIPS.PcehrSchemas;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    /// <summary>
    /// Contains methods for analysing the response messages returned by the PCEHR system
    /// and deciding whether to treat as a success, transient failure or permanent failure.
    /// </summary>
    public class FaultHelper
    {
        #region Private Members

        /// <summary>
        /// A precompiled regular expression to get the code from a message.
        /// </summary>
        private static Regex NpdrErrorCodeRegex = new Regex(@"MEDVIEW_REPOSITORY_\d+");

        private static HashSet<string> PCEHR_COMPLETED_WITH_WARNINGS_CODES = new HashSet<string>
        {
            "PCEHR_ERROR_3007",
        };

        private static HashSet<string> PCEHR_PERMANENT_FAILURE_CODES = new HashSet<string>
        {
            // Response Statuses
            "PCEHR_ERROR_3002",  // Document metadata failed validation (e.g. remove document via the wrong HPI-O)
            "PCEHR_ERROR_2501",  // Document not found (e.g. remove document that is already removed)

            // XDS Repository Errors
            "PCEHR_ERROR_3001",
            "PCEHR_ERROR_3002",  // Document metadata failed validation (e.g. supersede on different IHI or HPI-O)
            "PCEHR_ERROR_3003",
            "PCEHR_ERROR_3004",
            "PCEHR_ERROR_3005",
            "PCEHR_ERROR_3006",
            "PCEHR_ERROR_3008",
            "PCEHR_ERROR_3501",
            "PCEHR_ERROR_3502",
            "PCEHR_ERROR_3503",

            // Common SOAP Faults
            "PCEHR_ERROR_0001",
            "PCEHR_ERROR_0002",
            "PCEHR_ERROR_0003",
            "PCEHR_ERROR_0004",
            "PCEHR_ERROR_0006",
            "PCEHR_ERROR_0007",
            "PCEHR_ERROR_0008",
            "PCEHR_ERROR_0009",
            "PCEHR_ERROR_0010",
            "PCEHR_ERROR_0501",
            "PCEHR_ERROR_0502",
            "PCEHR_ERROR_0503",
            "PCEHR_ERROR_0504",
            "PCEHR_ERROR_0505", // Invalid HPI-O (e.g. different to certificate)
            "PCEHR_ERROR_0506",
            "PCEHR_ERROR_0509",
            "PCEHR_ERROR_0510",
            "PCEHR_ERROR_0511",
            "PCEHR_ERROR_0512",
            "PCEHR_ERROR_0513",
            "PCEHR_ERROR_0514",
            "PCEHR_ERROR_0519",
            "PCEHR_ERROR_0520",
            "PCEHR_ERROR_0521",
            "PCEHR_ERROR_0522",
            "PCEHR_ERROR_0523",
            "PCEHR_ERROR_0524",
            "PCEHR_ERROR_0525",
            "PCEHR_ERROR_0526",

            // NPDR Additional Responses
            "MEDVIEW_REPOSITORY_003",
            "MEDVIEW_REPOSITORY_004",
            "MEDVIEW_REPOSITORY_006",
            "MEDVIEW_REPOSITORY_007",
            "MEDVIEW_REPOSITORY_008",
            "MEDVIEW_REPOSITORY_010",
            "MEDVIEW_REPOSITORY_011",
            "MEDVIEW_REPOSITORY_012",
            "MEDVIEW_REPOSITORY_014",
            "MEDVIEW_REPOSITORY_015",
            "MEDVIEW_REPOSITORY_016",
        };

        private static HashSet<string> PCEHR_SUCCESS_MESSAGES = new HashSet<string>
        {
            "PCEHR_SUCCESS",
            "urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Success"
        };

        private static HashSet<string> PCEHR_TRANSIENT_FAILURE_CODES = new HashSet<string>
        {
            "PCEHR_ERROR_0005",
            "PCEHR_ERROR_0011",
            "PCEHR_ERROR_0012",
            "PCEHR_ERROR_0013",
            "PCEHR_ERROR_0014",
            "PCEHR_ERROR_0507",
            "PCEHR_ERROR_0515",
            "PCEHR_ERROR_0516",
            "PCEHR_ERROR_0517",
            "PCEHR_ERROR_0518",
        };

        /// <summary>
        /// A precompiled regular expression to get the code from a message.
        /// </summary>
        private static Regex PcehrErrorCodeRegex = new Regex(@"PCEHR_ERROR_\d+");

        #endregion Private Members

        #region Public Methods

        /// <summary>
        /// Classifies the registry error list. If the list is null or empty then
        /// the operation was completed successfully. If the highest severity is
        /// a warning then the operation was completed with warnings. Otherwise
        /// the codeContext from each error is processed by the ClassifyMessages
        /// method in this class. See the documentation on ClassifyMessages for
        /// more information.
        /// </summary>
        /// <param name="errorList">The registry error list (or null if no errors).</param>
        /// <returns>The fault action.</returns>
        internal static FaultAction Classify(Nehta.VendorLibrary.PCEHR.DocumentRepository.RegistryErrorList errorList)
        {
            FaultAction action = FaultAction.Unknown;
            if (errorList == null)
            {
                action = FaultAction.Completed;
            }
            else if (errorList.highestSeverity == PcehrResponseStrings.ErrorSeverityTypeWarning)
            {
                action = FaultAction.CompletedWithWarnings;
            }
            else
            {
                string[] allCodeContexts = (from error in errorList.RegistryError
                                            select error.codeContext).ToArray();
                action = ClassifyMessages(allCodeContexts);
            }
            return action;
        }

        /// <summary>
        /// Classifies each message according to its severity, and returns the
        /// most severe situation among those found, where the following are
        /// ordered from least to most severe:
        /// 1. "Completed"               - no further action required
        /// 2. "Completed with Warnings" - investigate as resources permit
        /// 3. "Transient Failure"       - try the request again
        /// 4. "Permanent Failure"       - investigate why it failed
        ///
        /// If there are no messages or the message "PCEHR_SUCCESS", this is
        /// classified as "Completed".
        ///
        /// If the message contains PCEHR_ERROR_nnnn where nnnn is a numeric
        /// error code that is described in the PCEHR Document Exchange Service
        /// Technical Service Specification (DEXS-TSS), then it will be
        /// classified either "CompletedWithWarnings" or "TransientFailure" or
        /// "PermanentFailure" according to the table in HIPS design documentation.
        ///
        /// When the message contains "already present in the registry" this is
        /// classified as "CompletedWithWarnings" so that HIPS will attempt to
        /// store the document in the local database.
        ///
        /// Otherwise it is classified as "Unknown", which should be treated
        /// similarly to "PermanentFailure" in that the operation is marked as
        /// failed, to be investigated by application support.
        /// </summary>
        /// <param name="messages">One or more response messages to be classified.</param>
        /// <returns>The most serious fault found among all the messages.</returns>
        internal static FaultAction ClassifyMessages(params string[] messages)
        {
            if (messages.Length == 0)
            {
                return FaultAction.Completed;
            }
            FaultAction action = FaultAction.Unknown;
            foreach (string message in messages)
            {
                switch (Classify(message))
                {
                    case FaultAction.PermanentFailure:
                        action = FaultAction.PermanentFailure;
                        break;

                    case FaultAction.TransientFailure:
                        if (action != FaultAction.PermanentFailure)
                        {
                            action = FaultAction.TransientFailure;
                        }
                        break;

                    case FaultAction.CompletedWithWarnings:
                        if (action != FaultAction.PermanentFailure
                            && action != FaultAction.TransientFailure)
                        {
                            action = FaultAction.CompletedWithWarnings;
                        }
                        break;

                    case FaultAction.Completed:
                        if (action != FaultAction.PermanentFailure
                            && action != FaultAction.TransientFailure
                            && action != FaultAction.CompletedWithWarnings)
                        {
                            action = FaultAction.Completed;
                        }
                        break;
                }
            }
            return action;
        }

        /// <summary>
        /// Converts a PCEHR fault action into a corresponding HIPS Response Indicator.
        /// </summary>
        /// <param name="action">The fault action.</param>
        /// <returns>The HIPS Response Indicator.</returns>
        internal static HipsResponseIndicator ConvertToHipsResponseIndicator(FaultAction action)
        {
            switch (action)
            {
                case FaultAction.Completed: return HipsResponseIndicator.OK;
                case FaultAction.CompletedWithWarnings: return HipsResponseIndicator.PcehrServiceWarning;
                case FaultAction.TransientFailure: return HipsResponseIndicator.PcehrServiceUnavailable;
                case FaultAction.PermanentFailure: return HipsResponseIndicator.PcehrServiceError;
                default: return HipsResponseIndicator.SystemError;
            }
        }

        #endregion Public Methods

        #region Private Methods

        /// <summary>
        /// Extracts the error code from the message and classifies it as either
        /// a successful completion with warnings, a transient failure that can
        /// be retried, or a permanent failure that needs to be investigated.
        /// </summary>
        /// <param name="message">A service message from the PCEHR system</param>
        /// <returns>The fault type</returns>
        private static FaultAction Classify(string message)
        {
            FaultAction action = FaultAction.Unknown;
            Match match = PcehrErrorCodeRegex.Match(message);

            // Try with NPDR error code pattern (MEDVIEW_REPOSITORY_nnn) too.
            if (!match.Success) match = NpdrErrorCodeRegex.Match(message);

            if (match.Success)
            {
                string code = match.Value;
                if (PCEHR_COMPLETED_WITH_WARNINGS_CODES.Contains(code))
                {
                    action = FaultAction.CompletedWithWarnings;
                }
                if (PCEHR_TRANSIENT_FAILURE_CODES.Contains(code))
                {
                    action = FaultAction.TransientFailure;
                }
                if (PCEHR_PERMANENT_FAILURE_CODES.Contains(code))
                {
                    action = FaultAction.PermanentFailure;
                }
            }
            else if (message.Contains(PcehrResponseStrings.RegistryErrorCodeContextDuplicateIdentifier))
            {
                action = FaultAction.CompletedWithWarnings;
            }
            else if (message.Contains(PcehrResponseStrings.HttpRequestForbidden))
            {
                action = FaultAction.PermanentFailure;
            }
            else if (message.Contains(PcehrResponseStrings.NoEndpointListening))
            {
                action = FaultAction.TransientFailure;
            }
            else if (PCEHR_SUCCESS_MESSAGES.Contains(message))
            {
                action = FaultAction.Completed;
            }

            return action;
        }

        #endregion Private Methods
    }
}