﻿using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Xml;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Configuration;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.PCEHR;
using Nehta.VendorLibrary.PCEHR.PCEHRProfile;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class GainPcehrAccess
    {
        /// <summary>
        /// Requests emergency access to the patient's records on the PCEHR
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <returns>GainPcehrAccessResponse</returns>
        public GainPcehrAccessResponse GainAccessEmergency(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            return GainAccess(patientIdentifier, user, true, false, null);
        }

        /// <summary>
        /// Request access with the access code being sent to the PCEHR
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="accessCode"></param>
        /// <param name="user"></param>
        /// <returns>GainPcehrAccessResponse</returns>
        public GainPcehrAccessResponse GainAccessWithCode(PatientIdentifierBase patientIdentifier, String accessCode, UserDetails user)
        {
            return GainAccess(patientIdentifier, user, false, true, accessCode);
        }

        /// <summary>
        /// Request for access without a code being required by PCEHR
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <returns>GainPcehrAccessResponse</returns>
        public GainPcehrAccessResponse GainAccessWithOutCode(PatientIdentifierBase patientIdentifier, UserDetails user)
        {
            return GainAccess(patientIdentifier, user, false, false, null);
        }

        /// <summary>
        /// Inserts an audit record for the Gain Access operation. If the code
        /// is provided, replaces it with the value of
        ///    XmlStringMap.AccessCodeRedacted
        /// in the SOAP request.
        /// </summary>
        /// <param name="user">The user who is attempting to gain access</param>
        /// <param name="hospital">The hospital that is attempting to gain access</param>
        /// <param name="patientMaster">The patient to whose PCEHR we are attempting to gain access</param>
        /// <param name="gainAccessClient">The client which has stored the request and response SOAP messages</param>
        /// <param name="auditOperationName">The name of the operation to audit (such as with code, without code or emergency)</param>
        /// <param name="response">The success or failure codes and details</param>
        private static bool InsertAccessAudit(UserDetails user, Hospital hospital, PatientMaster patientMaster, GainPCEHRAccessClient gainAccessClient, string auditOperationName, HipsResponse response)
        {
            if (auditOperationName == AuditOperationNames.GainAccessWithCode && !string.IsNullOrEmpty(gainAccessClient.SoapMessages.SoapRequest))
            {
                XmlDocument doc = new XmlDocument();
                doc.Load(new StringReader(gainAccessClient.SoapMessages.SoapRequest));
                XmlNamespaceManager xnm = new XmlNamespaceManager(new NameTable());
                xnm.AddNamespace("s", XmlStringMap.SoapEnvelopeNamespace);
                xnm.AddNamespace("p", XmlStringMap.PcehrProfileNamespace);
                XmlNode accessCodeNode = doc.SelectSingleNode(XmlStringMap.AccessCodeXPath, xnm);
                accessCodeNode.InnerText = XmlStringMap.AccessCodeRedacted;
                StringWriter writer = new StringWriter();
                doc.Save(writer);
                gainAccessClient.SoapMessages.SoapRequest = writer.ToString();
            }

            return Helpers.InsertAudit(patientMaster, user, hospital, auditOperationName, response, gainAccessClient.SoapMessages);
        }

        /// <summary>
        /// Private class that directs all gain access requests
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <param name="emergencyAccess"></param>
        /// <param name="codeRequired"></param>
        /// <param name="accessCode"></param>
        /// <returns>GainPcehrAccessResponse</returns>
        private GainPcehrAccessResponse GainAccess(PatientIdentifierBase patientIdentifier, UserDetails user, bool emergencyAccess, bool codeRequired, string accessCode)
        {
            GainPcehrAccessResponse response = new GainPcehrAccessResponse();

            string ihi;
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            PatientMaster patientMaster;

            //get hospital
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);

            if (status.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = status;
                return response;
            }

            //get patient
            HospitalPatient hospitalPatient;
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi)
            {
                response.HipsResponse = status;
                return response;
            }

            // If the IHI was last validated outside the configured period, attempt validation.
            IhiSearchResponse ihiResponse = new PatientIhiValidation().GetValidatedIhi(patientIdentifier, hospital, user, patientMaster);
            if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = ihiResponse.HipsResponse;
                return response;
            }

            // If the IHI is still invalid then HIPS will prevent access to the PCEHR.
            patientAccess.ValidateLocalIhiInformation(patientMaster, status);
            if (status.Status != HipsResponseIndicator.OK)
            {
                response.HipsResponse = status;
                return response;
            }

            //set IHI found
            ihi = patientMaster.Ihi;

            //populate user
            if (!User.PopulateAndValidateUser(hospital, user))
            {
                status.Status = HipsResponseIndicator.InvalidUser;
                response.HipsResponse = status;
                return response;
            }

            // Check that a code was provided if the type was Gain Access With Code.
            if (codeRequired && (string.IsNullOrEmpty(accessCode)))
            {
                //code is required but was not set
                HipsResponse noAccessCodeResponse = new HipsResponse(HipsResponseIndicator.InvalidAccessCode);
                noAccessCodeResponse.ResponseCodeDescription = "Access Code was expected, but not received.";
                response.HipsResponse = noAccessCodeResponse;
                return response;
            }

            X509Certificate2 certificate = Helpers.GetConnectionCertificate(hospital);
            Uri uri = Helpers.GainPCEHRAccessUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(patientIdentifier, ihi, user, hospital);

            // Instantiate the client
            GainPCEHRAccessClient gainAccessClient = new GainPCEHRAccessClient(uri, certificate, certificate);

            // Avoid the proxy
            System.Reflection.FieldInfo clientfield = gainAccessClient.GetType().GetField("profileClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            object ppc = clientfield.GetValue(gainAccessClient);
            System.Reflection.FieldInfo clientField2 = ppc.GetType().GetField("pcehrProfileClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            PCEHRProfilePortTypeClient ptc = clientField2.GetValue(ppc) as PCEHRProfilePortTypeClient;
            CustomBinding binding = ptc.Endpoint.Binding as CustomBinding;
            HttpsTransportBindingElement https = binding.Elements.First(a => a.GetType() == typeof(HttpsTransportBindingElement)) as HttpsTransportBindingElement;

            //Set the connection timeout for the GetDocument service
            binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
            binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);
            binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.DocumentConsumptionTimeoutSeconds);

            // Avoid the proxy
            if (Settings.Instance.AvoidProxy)
            {
                https.UseDefaultWebProxy = false;
            }

            // Add server certificate validation callback
            ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

            // Create a request
            gainPCEHRAccessPCEHRRecord accessRequest = new gainPCEHRAccessPCEHRRecord();
            accessRequest.authorisationDetails = new gainPCEHRAccessPCEHRRecordAuthorisationDetails();
            string auditOperationName;

            //if emergency access is required then set the access type to EmergencyAccess
            if (emergencyAccess)
            {
                accessRequest.authorisationDetails.accessType = gainPCEHRAccessPCEHRRecordAuthorisationDetailsAccessType.EmergencyAccess;
                auditOperationName = AuditOperationNames.GainEmergencyAccess;
            }

            //if a code string is required then the access type must be set to expect an access code and the access code must be sent in
            else if (codeRequired)
            {
                accessRequest.authorisationDetails.accessType = gainPCEHRAccessPCEHRRecordAuthorisationDetailsAccessType.AccessCode;
                accessRequest.authorisationDetails.accessCode = accessCode;
                auditOperationName = AuditOperationNames.GainAccessWithCode;
            }
            else
            {
                // PCEHR NOC requires that authorisationDetails is not provided when gaining access without a code.
                accessRequest.authorisationDetails = null;
                auditOperationName = AuditOperationNames.GainAccessWithoutCode;
            }

            gainPCEHRAccessResponseIndividual individual = new gainPCEHRAccessResponseIndividual();

            try
            {
                // Invoke the service
                responseStatusType responseStatus = gainAccessClient.GainPCEHRAccess(header, accessRequest, out individual);
                response.IHINumber = ihi;

                //check if there are no errors returned from the service
                if (responseStatus.code == "PCEHR_SUCCESS")
                {
                    response.IhiRecordStatus = Helpers.MaptoHIPSIhiRecordStatus(individual.ihiRecordStatus);
                    response.IhiStatus = Helpers.MaptoHIPSIhiStatus(individual.ihiStatus);
                    response.AccessPermission = GainPcehrAccessStatus.Permit;

                    //all has succeeded so add in the HIPSResponseObject
                    response.HipsResponse = new HipsResponse(HipsResponseIndicator.OK);
                }
                else //parse for the errors
                {
                    //fail has occurred so add in the HIPSResponseObject
                    response.HipsResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                    response.HipsResponse.ResponseCode = responseStatus.code;
                    response.HipsResponse.ResponseCodeDescription = responseStatus.description;
                    response.HipsResponse.ResponseCodeDetails = responseStatus.details;
                }
            }
            catch (FaultException fe)
            {
                HipsResponse faultResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError);
                FaultException<StandardErrorType> standardErrorType = fe as FaultException<StandardErrorType>;
                if (standardErrorType != null)
                {
                    faultResponse.ResponseCode = standardErrorType.Detail.errorCode.ToString();
                    faultResponse.ResponseCodeDescription = standardErrorType.Detail.message;
                    faultResponse.ResponseCodeDetails = standardErrorType.StackTrace;
                }
                else
                {
                    faultResponse.ResponseCode = fe.Reason.ToString();
                    faultResponse.ResponseCodeDescription = fe.Message;
                    faultResponse.ResponseCodeDetails = fe.StackTrace;
                }
                response.HipsResponse = faultResponse;
            }
            catch (Exception e)
            {
                HipsResponse exceptionResponse = new HipsResponse(HipsResponseIndicator.SystemError);
                exceptionResponse.HipsErrorMessage = e.Message;

                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    exceptionResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                exceptionResponse.ResponseCodeDetails = e.StackTrace;
                response.HipsResponse = exceptionResponse;
            }
            finally
            {
                gainAccessClient.Close();
            }

            InsertAccessAudit(user, hospital, patientMaster, gainAccessClient, auditOperationName, response.HipsResponse);

            return response;
        }

        /// <summary>
        /// Validator for the Service Certificate
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="certificate"></param>
        /// <param name="chain"></param>
        /// <param name="sslPolicyErrors"></param>
        /// <returns></returns>
        private bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return DocumentHelper.ValidateServiceCertificate(sender, certificate, chain, sslPolicyErrors);
        }
    }
}