﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Data.SqlTypes;
using System.Linq;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.HibIntegration.HL7.Message;
using HIPS.HibIntegration.HL7.Segment;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using Enums = HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.HibIntegration.Loader
{
    public class EpisodeLoader
    {
        private static readonly int UnknownForeignKey = -1;
        private SqlTransaction transaction;
        private UserDetails user;

        public EpisodeLoader(UserDetails user, SqlTransaction transaction)
        {
            this.transaction = transaction;
            this.user = user;
        }

        /// <summary>
        /// Creates or updates an episode using the information from a PAS message.
        /// </summary>
        /// <param name="patient">The hospital patient</param>
        /// <param name="message">The PAS message</param>
        /// <param name="episodeCreated">Whether the episode was new to HIPS</param>
        /// <returns>The episode that was created or updated.</returns>
        public Episode Load(HospitalPatient patient, HL7GenericPasMessage message, out bool episodeCreated)
        {
            MSH msh = message.MessageHeader;
            PV1 pv1 = message.PatientVisit;
            PV2 pv2 = message.PatientVisitAdditional;
            SCH sch = message.Schedule;
            ListSingleton lists = ListSingleton.Instance;
            EpisodeDl dataAccess = new EpisodeDl(user);

            //All Episode LifeCycle Types can be used in this case
            List<Episode> episodes = dataAccess.GetAll(patient.PatientId, null);
            string episodeNumber = pv1.VisitNumber.ID;
            Episode episode = episodes.Where(a => a.SourceSystemEpisodeId.Equals(episodeNumber)).FirstOrDefault();

            //if episode cannot be found then a new one is created with an Unknown Episode Life Cycle
            if (episode == null)
            {
                episode = new Episode();
                episode.PatientId = patient.PatientId.Value;
                episode.SourceSystemEpisodeId = episodeNumber;
                episode.EpisodeLifecycleId = (int)Enums.EpisodeLifecycle.UnknownAppointmentStatus;
            }

            Episode originalEpisode = CloneEpisode(episode);

            // Load the admission date/time if it is provided in this message
            DateTime? admissionDate = pv1.AdmitDateTime == null ? null : pv1.AdmitDateTime.TimestampValue;
            if (admissionDate.HasValue)
            {
                episode.AdmissionDate = admissionDate.Value;
            }
            if (episode.AdmissionDate < SqlDateTime.MinValue.Value || episode.AdmissionDate > SqlDateTime.MaxValue.Value)
            {
                // Either the admission date was not given or was outside the
                // range that SQL Server can store (01/01/1753 - 31/12/9999).
                // Episodes without an admission date are typically where the patient is
                // placed on a waiting list for elective surgery. In this case HIPS will
                // store the admission date as SqlDateTime.MaxValue (31/12/9999).
                episode.AdmissionDate = SqlDateTime.MaxValue.Value;
            }

            // Load the discharge date/time if it is provided in this message
            // If transfer messages come after discharge message, avoid clearing the discharge date.
            DateTime? dischargeDate = pv1.DischargeDateTime == null ? null : pv1.DischargeDateTime.TimestampValue;
            if (dischargeDate.HasValue)
            {
                episode.DischargeDate = dischargeDate;
            }
            if (episode.DischargeDate.HasValue
                && (episode.DischargeDate < SqlDateTime.MinValue.Value
                    || episode.DischargeDate > SqlDateTime.MaxValue.Value))
            {
                // The discharge date was outside the range that SQL server can
                // store (01/01/1753 - 31/12/9999). The episode will be stored
                // with a null discharge date.
                episode.DischargeDate = null;
            }
            if (pv2 != null && pv2.AdmitReason != null)
            {
                episode.AdmissionReason = pv2.AdmitReason.text;
            }

            // Load the assigned patient location if it is provided in this message
            if (pv1.AssignedPatientLocation != null && !string.IsNullOrEmpty(pv1.AssignedPatientLocation.pointofcare))
            {
                episode.Ward = pv1.AssignedPatientLocation.pointofcare;
                episode.Room = pv1.AssignedPatientLocation.room;
                episode.Bed = pv1.AssignedPatientLocation.bed;
            }

            LoadResponsibleProvider(patient.HospitalId, pv1, episode);
            LoadEpisodeLifecycle(msh, sch, episode);
            string episodeTypeCode = pv1.PatientClass.Split(new char[] { '^' })[0];
            EpisodeType episodeType = lists.AllEpisodeTypes.Where(a => a.Code.Equals(episodeTypeCode)).FirstOrDefault();
            episode.EpisodeTypeId = episodeType == null ? UnknownForeignKey : episodeType.EpisodeTypeId.Value;
            if (episode.EpisodeId.HasValue)
            {
                if (WasEpisodeChanged(originalEpisode, episode))
                {
                    if (!dataAccess.Update(episode, transaction))
                    {
                        throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, dataAccess.GetType().FullName));
                    }
                }
                episodeCreated = false;
            }
            else
            {
                if (!dataAccess.Insert(episode, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, dataAccess.GetType().FullName));
                }

                episodeCreated = true;
            }
            return episode;
        }

        /// <summary>
        /// Updates the episode lifecycle status based on the outpatient event reason or inpatient message trigger event.
        /// </summary>
        /// <param name="msh">Message header segment</param>
        /// <param name="sch">Outpatient schedule segment</param>
        /// <param name="episode">Episode schema object</param>
        private static void LoadEpisodeLifecycle(MSH msh, SCH sch, Episode episode)
        {
            Enums.EpisodeLifecycle lifecycle = Enums.EpisodeLifecycle.UnknownAppointmentStatus;
            if (sch != null && sch.EventReason != null)
            {
                // Outpatient Event Reasons
                switch (sch.EventReason.identifier)
                {
                    case "BK": lifecycle = Enums.EpisodeLifecycle.Booked; break;
                    case "AS": lifecycle = Enums.EpisodeLifecycle.AttendedSeen; break;
                    case "DE": lifecycle = Enums.EpisodeLifecycle.Deleted; break;
                    case "CP": lifecycle = Enums.EpisodeLifecycle.CancelledByPatient; break;
                    case "CH": lifecycle = Enums.EpisodeLifecycle.CancelledByHospital; break;
                    case "CO": lifecycle = Enums.EpisodeLifecycle.CancelledOther; break;
                    case "FT": lifecycle = Enums.EpisodeLifecycle.FailedToAttend; break;
                    case "U": lifecycle = Enums.EpisodeLifecycle.UnknownAppointmentStatus; break;
                }
            }
            else
            {
                string trigger = msh.MessageType.triggerevent;

                // Inpatient Admission Message Trigger Events
                if (trigger == TriggerEventResource.Admit)                // A01 Admit patient
                {
                    lifecycle = Enums.EpisodeLifecycle.Admitted;
                }
                else if (trigger == TriggerEventResource.CancelAdmit)     // A11 Cancel admission
                {
                    lifecycle = Enums.EpisodeLifecycle.CancelledAdmission;
                }
                else if (trigger == TriggerEventResource.PreAdmit)        // A05 Pre-admit patient
                {
                    lifecycle = Enums.EpisodeLifecycle.PreAdmit;
                }
                else if (trigger == TriggerEventResource.CancelPreAdmit)  // A38 Cancel Pre-admit
                {
                    lifecycle = Enums.EpisodeLifecycle.CancelledPreAdmit;
                }
                else if (trigger == TriggerEventResource.Discharge)       // A03 Discharge patient
                {
                    lifecycle = Enums.EpisodeLifecycle.Discharged;
                }
                else if (trigger == TriggerEventResource.CancelDischarge) // A13 Cancel Discharge (return to Admitted status)
                {
                    lifecycle = Enums.EpisodeLifecycle.Admitted;
                }
            }
            if (lifecycle != Enums.EpisodeLifecycle.UnknownAppointmentStatus)
            {
                episode.EpisodeLifecycleId = (int)lifecycle;
            }

            // If HIPS did not receive the A01, A05 or A03 but only some A02 or A08 messages for this episode,
            // it will infer the lifecycle status by comparing the admission and discharge times to the current
            // date and time.
            if (episode.EpisodeLifecycleId == (int)Enums.EpisodeLifecycle.UnknownAppointmentStatus)
            {
                if (episode.AdmissionDate < DateTime.Now && (!episode.DischargeDate.HasValue || episode.DischargeDate.Value > DateTime.Now))
                {
                    // Admitted because admission time is in the past and no discharge or the discharge time is in the future
                    episode.EpisodeLifecycleId = (int)Enums.EpisodeLifecycle.Admitted;
                }
                else if (episode.AdmissionDate > DateTime.Now)
                {
                    // Pre-Admit because admission time is in the future
                    episode.EpisodeLifecycleId = (int)Enums.EpisodeLifecycle.PreAdmit;
                }
                else if (episode.DischargeDate.HasValue && episode.DischargeDate.Value < DateTime.Now)
                {
                    // Discharged because discharge time is in the past
                    episode.EpisodeLifecycleId = (int)Enums.EpisodeLifecycle.Discharged;
                }
            }
        }

        /// <summary>
        /// Makes a new Episode object with copies of the key fields that
        /// we need to check to determine whether a change was made that needs to be
        /// applied in the database: AdmissionDate, DischargeDate, AdmissionReason,
        /// ResponsibleProviderId, EpisodeLifecycleId and EpisodeTypeId.
        /// </summary>
        /// <param name="original">The health provider object obtained from the database.</param>
        /// <returns>A new object with copies of the key fields of the supplied object.</returns>
        private Episode CloneEpisode(Episode original)
        {
            Episode clone = new Episode();
            clone.AdmissionDate = original.AdmissionDate;
            clone.DischargeDate = original.DischargeDate;
            clone.AdmissionReason = original.AdmissionReason;
            clone.ResponsibleProviderId = original.ResponsibleProviderId;
            clone.EpisodeLifecycleId = original.EpisodeLifecycleId;
            clone.EpisodeTypeId = original.EpisodeTypeId;
            clone.Ward = original.Ward;
            clone.Room = original.Room;
            clone.Bed = original.Bed;
            return clone;
        }

        /// <summary>
        /// Assigns the responsible provider to the episode.
        /// The provider information is added or updated.
        /// The responsible provider is the Attending Doctor,
        /// or if none then the Attending Doctor, or if none then unknown.
        /// </summary>
        /// <param name="hospitalId">The hospital ID</param>
        /// <param name="pv1">The patient visit segment of the HL7 message</param>
        /// <param name="episode">The episode</param>
        private void LoadResponsibleProvider(int hospitalId, PV1 pv1, Episode episode)
        {
            if (!string.IsNullOrEmpty(pv1.AttendingDoctor[0].IDnumberST))
            {
                HealthProviderIndividualLoader providerLoader = new HealthProviderIndividualLoader(user, transaction);
                HealthProviderIndividual provider = providerLoader.Load(hospitalId, pv1.AttendingDoctor[0]);
                episode.ResponsibleProviderId = provider.HealthProviderIndividualId.Value;
            }
            else if (!string.IsNullOrEmpty(pv1.AdmittingDoctor[0].IDnumberST))
            {
                HealthProviderIndividualLoader providerLoader = new HealthProviderIndividualLoader(user, transaction);
                HealthProviderIndividual provider = providerLoader.Load(hospitalId, pv1.AdmittingDoctor[0]);
                episode.ResponsibleProviderId = provider.HealthProviderIndividualId.Value;
            }
            else
            {
                episode.ResponsibleProviderId = UnknownForeignKey;
            }
        }

        /// <summary>
        /// Determines whether a change was made to one of the key fields that determine whether
        /// the Episode needs to be updated in the database: AdmissionDate, DischargeDate,
        /// AdmissionReason, ResponsibleProviderId, EpisodeLifecycleId or EpisodeTypeId.
        /// </summary>
        /// <param name="originalEpisode">The object returned from the CloneEpisode method.</param>
        /// <param name="changedEpisode">The object obtained from the database, after populating information from the HL7 message.</param>
        /// <returns>Whether the object needs to be saved.</returns>
        private bool WasEpisodeChanged(Episode originalEpisode, Episode changedEpisode)
        {
            return originalEpisode.AdmissionDate != changedEpisode.AdmissionDate
                || originalEpisode.DischargeDate != changedEpisode.DischargeDate
                || originalEpisode.AdmissionReason != changedEpisode.AdmissionReason
                || originalEpisode.ResponsibleProviderId != changedEpisode.ResponsibleProviderId
                || originalEpisode.EpisodeLifecycleId != changedEpisode.EpisodeLifecycleId
                || originalEpisode.EpisodeTypeId != changedEpisode.EpisodeTypeId
                || originalEpisode.Ward != changedEpisode.Ward
                || originalEpisode.Room != changedEpisode.Room
                || originalEpisode.Bed != changedEpisode.Bed;
        }
    }
}