﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the ClinicalDocument table
    /// </summary>
    public class ClinicalDocumentDl : BaseDl
    {
        #region Constructors

        public ClinicalDocumentDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets the clinical document with the specified episode ID and source system set ID.
        /// </summary>
        /// <param name="episodeId">The episode ID.</param>
        /// <param name="sourceSystemSetId">The source system set ID</param>
        /// <returns>The clinical document</returns>
        public ClinicalDocument Get(int episodeId, string sourceSystemSetId)
        {
            ClinicalDocument result = new ClinicalDocument();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentGet"))
                {
                    command.Parameters.Add(new SqlParameter("@EpisodeId", episodeId));
                    command.Parameters.Add(new SqlParameter("@SourceSystemSetId", sourceSystemSetId));
                    PopulateBusinessObject<ClinicalDocument>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentGet, sourceSystemSetId, episodeId), ex, User, LogMessage.HIPS_MESSAGE_037);
            }
            return result;
        }

        /// <summary>
        /// Gets the clinical documents for the specified episode and optionally filter by document type.
        /// </summary>
        /// <param name="episodeId">The episode id.</param>
        /// <param name="documentTypeId">The document type</param>
        /// <returns>List of clinical documents</returns>
        public List<ClinicalDocument> GetAll(int episodeId, int? documentTypeId)
        {
            List<ClinicalDocument> result = new List<ClinicalDocument>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentGet"))
                {
                    AddIntParameter("EpisodeId", episodeId, command);
                    AddIntParameter("DocumentTypeId", documentTypeId, command);
                    result = GetPopulatedBusinessList<ClinicalDocument>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentGetAll, episodeId, documentTypeId), ex, User, LogMessage.HIPS_MESSAGE_036);
            }
            return result;
        }

        /// <summary>
        /// Gets the clinical document with the specified source system set ID.
        /// Reminder: ensure the document is attached to the right episode / patient.
        /// </summary>
        /// <param name="sourceSystemSetId">The source system set ID</param>
        /// <returns>The clinical document</returns>
        public ClinicalDocument GetBySetId(string sourceSystemSetId)
        {
            ClinicalDocument result = new ClinicalDocument();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentGet"))
                {
                    command.Parameters.Add(new SqlParameter("@SourceSystemSetId", sourceSystemSetId));
                    PopulateBusinessObject<ClinicalDocument>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentGetBySetId, sourceSystemSetId), ex, User, LogMessage.HIPS_MESSAGE_037);
            }
            return result;
        }

        /// <summary>
        /// Gets a clinical documents uploaded for the specified hospital patient from the local repository
        /// If the SourceSystemDocumentId is provided then the specific version of the document
        /// will be returned, otherwise it is only the latest uploaded
        /// </summary>
        /// <param name="patientId">The patient identifier.</param>
        /// <param name="sourceSystemSetId">The source system set identifier.</param>
        /// <returns>
        /// List of clinical documents
        /// </returns>
        public HipsResponse GetLocalUploadedDocument(int patientId, out LocalClinicalDocument result, string sourceSystemSetId, string sourceSystemDocumentId = null)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            result = new LocalClinicalDocument();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentVersionLocalGet"))
                {
                    AddIntParameter("PatientId", patientId, command);
                    AddStringParameter("SourceSystemSetId", sourceSystemSetId, command);
                    if (!string.IsNullOrEmpty(sourceSystemDocumentId))
                    {
                        AddStringParameter("SourceSystemDocumentId", sourceSystemDocumentId, command);
                    }

                    LocalClinicalDocumentMetaData metaData = new LocalClinicalDocumentMetaData();
                    byte[] package = null;

                    SqlDataReader reader = command.ExecuteReader();
                    if (reader.HasRows)
                    {
                        reader.Read();
                        PopulateObject<LocalClinicalDocumentMetaData>(reader, metaData);
                        package = (byte[])reader.GetValue(reader.GetOrdinal("Package"));
                    }
                    reader.Close();
                    command.Connection.Close();
                    result.LocalClinicalDocumentMetaData = metaData;
                    result.Package = package;
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentLocalVersionGet, patientId, sourceSystemSetId), ex, User, LogMessage.HIPS_MESSAGE_161);
                response.Status = HipsResponseIndicator.DatabaseError;
                response.HipsErrorMessage = string.Format(ConstantsResource.ErrorMessageClinicalDocumentLocalVersionGet, patientId, sourceSystemSetId);
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            return response;
        }

        /// <summary>
        /// Gets the clinical documents uploaded for the specified hospital patient from the local repository
        /// If the SourceSystemSetId is provided then all the versions of the document
        /// will be returned, otherwise it is only the latest uploaded document
        /// </summary>
        /// <param name="patientId">The patient identifier.</param>
        /// <param name="results">The results.</param>
        /// <param name="sourceSystemSetId">The source system set identifier.</param>
        /// <param name="documentTypeCode">The document type code.</param>
        /// <returns>
        /// List of clinical documents
        /// </returns>
        public HipsResponse ListLocalUploadedDocuments(int patientId, out List<LocalClinicalDocumentMetaData> results, string sourceSystemSetId = null, string documentTypeCode = null)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            results = new List<LocalClinicalDocumentMetaData>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentUploadList"))
                {
                    AddIntParameter("PatientId", patientId, command);
                    if (!string.IsNullOrEmpty(sourceSystemSetId))
                    {
                        AddStringParameter("SourceSystemSetId", sourceSystemSetId, command);
                        AddBoolParameter("CurrentOnly", false, command);
                    }

                    if (!string.IsNullOrEmpty(documentTypeCode))
                    {
                        AddStringParameter("DocumentTypeCode", documentTypeCode, command);
                    }

                    results = GetPopulatedBusinessList<LocalClinicalDocumentMetaData>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentList, patientId), ex, User, LogMessage.HIPS_MESSAGE_160);
                response.Status = HipsResponseIndicator.DatabaseError;
                response.HipsErrorMessage = string.Format(ConstantsResource.ErrorMessageClinicalDocumentList, patientId);
                response.ResponseCode = ex.GetType().Name;
                response.ResponseCodeDescription = ex.Message;
            }
            return response;
        }

        /// <summary>
        /// Saves the specified clinical document.
        /// </summary>
        /// <param name="ClinicalDocument">The ClinicalDocument.</param>
        /// <returns></returns>
        public bool Save(ClinicalDocument ClinicalDocument, SqlTransaction transaction = null)
        {
            if (ClinicalDocument.ClinicalDocumentId.HasValue && ClinicalDocument.ClinicalDocumentId > -1)
            {
                return Update(ClinicalDocument, transaction);
            }
            return Insert(ClinicalDocument, transaction);
        }

        /// <summary>
        /// Inserts the specified ClinicalDocument.
        /// </summary>
        /// <param name="ClinicalDocument">The ClinicalDocument.</param>
        /// <returns></returns>
        private bool Insert(ClinicalDocument ClinicalDocument, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentInsert", transaction))
            {
                return base.Insert<ClinicalDocument>(ClinicalDocument, command);
            }
        }

        /// <summary>
        /// Updates the specified ClinicalDocument.
        /// </summary>
        /// <param name="ClinicalDocument">The ClinicalDocument.</param>
        /// <returns></returns>
        private bool Update(ClinicalDocument ClinicalDocument, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentUpdate", transaction))
            {
                return base.Update<ClinicalDocument>(ClinicalDocument, command);
            }
        }

        #endregion Methods
    }
}