﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the health provider organisation patient table
    /// </summary>
    public class HealthProviderOrganisationPatientDl : BaseDl
    {
        #region Constructors

        public HealthProviderOrganisationPatientDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Deletes the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Delete(HealthProviderOrganisationPatient item)
        {
            using (SqlCommand command = GetSqlCommand("hips.HealthProviderOrganisationPatientDelete"))
            {
                return base.Delete<HealthProviderOrganisationPatient>(item, command);
            }
        }

        /// <summary>
        /// Gets the specified health provider organisation patient by HPI-O and patient master ID.
        /// </summary>
        /// <param name="hpiO">The HPI-O.</param>
        /// <param name="patientMasterId">The patient master ID.</param>
        /// <param name="item">The item.</param>
        /// <returns>Whether a hospital patient record was fetched</returns>
        public bool Get(string hpiO, int patientMasterId, out HealthProviderOrganisationPatient item)
        {
            item = new HealthProviderOrganisationPatient();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HealthProviderOrganisationPatientGet"))
                {
                    AddStringParameter("HpiO", hpiO, command);
                    AddIntParameter("PatientMasterId", patientMasterId, command);
                    result = PopulateBusinessObject<HealthProviderOrganisationPatient>(command.ExecuteReader(), item);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessageHealthProviderOrganisationPatientGet, hpiO, patientMasterId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_048);
            }
            return result;
        }

        /// <summary>
        /// Gets all health provider organisation patients updated since the given date/time.
        /// </summary>
        /// <param name="lastUpdated">The date/time.</param>
        /// <returns>List of health provider organisation patients</returns>
        public List<HealthProviderOrganisationPatient> GetAll(DateTime lastUpdated)
        {
            List<HealthProviderOrganisationPatient> results = new List<HealthProviderOrganisationPatient>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HealthProviderOrganisationPatientGet"))
                {
                    command.Parameters.Add(new SqlParameter("@DateModified", lastUpdated));
                    results = GetPopulatedBusinessList<HealthProviderOrganisationPatient>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessageHealthProviderOrganisationPatientGetAll, lastUpdated);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_049);
            }
            return results;
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Insert(HealthProviderOrganisationPatient item)
        {
            using (SqlCommand command = GetSqlCommand("hips.HealthProviderOrganisationPatientInsert"))
            {
                return base.Insert<HealthProviderOrganisationPatient>(item, command);
            }
        }

        /// <summary>
        /// Updates the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Update(HealthProviderOrganisationPatient item)
        {
            using (SqlCommand command = GetSqlCommand("hips.HealthProviderOrganisationPatientUpdate"))
            {
                return base.Update<HealthProviderOrganisationPatient>(item, command);
            }
        }

        #endregion Methods
    }
}