﻿using System;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the patient master table
    /// </summary>
    public class PatientMasterDl : BaseDl
    {
        #region Constructors

        public PatientMasterDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Returns name of stored procedure for retrieving information from patient master table.
        /// If PUMA is in use this stored procedure will coalesce the information from the PUMA
        /// Patient Master Override table.
        /// </summary>
        private string PatientMasterGet
        {
            get
            {
                if (Settings.Instance.PatientMasterOverride)
                {
                    return "puma.PatientMasterGet";
                }
                else
                {
                    return "hips.PatientMasterGet";
                }
            }
        }

        /// <summary>
        /// Returns name of stored procedure for updating information in patient master table.
        /// If PUMA is in use this stored procedure will remove overrides from the PUMA
        /// Patient Master Override table once the information from PAS matches the override.
        /// </summary>
        private string PatientMasterUpdate
        {
            get
            {
                if (Settings.Instance.PatientMasterOverride)
                {
                    return "puma.PatientMasterUpdate";
                }
                else
                {
                    return "hips.PatientMasterUpdate";
                }
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Gets the specified patient master record by ID.
        /// </summary>
        /// <param name="patientMasterId">The patient master ID.</param>
        /// <param name="item">The item.</param>
        /// <param name="transaction">The existing transaction (optional)</param>
        /// <returns>Indicator that patient was found (OK), not found (InvalidPatient) or error (DatabaseError).</returns>
        public HipsResponse Get(int patientMasterId, int healthProviderOrganisationNetworkId, out PatientMaster item, SqlTransaction transaction = null)
        {
            item = new PatientMaster();
            HipsResponse result = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                using (SqlCommand command = GetSqlCommand(PatientMasterGet, transaction))
                {
                    command.Parameters.Add(new SqlParameter("@PatientMasterId", patientMasterId));
                    command.Parameters.Add(new SqlParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId));
                    if (PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), item))
                    {
                        AddressDl addressDl = new AddressDl();
                        item.Addresses = addressDl.GetAllByPatient(item.PatientMasterId.Value, transaction);
                        ContactDl contactDl = new ContactDl();
                        item.Contacts = contactDl.GetAllByPatient(item.PatientMasterId.Value, transaction);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        item.Names = nameDl.GetAllByPatient(item.PatientMasterId.Value, transaction);
                    }
                    else
                    {
                        result.Status = HipsResponseIndicator.InvalidPatient;
                    }
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterGetById, patientMasterId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_059);
                result.Status = HipsResponseIndicator.DatabaseError;
                result.HipsErrorMessage = message;
                result.ResponseCodeDescription = ex.Message;
            }
            return result;
        }

        /// <summary>
        /// Gets the specified patient based on the demographic details passed in.
        /// </summary>
        /// <param name="currentLastName">Current last name of the patient.</param>
        /// <param name="currentFirstNames">Current first names of the patient.</param>
        /// <param name="dateOfBirth">The date of birth.</param>
        /// <param name="sexID">The sex ID.</param>
        /// <param name="medicareNumber">The medicare card number.</param>
        /// <param name="medicareIrn">The medicare individual reference number.</param>
        /// <param name="dvaNumber">The DVA file number.</param>
        /// <param name="item">The found patient master item.</param>
        /// <returns>Whether a match was found</returns>
        public bool GetByDemographics(string currentLastName, string currentFirstNames, DateTime dateOfBirth, int sexID, string medicareNumber, string medicareIrn, string dvaNumber, out PatientMaster item)
        {
            item = new PatientMaster();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand(PatientMasterGet))
                {
                    AddStringParameter("@CurrentLastName", currentLastName, command);
                    AddStringParameter("@CurrentFirstNames", currentFirstNames, command);
                    AddStringParameter("@MedicareNumber", medicareNumber, command);
                    AddStringParameter("@MedicareIrn", medicareIrn, command);
                    AddStringParameter("@DvaNumber", dvaNumber, command);
                    AddIntParameter("@CurrentSexId", sexID, command);
                    AddDateParameter("@DateOfBirth", dateOfBirth, command);
                    result = PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), item);
                    if (item.PatientMasterId.HasValue)
                    {
                        AddressDl addressDl = new AddressDl();
                        item.Addresses = addressDl.GetAllByPatient(item.PatientMasterId.Value);
                        ContactDl contactDl = new ContactDl();
                        item.Contacts = contactDl.GetAllByPatient(item.PatientMasterId.Value);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        item.Names = nameDl.GetAllByPatient(item.PatientMasterId.Value);
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterGetByDemographics,
                    currentLastName, currentFirstNames, dateOfBirth, sexID, medicareNumber, medicareIrn, dvaNumber);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_056);
            }
            return result;
        }

        /// <summary>
        /// Gets the specified patient based on the demographic details passed in for the selected hospital.
        /// </summary>
        /// <param name="currentLastName">Current last name of the patient.</param>
        /// <param name="currentFirstNames">Current first names of the patient.</param>
        /// <param name="dateOfBirth">The date of birth.</param>
        /// <param name="sexID">The sex ID.</param>
        /// <param name="medicareNumber">The medicare card number.</param>
        /// <param name="medicareIrn">The medicare individual reference number.</param>
        /// <param name="dvaNumber">The DVA file number.</param>
        /// <param name="hospital">The hospital to search within.</param>
        /// <param name="item">The found patient master item.</param>
        /// <returns>Whether a match was found</returns>
        public bool GetByDemographicsWithinHospital(string currentLastName, string currentFirstNames, DateTime dateOfBirth, int sexID, string medicareNumber, string medicareIrn, string dvaNumber, Hospital hospital, out PatientMaster item)
        {
            item = new PatientMaster();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PatientMasterDemographicsByHospitalGet"))
                {
                    AddStringParameter("@CurrentLastName", currentLastName, command);
                    AddStringParameter("@CurrentFirstNames", currentFirstNames, command);
                    AddStringParameter("@MedicareNumber", medicareNumber, command);
                    AddStringParameter("@MedicareIrn", medicareIrn, command);
                    AddStringParameter("@DvaNumber", dvaNumber, command);
                    AddIntParameter("@CurrentSexId", sexID, command);
                    AddDateParameter("@DateOfBirth", dateOfBirth, command);
                    AddIntParameter("@HospitalId", hospital.HospitalId, command);
                    result = PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), item);
                    if (item.PatientMasterId.HasValue)
                    {
                        AddressDl addressDl = new AddressDl();
                        item.Addresses = addressDl.GetAllByPatient(item.PatientMasterId.Value);
                        ContactDl contactDl = new ContactDl();
                        item.Contacts = contactDl.GetAllByPatient(item.PatientMasterId.Value);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        item.Names = nameDl.GetAllByPatient(item.PatientMasterId.Value);
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterGetByDemographics,
                    currentLastName, currentFirstNames, dateOfBirth, sexID, medicareNumber, medicareIrn, dvaNumber);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_056);
            }
            return result;
        }

        /// <summary>
        /// Gets the specified patient based on the hospital code and MRN.
        /// </summary>
        /// <param name="hospitalID">The hospital ID.</param>
        /// <param name="mrn">The Mrn.</param>
        /// <param name="codeSystemId">The namespace id.</param>
        /// <param name="currentLastName">Last name of the current.</param>
        /// <param name="currentFirstNames">The current first names.</param>
        /// <param name="dateOfBirth">The date of birth.</param>
        /// <param name="sexID">The sex ID.</param>
        /// <param name="medicareNumber">The medicare number.</param>
        /// <param name="medicareIrn">The medicare number sequence.</param>
        /// <param name="dvaNumber">The dva number.</param>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool GetByHospitalCodeMrn(string mrn, string hospitalCode, int codeSystemId, int healthProviderOrganisationNetworkId, out PatientMaster item)
        {
            item = new PatientMaster();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand(PatientMasterGet))
                {
                    AddStringParameter("@Mrn", mrn, command);
                    AddIntParameter("@CodeSystemId", codeSystemId, command);
                    AddStringParameter("@HospitalCode", hospitalCode, command);
                    AddIntParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId, command);
                    result = PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), item);
                    if (item.PatientMasterId.HasValue)
                    {
                        AddressDl addressDl = new AddressDl();
                        item.Addresses = addressDl.GetAllByPatient(item.PatientMasterId.Value);
                        ContactDl contactDl = new ContactDl();
                        item.Contacts = contactDl.GetAllByPatient(item.PatientMasterId.Value);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        item.Names = nameDl.GetAllByPatient(item.PatientMasterId.Value);
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterGetByHospitalCodeMrn,
                    mrn, hospitalCode, codeSystemId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_054);
            }
            return result;
        }

        /// <summary>
        /// Gets the specified patient based on the hospital ID and MRN.
        /// </summary>
        /// <param name="hospitalID">The hospital ID.</param>
        /// <param name="mrn">The Mrn.</param>
        /// <param name="codeSystemId">The namespace id.</param>
        /// <param name="currentLastName">Last name of the current.</param>
        /// <param name="currentFirstNames">The current first names.</param>
        /// <param name="dateOfBirth">The date of birth.</param>
        /// <param name="sexID">The sex ID.</param>
        /// <param name="medicareNumber">The medicare number.</param>
        /// <param name="medicareIrn">The medicare number sequence.</param>
        /// <param name="dvaNumber">The dva number.</param>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool GetByHospitalIdMrn(int hospitalId, string mrn, out PatientMaster item)
        {
            item = new PatientMaster();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand(PatientMasterGet))
                {
                    AddIntParameter("@HospitalId", hospitalId, command);
                    AddStringParameter("@Mrn", mrn, command);
                    result = PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), item);
                    if (item.PatientMasterId.HasValue)
                    {
                        AddressDl addressDl = new AddressDl();
                        item.Addresses = addressDl.GetAllByPatient(item.PatientMasterId.Value);
                        ContactDl contactDl = new ContactDl();
                        item.Contacts = contactDl.GetAllByPatient(item.PatientMasterId.Value);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        item.Names = nameDl.GetAllByPatient(item.PatientMasterId.Value);
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterGetByHospitalIdMrn,
                    mrn, hospitalId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_055);
            }
            return result;
        }

        /// <summary>
        /// Gets the specified patient based on the IHI.
        /// </summary>
        /// <param name="ihi">The IHI</param>
        /// <param name="item">The item.</param>
        /// <returns>Whether a patient was found</returns>
        public bool GetByIhi(string ihi, int healthProviderOrganisationNetworkId, out PatientMaster item)
        {
            item = new PatientMaster();
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand(PatientMasterGet))
                {
                    AddStringParameter("@Ihi", ihi, command);
                    AddIntParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId, command);
                    result = PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), item);
                    if (item.PatientMasterId.HasValue)
                    {
                        AddressDl addressDl = new AddressDl();
                        item.Addresses = addressDl.GetAllByPatient(item.PatientMasterId.Value);
                        ContactDl contactDl = new ContactDl();
                        item.Contacts = contactDl.GetAllByPatient(item.PatientMasterId.Value);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        item.Names = nameDl.GetAllByPatient(item.PatientMasterId.Value);
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterGetByIhi, ihi);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_057);
            }

            return result;
        }

        /// <summary>
        /// Gets the specified patient based on the state/territory patient identifier.
        /// </summary>
        /// <param name="ihi">The state/territory patient identifier</param>
        /// <param name="item">The item.</param>
        /// <param name="transaction">An existing transaction in which to perform the query (optional).</param>
        /// <returns>Whether a patient was found</returns>
        public HipsResponse GetByStatePatientId(string statePatientId, int healthProviderOrganisationNetworkId, out PatientMaster item, SqlTransaction transaction = null)
        {
            item = new PatientMaster();
            HipsResponse result = new HipsResponse(HipsResponseIndicator.OK);
            try
            {
                using (SqlCommand command = GetSqlCommand(PatientMasterGet, transaction))
                {
                    AddStringParameter("@StatePatientId", statePatientId, command);
                    AddIntParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId, command);
                    PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), item);
                    if (item.PatientMasterId.HasValue)
                    {
                        AddressDl addressDl = new AddressDl();
                        item.Addresses = addressDl.GetAllByPatient(item.PatientMasterId.Value, transaction);
                        ContactDl contactDl = new ContactDl();
                        item.Contacts = contactDl.GetAllByPatient(item.PatientMasterId.Value, transaction);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        item.Names = nameDl.GetAllByPatient(item.PatientMasterId.Value, transaction);
                    }
                    else
                    {
                        result.Status = HipsResponseIndicator.InvalidPatient;
                    }
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterGetByStatePatientId, statePatientId);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_058);
                result.Status = HipsResponseIndicator.DatabaseError;
                result.HipsErrorMessage = message;
                result.ResponseCodeDescription = ex.Message;
            }
            return result;
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Insert(PatientMaster item, SqlTransaction transaction)
        {
            bool updated = false;
            bool useExistingTransaction = transaction != null;

            using (SqlCommand command = GetSqlCommand("hips.PatientMasterInsert"))
            {
                try
                {
                    if (!useExistingTransaction)
                    {
                        transaction = command.Connection.BeginTransaction();
                        command.Transaction = transaction;
                    }

                    updated = base.Insert<PatientMaster>(item, command);
                    if (updated)
                    {
                        updated = UpdateName(item, transaction);
                    }
                    if (updated)
                    {
                        updated = UpdateAddress(item, transaction);
                    }
                    if (updated)
                    {
                        updated = UpdateContact(item, transaction);
                    }
                    if (updated)
                    {
                        if (!useExistingTransaction)
                        {
                            transaction.Commit();
                            command.Connection.Close();
                        }
                    }
                    else
                    {
                        transaction.Rollback();
                        command.Connection.Close();
                    }
                }
                catch (Exception ex)
                {
                    if (transaction != null)
                    {
                        transaction.Rollback();
                    }
                    command.Connection.Close();
                    EventLogger.WriteLog(ConstantsResource.ErrorPatientMasterInsert, ex, User, LogMessage.HIPS_MESSAGE_060);
                    updated = false;
                }
            }
            return updated;
        }

        /// <summary>
        /// Merges the specified items.
        /// </summary>
        /// <param name="fromPatientMaster">The non-surviving patient master</param>
        /// <param name="toPatientMaster">The surviving patient master</param>
        /// <returns>Whether the merge was successful</returns>
        public bool Merge(PatientMaster fromPatientMaster, PatientMaster toPatientMaster, int healthProviderOrganisationNetworkId, SqlTransaction transaction)
        {
            bool result = false;
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PatientMasterMerge", transaction))
                {
                    command.Parameters.Add(new SqlParameter("@FromPatientMasterId", fromPatientMaster.PatientMasterId.Value));
                    command.Parameters.Add(new SqlParameter("@ToPatientMasterId", toPatientMaster.PatientMasterId.Value));
                    command.Parameters.Add(new SqlParameter("@UserModified", GetUserModified()));
                    command.Parameters.Add(new SqlParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId));
                    result = PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), toPatientMaster);
                    if (toPatientMaster.PatientMasterId.HasValue)
                    {
                        AddressDl addressDl = new AddressDl();
                        toPatientMaster.Addresses = addressDl.GetAllByPatient(toPatientMaster.PatientMasterId.Value);
                        ContactDl contactDl = new ContactDl();
                        toPatientMaster.Contacts = contactDl.GetAllByPatient(toPatientMaster.PatientMasterId.Value);
                        PatientMasterNameDl nameDl = new PatientMasterNameDl();
                        toPatientMaster.Names = nameDl.GetAllByPatient(toPatientMaster.PatientMasterId.Value);
                    }
                    if (result)
                    {
                        // Create a new command in the same transaction to repopulate the "from" patient master.
                        using (SqlCommand command2 = GetSqlCommand("hips.PatientMasterGet", transaction))
                        {
                            AddIntParameter("@PatientMasterId", fromPatientMaster.PatientMasterId, command2);
                            AddIntParameter("@HealthProviderOrganisationNetworkId", healthProviderOrganisationNetworkId, command2);
                            result = PopulateBusinessObject<PatientMaster>(command2.ExecuteReader(), fromPatientMaster);
                        }
                    }
                    if (!result)
                    {
                        transaction.Rollback();
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format(ConstantsResource.ErrorMessagePatientMasterMerge,
                    fromPatientMaster.PatientMasterId.Value,
                    toPatientMaster.PatientMasterId.Value);
                EventLogger.WriteLog(message, ex, User, LogMessage.HIPS_MESSAGE_062);
                transaction.Rollback();
                Command.Connection.Close();
                result = false;
            }
            return result;
        }

        /// <summary>
        /// Updates the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Update(PatientMaster item, SqlTransaction transaction)
        {
            bool updated = false;

            using (SqlCommand command = GetSqlCommand(PatientMasterUpdate, transaction))
            {
                try
                {
                    //transaction = command.Connection.BeginTransaction();
                    //command.Transaction = transaction;

                    updated = base.Update<PatientMaster>(item, command);
                    if (updated)
                    {
                        updated = UpdateName(item, transaction);
                    }

                    if (updated)
                    {
                        updated = UpdateAddress(item, transaction);
                    }
                    if (updated)
                    {
                        updated = UpdateContact(item, transaction);
                    }
                    if (updated)
                    {
                        //transaction.Commit();
                        //command.Connection.Close();
                    }
                    else
                    {
                        transaction.Rollback();
                        command.Connection.Close();
                    }
                }
                catch (Exception ex)
                {
                    if (transaction != null)
                    {
                        transaction.Rollback();
                    }
                    command.Connection.Close();
                    EventLogger.WriteLog(ConstantsResource.ErrorPatientMasterUpdate, ex, User, LogMessage.HIPS_MESSAGE_061);
                    updated = false;
                }
            }

            return updated;
        }

        #region Childen Methods

        /// <summary>
        /// Updates the address records.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        private bool UpdateAddress(PatientMaster item, SqlTransaction transaction)
        {
            AddressDl addressDl = new AddressDl();
            return addressDl.Update(item.Addresses, item.PatientMasterId.Value, transaction);
        }

        /// <summary>
        /// Updates the contact.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        private bool UpdateContact(PatientMaster item, SqlTransaction transaction = null)
        {
            ContactDl contactDl = new ContactDl();
            return contactDl.Update(item.Contacts, item.PatientMasterId.Value, transaction);
        }

        /// <summary>
        /// Updates the name.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        private bool UpdateName(PatientMaster item, SqlTransaction transaction)
        {
            PatientMasterNameDl nameDl = new PatientMasterNameDl();
            return nameDl.Update(item.Names, item.PatientMasterId.Value, transaction);
        }

        #endregion Childen Methods

        #endregion Methods
    }
}