﻿using System.Collections.Generic;
using System.IO;
using System.Linq;
using Ionic.Zip;

namespace Test.CommonCcaNoc.Helpers
{
    /// <summary>
    /// This class contains helper methods to extract files from CDA packages.
    /// </summary>
    public class CdaPackage
    {
        public const string DOCUMENT_FILE_NAME = "CDA_ROOT.XML";
        public const string SIGNATURE_FILE_NAME = "CDA_SIGN.XML";
        public const string LOGO_FILE_NAME = "LOGO.PNG";
        public const string PDF_EXTENSION_NAME = ".PDF";
        
        /// <summary>
        /// Extracts the file with the expected filename for the document, CDA_ROOT.XML, from the package.
        /// </summary>
        /// <param name="package"></param>
        /// <returns></returns>
        public static byte[] ExtractDocument(byte[] package)
        {
            MemoryStream outMemoryStream = new MemoryStream();
            ICollection<ZipEntry> entries = ZipFile.Read(new MemoryStream(package)).Entries;
            entries.First(a => a.FileName.EndsWith(DOCUMENT_FILE_NAME)).Extract(outMemoryStream);
            return outMemoryStream.ToArray();
        }

        /// <summary>
        /// Extracts the file with the expected filename for the signature, CDA_SIGN.XML, from the package.
        /// </summary>
        /// <param name="package"></param>
        /// <returns></returns>
        public static byte[] ExtractSignature(byte[] package)
        {
            MemoryStream outMemoryStream = new MemoryStream();
            ICollection<ZipEntry> entries = ZipFile.Read(new MemoryStream(package)).Entries;
            entries.First(a => a.FileName.EndsWith(SIGNATURE_FILE_NAME)).Extract(outMemoryStream);
            return outMemoryStream.ToArray();
        }

        /// <summary>
        /// Extracts the file with the expected filename for the logo, LOGO.PNG, from the package.
        /// </summary>
        /// <param name="package"></param>
        /// <returns></returns>
        public static byte[] ExtractLogo(byte[] package)
        {
            MemoryStream outMemoryStream = new MemoryStream();
            ICollection<ZipEntry> entries = ZipFile.Read(new MemoryStream(package)).Entries;
            entries.First(a => a.FileName.ToUpper().EndsWith(LOGO_FILE_NAME)).Extract(outMemoryStream);
            return outMemoryStream.ToArray();
        }

        /// <summary>
        /// Extracts the file with the expected extension of PDF, from the package.
        /// </summary>
        /// <param name="package"></param>
        /// <returns></returns>
        public static byte[] ExtractPDFAttachment(byte[] package)
        {
            MemoryStream outMemoryStream = new MemoryStream();
            ICollection<ZipEntry> entries = ZipFile.Read(new MemoryStream(package)).Entries;
            entries.First(a => a.FileName.ToUpper().EndsWith(PDF_EXTENSION_NAME)).Extract(outMemoryStream);
            return outMemoryStream.ToArray();
        }

        /// <summary>
        /// Gets a list of attachment entries in the CDA package ZIP file.
        /// Any file other than the CDA root or signature is considered an attachment.
        /// </summary>
        /// <param name="package"></param>
        /// <returns></returns>
        public static List<ZipEntry> GetAttachmentEntries(byte[] package)
        {
            ICollection<ZipEntry> entries = ZipFile.Read(new MemoryStream(package)).Entries;
            return entries.Where(a => !a.IsDirectory
                && !a.FileName.EndsWith(DOCUMENT_FILE_NAME)
                && !a.FileName.EndsWith(SIGNATURE_FILE_NAME)).ToList();
        }
    }
}