﻿using System;
using System.Security.Cryptography;
using System.Xml;
using HIPS.PcehrDataStore.Schemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.CdaPackaging.CdaXmlDocument
{
    /// <summary>
    /// Conformance Test Specification: CDA Packaging
    ///                                 Version 1.4 — 12 September 2012
    /// Profile Name:	CDA XML Document
    /// Group: Package Attachments
    /// </summary>
    [TestClass]
    public class PackageAttachments : CcaTest
    {
        private XmlNode edElement;
        private ClinicalDocumentVersion version;
        private CdaDocument uploadedDocument;

        /// <summary>
        /// Shared helper method to ensure that a document has been uploaded
        /// and get the package, uploaded document and logo reference.
        /// The document will be reused between the tests in this class.
        /// </summary>
        private void UploadDocument()
        {
            SharedPackage shared = SharedPackage.GetSharedInstance(this, SampleDocumentType.DischargeSummary3A);
            this.uploadedDocument = shared.CdaDocument;
            this.version = shared.Version;
            this.edElement = uploadedDocument.GetLogoReference();
        }

        /// <summary>
        /// Where the CDA XML document contains a packaged attachment, verify that the packaged attachment is represented using an ED-element. An example of an ED element is:
        /// <code><![CDATA[
        ///   <value mediaType="image/jpeg" integrityCheckAlgorithm="SHA-1" integrityCheck="0NyFNJ74XRAJjvsihGPqGePn0gU=">
        ///     <reference value="a19605b5-6c76-4608-9046-86c417f1e43c"/>
        ///   </value>
        /// ]]></code>
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_015()
        {
            UploadDocument();
            LogAssert.IsNotNull(edElement, DialogueResource.LogoEdElement);
        }

        /// <summary>
        /// Verify that an integrityCheckAlgorithm attribute of the ED-element has the value "SHA-1".
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_016()
        {
            UploadDocument();
            XmlAttribute integrityCheckAlgorithm = edElement.Attributes["integrityCheckAlgorithm"];
            LogAssert.IsNotNull(integrityCheckAlgorithm, DialogueResource.IntegrityCheckAlgorithmAttribute);
            LogAssert.AreEqual("SHA-1", integrityCheckAlgorithm.Value, DialogueResource.IntegrityCheckAlgorithmAttribute);
        }

        /// <summary>
        /// Verify that the ED-element has one and only one cda:reference element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_017()
        {
            UploadDocument();
            XmlNodeList references = edElement.SelectNodes("x:reference", uploadedDocument.NamespaceManager);
            LogAssert.AreEqual(1, references.Count, DialogueResource.NumberOfCdaReferenceElements);
        }

        /// <summary>
        /// Verify that the cda:reference element has a value which is not a zero-length string containing a URI of the form:
        ///     scheme ":" ["//" authority "/"] path [ "?" query ] [ "#" fragment ]
        /// (Refer to Uniform Resource Identifier (URI): Generic Syntax [RFC 3986] for more details.)
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_018()
        {
            UploadDocument();
            XmlNodeList references = edElement.SelectNodes("x:reference", uploadedDocument.NamespaceManager);
            string value = references[0].Attributes["value"].Value;
            LogAssert.IsFalse(string.IsNullOrEmpty(value), DialogueResource.ValueNotZeroLengthString, DialogueResource.ValueIsZeroLengthString);
            LogAssert.IsTrue(Uri.IsWellFormedUriString(value, UriKind.RelativeOrAbsolute), DialogueResource.ValueIsWellFormedUri, DialogueResource.ValueNotWellFormedUri);
        }

        /// <summary>
        /// Verify that the ED-Element does not contain any reference to an equivalent identifier in the package that the CDA XML document is a member of unless that ED-element represents a packaged attachment.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_019()
        {
            UploadDocument();
            byte[] logoFile = CdaPackage.ExtractLogo(version.Package);
            LogAssert.IsNotNull(logoFile, DialogueResource.LogoFileInCdaPackage);
        }

        /// <summary>
        /// Verify that the value of an integrityCheck attribute of the ED-Elementis equal to the SHA-1 digest of the byte stream, where the packaged attachment is an atomic packaged attachment.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_020()
        {
            UploadDocument();
            XmlAttribute integrityCheckAlgorithm = edElement.Attributes["integrityCheckAlgorithm"];
            byte[] logoFile = CdaPackage.ExtractLogo(version.Package);
            string calculatedDigest = Convert.ToBase64String(SHA1.Create().ComputeHash(logoFile));
            XmlAttribute integrityCheck = edElement.Attributes["integrityCheck"];
            LogAssert.IsNotNull(integrityCheckAlgorithm, DialogueResource.IntegrityCheckAttribute);
            LogAssert.AreEqual(calculatedDigest, integrityCheck.Value, DialogueResource.IntegrityCheckAttribute);
        }

        /// <summary>
        /// Verify that the value of the mediaType attribute in the ED-Element
        /// is an agreed Internet media type (or the value
        /// "application/octet-string" if there is no agreed value), where the
        /// packaged attachment is an atomic packaged attachment.
        /// (Note: In the scope of PCEHR the agreed values are defined in the
        /// Common Conformance Profile for Clinical Documents)
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_021()
        {
            UploadDocument();
            XmlAttribute mediaType = edElement.Attributes["mediaType"];
            LogAssert.IsNotNull(mediaType, DialogueResource.MediaTypeAttribute);
            LogAssert.AreEqual("image/png", mediaType.Value, DialogueResource.MediaTypeAttribute);
        }
    }
}