﻿using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Schema;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentProduction
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  uploadDocument
    /// </summary>
    [TestClass]
    public class NOC_UploadDocument : CcaTest
    {
        /// <summary>
        /// Test No:         33
        ///
        /// Objective/Input:
        /// Perform a Provide and Register Document set - b operation with valid
        /// input elements for a Shared Health Summary document.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_033()
        {
            TestSuccessfulUpload(SampleDocumentType.SharedHealthSummary);
        }

        /// <summary>
        /// Test No:         34
        ///
        /// Objective/Input:
        /// Perform a Provide and Register Document set - b operation with valid
        /// input elements for an Event Summary document.
        /// </summary> 
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_034()
        {
            TestSuccessfulUpload(SampleDocumentType.EventSummary);
        }

        /// <summary>
        /// Test No:         35
        ///
        /// Objective/Input:
        /// Perform a Provide and Register Document set - b operation with valid
        /// input elements for a Discharge Summary document.
        /// </summary> 
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_035()
        {
            TestSuccessfulUpload(SampleDocumentType.DischargeSummary3A);
        }

   /*  NOC test for eReferral is not in scope of HIPS 4.1
        /// <summary>
        /// Test No:         36
        ///
        /// Objective/Input:
        /// Perform a Provide and Register Document set - b operation with valid
        /// input elements for an eReferral document.
        /// </summary> 
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_036()
        {
            TestSuccessfulUpload(SampleDocumentType.eReferral);
        }
   */

        /// <summary>
        /// Test No:         37
        ///
        /// Objective/Input:
        /// Perform a Provide and Register Document set - b operation with valid
        /// input elements for a Specialist Letter document.
        /// </summary> 
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_037()
        {
            TestSuccessfulUpload(SampleDocumentType.SpecialistLetter);
        }

        /// <summary>
        /// Test No:         112
        ///
        /// Objective/Input:
        /// Perform a Provide and Register Document set - b operation with valid
        /// input elements for a PCEHR Prescription Record.
        /// </summary> 
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_112()
        {
            TestSuccessfulUpload(SampleDocumentType.PcehrPrescriptionRecord);
        }

        /// <summary>
        /// Test No:         113
        ///
        /// Objective/Input:
        /// Perform a Provide and Register Document set - b operation with valid
        /// input elements for a PCEHR Dispense Record.
        /// </summary> 
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_113()
        {
            TestSuccessfulUpload(SampleDocumentType.PcehrDispenseRecord);
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     140
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98 
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154 
        ///     DEXS-L155 DEXS-L156  DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94 
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125 
        ///     DocumentMetadata DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109 
        ///     DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117 
        ///     DEXS-L118  DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 
        ///     DEXS-T55 DEXS-T56 DEXS-T57 DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 
        ///     DEXS-T62 DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103
        ///     DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123 
        ///     DEXS-T126 DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132
        ///     DEXS-T147 DEXS-T150 DEXS-T143 DEXS-T144 
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.03 UPLOAD.04 
        ///     CLD.06 CLD.07 CLD.08 CLD.09 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a Diagnostic Imaging Report document. 
        /// Expected Result	
        ///     - The document is submitted successfully with the input data field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header 
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     - Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     - WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///        Field Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///        identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been 
        ///        converted to UTC if the time within the CDA document contains a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in the
        ///        same folder as the CDA_ROOT.xml for the purpose of sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '100.16957'
        ///     - The XDSDocumentEntry.classCodeDisplayName and XDSDocumentEntry.typeCodeDisplayName are set to 'Diagnostic Imaging Report'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the latest imaging date recorded
        ///     - The XDSDocumentEntry.serviceStopTime is set to the latest imaging date recorded
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)- 
        ///     - The Title slot is present and set to 'Diagnostic Imaging Report'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)- 
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///        as-is if the time within the CDA document contains a date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not padded
        ///        with zeroes to increase precision if the time within the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_140()
        {
            this.TestSuccessfulUpload(SampleDocumentType.DiagnosticImagingReport);
        }

        /// <summary>
        /// Operations
        ///     uploadDocument
        /// Test No.
        ///     141
        /// Conformance Point
        ///     DEXS-L1 DEXS-L2 DEXS-L3 DEXS-L4 DEXS-L5 DEXS-L6 DEXS-L98 
        ///     DEXS-L102 DEXS-L103 DEXS-L104 DEXS-L105 DEXS-L153 DEXS-L154 
        ///     DEXS-L155 DEXS-L156  DEXS-T5 DEXS-T6 DEXS-T7 DEXS-T8 DEXS-T9
        ///     DEXS-T10 DEXS-T11 DEXS-T12 DEXS-T13 DEXS-T73 DEXS-T93 DEXS-T94 
        ///     DEXS-T95 DEXS-T99 DEXS-T100 DEXS-T101 DEXS-T117 DEXS-T125 
        ///     DocumentMetadata DEXS-L106 DEXS-L107 DEXS-L108 DEXS-L109 
        ///     DEXS-L112 DEXS-L113 DEXS-L114 DEXS-L115 DEXS-L116 DEXS-L117 
        ///     DEXS-L118  DEXS-T50 DEXS-T51 DEXS-T52 DEXS-T53 DEXS-T54 
        ///     DEXS-T55 DEXS-T56 DEXS-T57 DEXS-T58 DEXS-T59 DEXS-T60 DEXS-T61 
        ///     DEXS-T62 DEXS-T63 DEXS-T64 DEXS-T65 DEXS-T67 DEXS-T69 DEXS-T103
        ///     DEXS-T104 DEXS-T105 DEXS-T120 DEXS-T121 DEXS-T122 DEXS-T123 
        ///     DEXS-T126 DEXS-T128 DEXS-T129 DEXS-T130 DEXS-T131 DEXS-T132
        ///     DEXS-T147 DEXS-T150 DEXS-T143 DEXS-T144 
        ///     UPLOAD.01 UPLOAD.02 UPLOAD.03 UPLOAD.04 
        ///     CLD.06 CLD.07 CLD.08 CLD.09 CLD.14 CLD.15 CLD.16
        /// Test Type
        ///     Positive
        /// Role
        ///     Clinical Information System (CIS) or Contracted Service Provider (CSP)
        /// Objective/Input
        ///     Perform a Provide and Register Document set - b operation with
        ///     valid input elements for a Pathology Report document. 
        /// Expected Result	
        ///     - The document is submitted successfully with the input data field ihe:ProviderAndRegisterDocumentSetRequest. 
        ///     - The operation request includes the full Document Exchange Common Header 
        ///     - The Provide & Register Document Set transactions contains exactly only one XDS Document element
        ///     - The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        ///     - The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        ///     - Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        ///     - The request conforms with the B2B_DocumentRepository.wsdl
        ///     - WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        ///     - Only XDSDocumentEntry fields which are required in the XDS.b
        ///        Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        ///     - The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        ///     - The XDSDocumentEntry.confidentialityCode contains 'N/A'
        ///     - The XDSDocumentEntry.uniqueId shall contain an OID
        ///     - The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document 
        ///     - The XDSDocumentEntry.sourcePatientId is identical to the XDSDocumentEntry.patientId field within the supplied CDA Document
        ///     - The XDSDocumentEntry.formatCode field contains the Template ID of the Template to which the CDA Document asserts conformance
        ///     - The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        ///     - Only XDSSubmissionSet fields which are shown in the XDS.b
        ///        Field Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        ///     - The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID' 
        ///     - The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson entity
        ///     - The XDSSubmissionSet.authorInstitution entity is identical to the value provided within the XDSDocumentEntry.authorInstitution
        ///     - The XDSSubmission.contentTypeCode entity is identical to the value in the XDSDocumentEntry.classCode entity
        ///     - The XDSSubmissionSet.contentTypeCodeDisplayName entity is
        ///        identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        ///     - The XDSSubmissionSet.patientID entity is identical to the value provided within the XDSDocumentEntry.sourcePatientId entity
        ///     - The XDS.b Association is used to managed document versioning
        ///     - The XDS.b Association Type is set to 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'
        ///     - The creationTime/serviceStartTime/serviceStopTime has been 
        ///        converted to UTC if the time within the CDA document contains a date-time value with a timezone
        ///     - Attachments included in the CDA Package SHALL be located in the
        ///        same folder as the CDA_ROOT.xml for the purpose of sending to the PCEHR system
        ///     - The XDSDocumentEntry. authorInstitution name field matches the Organisation Name used in the HI Service
        ///     - The XDSDocumentEntry.authorPerson field is set to the author name and author identifier from the CDA document
        ///     - The XDSDocumentEntry.classCode and XDSDocumentEntry.typeCode fields are set to  '100.32001'
        ///     - The XDSDocumentEntry.classCodeDisplayName and XDSDocumentEntry.typeCodeDisplayName are set to 'Pathology Report'
        ///     - The XDSDocumentEntry.creationTime is set to the CDA effectiveTime
        ///     - The XDSDocumentEntry.serviceStartTime is set to the latest specimen collection date recorded
        ///     - The XDSDocumentEntry.serviceStopTime is set to the latest specimen collection date recorded
        ///     
        ///     If the CIS or CSP has implemented CUP R1 (Clinical Usability Software Requirements V1)- 
        ///     - The Title slot is present and set to 'Pathology Report'
        ///     
        ///     If the CIS or CSP has implemented CUP R2 (Clinical Usability Software Recommendations V1.1)- 
        ///     - The Title slot is not present
        ///     - The creationTime/serviceStartTime/serviceStopTime is recorded
        ///        as-is if the time within the CDA document contains a date-only value
        ///     - The creationTime/serviceStartTime/serviceStopTime is not padded
        ///        with zeroes to increase precision if the time within the CDA document contains a date-only value
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_141()
        {
            this.TestSuccessfulUpload(SampleDocumentType.PathologyReport);
        }

        /// <summary>
        /// Tests the successful upload of a particular document type.
        /// 
        /// Expected Result:
        ///
        /// A. The document is submitted successfully with the input data field ihe:ProviderAndRegisterDocumentSetRequest.
        /// B. The operation request includes the full Document Exchange Common Header
        /// C. The Provide & Register Document Set transactions contains exactly only one XDS Document element
        /// D. The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        /// E. The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        /// F. Document Addendum, Document Transformation and Folder Management Options are NOT allowed
        /// G. The request conforms with the B2B_DocumentRepository.wsdl
        /// H. WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        /// Metadata
        /// I. Only XDSDocumentEntry fields which are required in the XDS.b Field Name of Table 2 in the PCEHR Document Exchange Service TSS are provided
        /// J. The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header
        /// K. The XDSDocumentEntry.confidentialityCode contains 'N/A'
        /// L. The XDSDocumentEntry.uniqueId contains an OID
        /// M. The XDSDocumentEntry.typeCode is the same value as the XDSDocumentEntry.classCode field
        /// N. The XDSDocumentEntry.typeCodeDisplayName is the same value as the XDSDocumentEntry.classCodeDisplayName field
        /// O. The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document
        /// P. The XDSDocumentEntry.sourcePatientId is identical to the XDSDocumentEntry.patientId field within the supplied CDA Document
        /// Q. The XDSDocumentEntry.formatCode field contains the Template ID of the Template to which the CDA Document asserts conformance
        /// R. The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        /// S. Only XDSSubmissionSet fields which are shown in the XDS.b Field Name of Table 3 in the  PCEHR Document Exchange Service TSS are provided
        /// T. The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID'
        /// U. The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson
        /// V. The XDSSubmissionSet.authorInstitution entity is identical to the value provided within the XDSDocumentEntry.authorInstitution
        /// W. The XDSSubmission.contentTypeCode entity is identical to the value in the XDSDocumentEntry.classCode entity
        /// X. The XDSSubmissionSet.contentTypeCodeDisplayName entity is identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        /// Y. The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson entity
        /// Z. The XDSSubmissionSet.personID entity is identical to the value provided within the XDSDocumentEntry.sourcePersonId entity
        /// AA. The XDS.b Association is used to managed document versioning
        /// AB. The XDS.b Association Type is set to 'urn:ihe:ihi:2007:AssociationType:HasMember'
        /// </summary>
        /// <param name="docType">The document type.</param>
        private void TestSuccessfulUpload(SampleDocumentType docType)
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(docType);           
            string documentFormatCode = document.GetFormatCode();
            HipsResponse hipsResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(document.GetBytes(), identifier, user, document.GetAttachments(), patient.TargetEpisode.AdmissionDate, documentFormatCode);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, hipsResponse, DialogueResource.HipsServiceUploadDocument);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.UploadDocument, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Confirm that the payload is well formed and complies with the Request XML Schema
            XmlDocument soapRequest = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                soapRequest.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
            nsmgr.AddNamespace("a", "http://www.w3.org/2005/08/addressing");
            nsmgr.AddNamespace("xds", "urn:ihe:iti:xds-b:2007");
            nsmgr.AddNamespace("lcm", "urn:oasis:names:tc:ebxml-regrep:xsd:lcm:3.0");
            nsmgr.AddNamespace("rim", "urn:oasis:names:tc:ebxml-regrep:xsd:rim:3.0");
            XmlNode payload = soapRequest.SelectSingleNode("/s:Envelope/s:Body/xds:ProvideAndRegisterDocumentSetRequest", nsmgr);
            soapRequest.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            soapRequest.Schemas.Add(nsmgr.LookupNamespace("c"), "PCEHR_CommonTypes.xsd");
            soapRequest.Schemas.Add(nsmgr.LookupNamespace("xds"), "XDS.b_DocumentRepository.xsd");
            soapRequest.Schemas.Add(nsmgr.LookupNamespace("lcm"), "lcm.xsd");
            soapRequest.Schemas.Add(nsmgr.LookupNamespace("rim"), "rim.xsd");
            try
            {
                soapRequest.Validate(null);
                LogAssert.IsTrue(soapRequest.SchemaInfo.Validity == XmlSchemaValidity.Valid,
                    "SOAP request validates against SOAP envelope schema",
                    "SOAP request is invalid");

                soapRequest.Validate(null, payload);
                LogAssert.IsTrue(payload.SchemaInfo.Validity == XmlSchemaValidity.Valid,
                    "ProvideAndRegisterDocumentSetRequest validates against XDS.b Document Repository schema.",
                    "ProvideAndRegisterDocumentSetRequest is invalid");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }
            catch (XmlSchemaValidationException ex)
            {
                LogAssert.Fail(ex.Message);
            }
            PcehrHeaderTests headerTests = new PcehrHeaderTests(this);
            headerTests.CheckHeader(soapRequest, DialogueResource.HipsServiceUploadDocument);
            EvaluationC(payload, nsmgr);
            EvaluationD(payload, nsmgr);
            EvaluationE(payload, nsmgr);
            EvaluationF(payload, nsmgr);
            EvaluationH(soapRequest, nsmgr);
            EvaluationI(payload, nsmgr);
            EvaluationJ(payload, nsmgr);
            EvaluationK(payload, nsmgr);
            EvaluationL(payload, nsmgr);
            EvaluationM(payload, nsmgr);
            EvaluationN(payload, nsmgr);
            EvaluationO(payload, nsmgr, document);
            EvaluationP(payload, nsmgr);
            EvaluationQ(payload, nsmgr, document, documentFormatCode);
            EvaluationR(payload, nsmgr);
            EvaluationS(payload, nsmgr);
            EvaluationT(payload, nsmgr);
            EvaluationU(payload, nsmgr);
            EvaluationV(payload, nsmgr);
            EvaluationW(payload, nsmgr);
            EvaluationX(payload, nsmgr);

            // Y is skipped because it is the same as U
            EvaluationZ(payload, nsmgr);

            // AA is skipped because it cannot be demonstrated in this message for an original upload.
            EvaluationAB(payload, nsmgr);

            // Requirements on serviceStartTime and serviceStopTime added in
            // PCEHR DEXS TSS 1.5.1 dated 31 December 2014.
            EvaluationStartStopTime(payload, nsmgr, document, documentFormatCode);
        }

        /// <summary>
        /// Requirements on serviceStartTime and serviceStopTime added in
        /// PCEHR DEXS TSS 1.5.1 dated 31 December 2014.
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        /// <param name="document">The CDA document.</param>
        /// <param name="documentFormatCode">The document format code.</param>
        private void EvaluationStartStopTime(XmlNode payload, XmlNamespaceManager nsmgr, CdaDocument document, string documentFormatCode)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Slot[@name='serviceStartTime']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.serviceStartTime using XPath {0}", xPath); 
            string serviceStartTime = payload.SelectSingleNode(xPath, nsmgr).InnerText;
            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Slot[@name='serviceStopTime']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.serviceStopTime using XPath {0}", xPath);
            string serviceStopTime = payload.SelectSingleNode(xPath, nsmgr).InnerText;

            switch (document.DocumentType)
            {
                // Specialist Letter - effective time
                case SampleDocumentType.SpecialistLetter:
                    Log("Info: Expected value for specialist letter is effective time.");
                    LogAssert.AreEqual(document.GetEffectiveTime().ToUtc(), serviceStartTime, "XDSDocumentEntry.serviceStartTime");
                    LogAssert.AreEqual(document.GetEffectiveTime().ToUtc(), serviceStopTime, "XDSDocumentEntry.serviceStopTime");
                    break;

                // Discharge Summary - admit and discharge time
                case SampleDocumentType.DischargeSummary1A:
                case SampleDocumentType.DischargeSummary1B:
                case SampleDocumentType.DischargeSummary2:
                case SampleDocumentType.DischargeSummary3A:
                case SampleDocumentType.DischargeSummary3B:
                    Log("Info: Expected values for discharge summary are admit and discharge time.");
                    LogAssert.AreEqual(document.GetAdmitTime().ToUtc(), serviceStartTime, "XDSDocumentEntry.serviceStartTime");
                    LogAssert.AreEqual(document.GetDischargeTime().ToUtc(), serviceStopTime, "XDSDocumentEntry.serviceStopTime");
                    break;

                // eHealth Prescription Record - author time
                case SampleDocumentType.PcehrPrescriptionRecord:
                    Log("Info: Expected value for eHealth Prescription Record is author time.");
                    LogAssert.AreEqual(document.GetAuthorTime().ToUtc(), serviceStartTime, "XDSDocumentEntry.serviceStartTime");
                    LogAssert.AreEqual(document.GetAuthorTime().ToUtc(), serviceStopTime, "XDSDocumentEntry.serviceStopTime");
                    break;

                // eHealth Dispense Record - supply time
                case SampleDocumentType.PcehrDispenseRecord:
                    Log("Info: Expected value for eHealth Dispense Record is supply time.");
                    LogAssert.AreEqual(document.GetSupplyTime().ToUtc(), serviceStartTime, "XDSDocumentEntry.serviceStartTime");
                    LogAssert.AreEqual(document.GetSupplyTime().ToUtc(), serviceStopTime, "XDSDocumentEntry.serviceStopTime");
                    break;

                // Pathology Report - specimen collection time
                case SampleDocumentType.PathologyReport:
                    Log("Info: Expected value for Pathology Report is specimen collection time.");
                    LogAssert.AreEqual(document.GetSpecimenCollectionTime().ToUtc(), serviceStartTime, "XDSDocumentEntry.serviceStartTime");
                    LogAssert.AreEqual(document.GetSpecimenCollectionTime().ToUtc(), serviceStopTime, "XDSDocumentEntry.serviceStopTime");
                    break;

                // Diagnostic Imaging Report - imaging time
                case SampleDocumentType.DiagnosticImagingReport:
                    Log("Info: Expected value for Diagnostic Imaging Report is imaging time.");
                    LogAssert.AreEqual(document.GetImagingTime().ToUtc(), serviceStartTime, "XDSDocumentEntry.serviceStartTime");
                    LogAssert.AreEqual(document.GetImagingTime().ToUtc(), serviceStopTime, "XDSDocumentEntry.serviceStopTime");
                    break;

                // Unless otherwise stated (DEXS-T 133 and 138)
                default:
                    Log("Info: Expected value is encompassing encounter effective time (low to high if available) otherwise document effective time.");
                    string expectedStartTime = (document.GetAdmitTime() 
                        ?? document.GetEncompassingEncounterEffectiveTime() 
                        ?? document.GetEffectiveTime()).ToUtc();
                    string expectedStopTime = (document.GetDischargeTime() 
                        ?? document.GetEncompassingEncounterEffectiveTime() 
                        ?? document.GetEffectiveTime()).ToUtc();
                    LogAssert.AreEqual(expectedStartTime, serviceStartTime, "XDSDocumentEntry.serviceStartTime");
                    LogAssert.AreEqual(expectedStopTime, serviceStopTime, "XDSDocumentEntry.serviceStopTime");
                    break;
            }
        }

        /// <summary>
        /// C. The Provide & Register Document Set transactions contains exactly only one XDS Document element
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationC(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "xds:Document";
            Log("Info: Searching payload for {0}", xPath);
            XmlNodeList documentNodes = payload.SelectNodes(xPath, nsmgr);
            LogAssert.AreEqual(1, documentNodes.Count, "Number of XDS Document elements");
        }

        /// <summary>
        /// D. The Provide & Register Document Set transactions contains exactly one XDSDocumentEntry element
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationD(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject[@objectType='urn:uuid:7edca82f-054d-47f2-a032-9b2a5b5186c1']";
            Log("Info: Searching payload for XDSDocumentEntry elements using XPath {0}", xPath);
            XmlNodeList xdsDocumentEntryNodes = payload.SelectNodes(xPath, nsmgr);
            LogAssert.AreEqual(1, xdsDocumentEntryNodes.Count, "Number of XDSDocumentEntry elements");
        }

        /// <summary>
        /// E. The Provide & Register Document Set transactions DOES NOT contain XDS Folder elements
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationE(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject[@objectType='urn:uuid:d9d542f3-6cc4-48b6-8870-ea235fbc94c2']";
            Log("Info: Searching payload for XDSFolder elements using XPath {0}", xPath);
            XmlNodeList xdsFolderNodes = payload.SelectNodes(xPath, nsmgr);
            LogAssert.AreEqual(0, xdsFolderNodes.Count, "Number of XDS Folder elements");
        }

        /// <summary>
        /// F. Document Addendum, Document Transformation and Folder Management Options are NOT allowed.
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationF(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:Association[@associationType='urn:ihe:iti:2007:AssociationType:APND']";
            Log("Info: Searching payload for Document Addendum associations using XPath {0}", xPath);
            XmlNodeList apndAssociationNodes = payload.SelectNodes(xPath, nsmgr);
            LogAssert.AreEqual(0, apndAssociationNodes.Count, "Number of Document Addendum elements");

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:Association[@associationType='urn:ihe:iti:2007:AssociationType:XFRM']";
            Log("Info: Searching payload for Document Transformation associations using XPath {0}", xPath);
            XmlNodeList xfrmAssociationNodes = payload.SelectNodes(xPath, nsmgr);
            LogAssert.AreEqual(0, xfrmAssociationNodes.Count, "Number of Document Transformation elements");
        }

        /// <summary>
        /// Checks H. WS-Action must be urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b
        /// </summary>
        /// <param name="soapRequest"></param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationH(XmlDocument soapRequest, XmlNamespaceManager nsmgr)
        {
            string xPath = "/s:Envelope/s:Header/a:Action";
            Log("Info: Searching request for WS-Action using XPath {0}", xPath);
            XmlNode actionNode = soapRequest.SelectSingleNode(xPath, nsmgr);
            LogAssert.AreEqual("urn:ihe:iti:2007:ProvideAndRegisterDocumentSet-b", actionNode.InnerText, "WS-Action");
        }

        /// <summary>
        /// Checks I: Only XDSDocumentEntry fields which are required in the
        /// XDS.b Field Name of Table 2 in the PCEHR Document Exchange Service
        /// TSS are provided.
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationI(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string[] shownSlotNames =
            {
                "creationTime",
                "serviceStartTime",
                "serviceStopTime",
                "languageCode",
                "sourcePatientId",
            };
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Slot";
            Log("Info: Searching payload XDSDocumentEntry slots using XPath {0}", xPath);
            XmlNodeList xdsDocumentEntrySlots = payload.SelectNodes(xPath, nsmgr);
            foreach (XmlNode node in xdsDocumentEntrySlots)
            {
                string name = node.Attributes["name"].Value;
                bool validName = shownSlotNames.Contains(name);
                string passMessage = string.Format("XDSDocumentEntry slot {0} is shown in the table.", name);
                string failMessage = string.Format("XDSDocumentEntry slot {0} is not shown in the table", name);
                LogAssert.IsTrue(validName, passMessage, failMessage);
            }

            Dictionary<string, string> requiredXDSDocumentEntryClassificationSchemes = new Dictionary<string, string>()
            {
                {"urn:uuid:93606bcf-9494-43ec-9b4e-a7748d1a838d", "XDSDocumentEntry.author"},
                {"urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a", "XDSDocumentEntry.classCode"},
                {"urn:uuid:a09d5840-386c-46f2-b5ad-9c3699a4309d", "XDSDocumentEntry.formatCode"},
                {"urn:uuid:2e82c1f6-a085-4c72-9da3-8640a32e42ab", "XDSDocumentEntry.uniqueId"},
                {"urn:uuid:f33fb8ac-18af-42cc-ae0e-ed0b0bdb91e1", "XDSDocumentEntry.healthCareFacilityTypeCode"},
                {"urn:uuid:cccf5598-8b07-4b77-a05e-ae952c785ead", "XDSDocumentEntry.practiceSettingCode"},
                {"urn:uuid:f4f85eac-e6cb-4883-b524-f2705394840f", "XDSDocumentEntry.confidentialityCode"},
                {"urn:uuid:f0306f51-975f-434e-a61c-c59651d33983", "XDSDocumentEntry.typeCode"},
                {"urn:uuid:58a6f841-87b3-4a3e-92fd-a8ffeff98427", "XDSDocumentEntry.patientId"},
            };
            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification";
            Log("Info: Searching payload for XDSDocumentEntry classifications using XPath {0}", xPath);
            XmlNodeList xdsDocumentEntryClassifications = payload.SelectNodes(xPath, nsmgr);
            foreach (XmlNode node in xdsDocumentEntryClassifications)
            {
                string classificationScheme = node.Attributes["classificationScheme"].Value;
                bool validClassificationScheme = requiredXDSDocumentEntryClassificationSchemes.ContainsKey(classificationScheme);
                string passMessage = string.Format("XDSDocumentEntry classification {1} with UUID {0} is shown in the table.",
                    classificationScheme, requiredXDSDocumentEntryClassificationSchemes[classificationScheme]);
                string failMessage = string.Format("XDSDocumentEntry classification with UUID {0} is not shown in the table.",
                    classificationScheme);
                LogAssert.IsTrue(validClassificationScheme, passMessage, failMessage);
            }
        }

        /// <summary>
        /// Check J: The XDSDocumentEntry.sourcePatientId contains 16-digit IHI number in the PCEHR Header.
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationJ(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber";
            Log("Info: Searching SOAP request for ihiNumber using XPath {0}", xPath);
            XmlNode ihiNumberNode = payload.SelectSingleNode(xPath, nsmgr);
            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Slot[@name='sourcePatientId']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.sourcePatientId using XPath {0}", xPath);
            XmlNode sourcePatientIdNode = payload.SelectSingleNode(xPath, nsmgr);
            bool sourcePatientIdContainsIhiNumber = sourcePatientIdNode.InnerText.Contains(ihiNumberNode.InnerText);
            string passMessage = string.Format("Source Patient ID {0} contains IHI number {1}", sourcePatientIdNode.InnerText, ihiNumberNode.InnerText);
            string failMessage = string.Format("Source Patient ID {0} does not contain IHI number {1}", sourcePatientIdNode.InnerText, ihiNumberNode.InnerText);
            LogAssert.IsTrue(sourcePatientIdContainsIhiNumber, passMessage, failMessage);
        }

        /// <summary>
        /// Check K. The XDSDocumentEntry.confidentialityCode contains 'N/A'
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationK(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:f4f85eac-e6cb-4883-b524-f2705394840f']/rim:Name/rim:LocalizedString";
            Log("Info: Searching payload for XDSDocumentEntry.confidentialityCode using XPath {0}", xPath);
            XmlNode confidentialityCodeNode = payload.SelectSingleNode(xPath, nsmgr);
            LogAssert.AreEqual("N/A", confidentialityCodeNode.Attributes["value"].Value, "Confidentiality Code Classification Name LocalizedString value");
        }

        /// <summary>
        /// Check L. The XDSDocumentEntry.uniqueId contains an OID.
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationL(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:ExternalIdentifier[@identificationScheme='urn:uuid:2e82c1f6-a085-4c72-9da3-8640a32e42ab']";
            Log("Info: Searching payload for XDSDocumentEntry.uniqueId using XPath {0}", xPath);
            XmlNode uniqueIdNode = payload.SelectSingleNode(xPath, nsmgr);
            string uniqueId = uniqueIdNode.Attributes["value"].Value;
            bool isOid = Regex.IsMatch(uniqueId, @"^\d+(\.\d+)*(\^.*)?$");
            LogAssert.IsTrue(isOid,
                string.Format("Unique ID {0} is an OID", uniqueId),
                string.Format("Unique ID {0} is not an OID", uniqueId));
        }

        /// <summary>
        /// Check M. The XDSDocumentEntry.typeCode is the same value as the XDSDocumentEntry.classCode field
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationM(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a']";
            Log("Info: Searching payload for XDSDocumentEntry.classCode using XPath {0}", xPath);
            XmlNode classCodeNode = payload.SelectSingleNode(xPath, nsmgr);
            string classCode = classCodeNode.Attributes["nodeRepresentation"].Value;
            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:f0306f51-975f-434e-a61c-c59651d33983']";
            Log("Info: Searching payload for XDSDocumentEntry.typeCode using XPath {0}", xPath);
            XmlNode typeCodeNode = payload.SelectSingleNode(xPath, nsmgr);
            string typeCode = typeCodeNode.Attributes["nodeRepresentation"].Value;

            LogAssert.AreEqual(classCode, typeCode, "XDSDocumentEntry.classCode", "XDSDocumentEntry.typeCode");
        }

        /// <summary>
        /// PCEHR Document Exchange Service IHE XDS-b Technical Service Specification v1.4 12/04/2013
        /// Table 3 - XDSDocumentEntry Document Type and Class Code value set.
        /// </summary>
        private class Table3
        {
            private static List<Table3> allEntries;

            public string CodingSystem { get; set; }

            public string Code { get; set; }

            public string ClassCodeDisplayName { get; set; }

            public string TypeCodeDisplayName { get; set; }

            private Table3(string codingSystem, string code, string classCodeDisplayName, string typeCodeDisplayName)
            {
                CodingSystem = codingSystem;
                Code = code;
                ClassCodeDisplayName = classCodeDisplayName;
                TypeCodeDisplayName = typeCodeDisplayName;
            }

            /// <summary>
            /// Gets the entries in the table indexed by their code.
            /// </summary>
            public static Dictionary<string, Table3> ByCode
            {
                get
                {
                    return AllEntries.ToDictionary(a => a.Code);
                }
            }

            /// <summary>
            /// Gets all the entries in the table.
            /// Source is PCEHR Document Exchange Service Using the IHE XDS.b Platform,
            /// Technical Service Specification, Table 3, XDSDocumentEntry Document Type
            /// and Class Code value set.
            /// </summary>
            public static List<Table3> AllEntries
            {
                get
                {
                    if (allEntries == null)
                    {
                        allEntries = new List<Table3>();

                        // PCEHR DEXS TSS v1.5.1 31/12/2014 changed ClassCodeDisplayName to match TypeCodeDisplayName.
                        allEntries.Add(new Table3("LOINC", "60591-5", "Shared Health Summary", "Shared Health Summary"));
                        allEntries.Add(new Table3("LOINC", "57133-1", "e-Referral", "e-Referral"));
                        allEntries.Add(new Table3("LOINC", "51852-2", "Specialist Letter", "Specialist Letter"));
                        allEntries.Add(new Table3("LOINC", "18842-5", "Discharge Summary", "Discharge Summary"));
                        allEntries.Add(new Table3("LOINC", "34133-9", "Event Summary", "Event Summary"));

                        // PCEHR DEXS TSS v1.5.1 31/12/2014 removed e-Prescription, Dispense Record and Prescription Request.
                        ////allEntries.Add(new Table3("NCTIS", "100.16100", "e-Prescription", "e-Prescription"));
                        ////allEntries.Add(new Table3("NCTIS", "100.16112", "Dispense Record", "Dispense Record"));
                        ////allEntries.Add(new Table3("NCTIS", "100.16285", "Prescription Request", "Prescription Request"));

                        allEntries.Add(new Table3("NCTIS", "100.16650", "Pharmaceutical Benefits Report", "Pharmaceutical Benefits Report"));
                        allEntries.Add(new Table3("NCTIS", "100.16659", "Australian Childhood Immunisation Register", "Australian Childhood Immunisation Register"));
                        allEntries.Add(new Table3("NCTIS", "100.16644", "Medicare/DVA Benefits Report", "Medicare/DVA Benefits Report"));
                        allEntries.Add(new Table3("NCTIS", "102.16671", "Australian Organ Donor Register", "Australian Organ Donor Register"));

                        // PCEHR DEXS TSS v1.5.1 31/12/2014 changed "Consumer Entered Notes" to "Personal Health Note".
                        allEntries.Add(new Table3("NCTIS", "100.16681", "Personal Health Note", "Personal Health Note"));

                        // PCEHR DEXS TSS v1.5.1 31/12/2014 changed "Consumer Entered Health Summary" to "Personal Health Summary".
                        allEntries.Add(new Table3("NCTIS", "100.16685", "Personal Health Summary", "Personal Health Summary"));

                        allEntries.Add(new Table3("NCTIS", "100.16696", "Advanced Care Directive Custodian Record", "Advanced Care Directive Custodian Record"));

                        // PCEHR DEXS TSS v1.5.1 31/12/2014 changed "PCEHR Prescription Record" to "eHealth Prescription Record".
                        allEntries.Add(new Table3("NCTIS", "100.16764", "eHealth Prescription Record", "eHealth Prescription Record"));

                        // PCEHR DEXS TSS v1.5.1 31/12/2014 changed "PCEHR Dispense Record" to "eHealth Dispense Record".
                        allEntries.Add(new Table3("NCTIS", "100.16765", "eHealth Dispense Record", "eHealth Dispense Record"));

                        // PCEHR DEXS TSS v1.5.1 31/12/2014 added "Diagnostic Imaging Report",
                        // "Pathology Report", Consumer Entered Measurements" and "Child Parent Questionnaire".
                        allEntries.Add(new Table3("NCTIS", "100.16957", "Diagnostic Imaging Report", "Diagnostic Imaging Report"));
                        allEntries.Add(new Table3("NCTIS", "100.32001", "Pathology Report", "Pathology Report"));
                        allEntries.Add(new Table3("NCTIS", "100.16870", "Consumer Entered Measurements", "Consumer Entered Measurements"));
                        allEntries.Add(new Table3("NCTIS", "100.16919", "Child Parent Questionnaire", "Child Parent Questionnaire"));
                    }
                    return allEntries;
                }
            }
        }

        /// <summary>
        /// Check N. The XDSDocumentEntry.typeCodeDisplayName is set to the appropriate typeCodeDisplayName
        /// and the XDSDocumentEntry.classCodeDisplayName is set to the appropriate classCodeDisplayName
        ///
        /// Note: Accenture advised the above text on 10/07/2013 as a replacement for the incorrect text in NOC Test Plan 4.0:
        /// "N. The XDSDocumentEntry.typeCodeDisplayName is the same value as the XDSDocumentEntry.classCodeDisplayName field"
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationN(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a']";
            Log("Info: Searching payload for XDSDocumentEntry.classCode using XPath {0}", xPath);
            XmlNode classCodeNode = payload.SelectSingleNode(xPath, nsmgr);
            string classCode = classCodeNode.Attributes["nodeRepresentation"].Value;
            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:f0306f51-975f-434e-a61c-c59651d33983']";
            Log("Info: Searching payload for XDSDocumentEntry.typeCode using XPath {0}", xPath);
            XmlNode typeCodeNode = payload.SelectSingleNode(xPath, nsmgr);
            string typeCode = typeCodeNode.Attributes["nodeRepresentation"].Value;
            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a']/rim:Name/rim:LocalizedString";
            Log("Info: Searching payload for XDSDocumentEntry.classCodeDisplayName using XPath {0}", xPath);
            XmlNode classCodeDisplayNameNode = payload.SelectSingleNode(xPath, nsmgr);
            string classCodeDisplayName = classCodeDisplayNameNode.Attributes["value"].Value;
            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:f0306f51-975f-434e-a61c-c59651d33983']/rim:Name/rim:LocalizedString";
            Log("Info: Searching payload for XDSDocumentEntry.typeCodeDisplayName using XPath for {0}", xPath);
            XmlNode typeCodeDisplayNameNode = payload.SelectSingleNode(xPath, nsmgr);
            string typeCodeDisplayName = typeCodeDisplayNameNode.Attributes["value"].Value;
            LogAssert.IsTrue(Table3.ByCode.ContainsKey(classCode), 
                string.Format("TSS Table 3 contains code {0}", classCode),
                string.Format("TSS Table 3 does not contain code {0}", classCode));
            LogAssert.AreEqual(Table3.ByCode[classCode].ClassCodeDisplayName, classCodeDisplayName, "XDSDocumentEntry.classCodeDisplayName");
            LogAssert.AreEqual(Table3.ByCode[typeCode].TypeCodeDisplayName, typeCodeDisplayName, "XDSDocumentEntry.typeCodeDisplayName");
        }

        /// <summary>
        /// Check O. The XDSDocumentEntry.uniqueId is identical to the DocumentID field within the CDA document
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationO(XmlNode payload, XmlNamespaceManager nsmgr, CdaDocument document)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:ExternalIdentifier[@identificationScheme='urn:uuid:2e82c1f6-a085-4c72-9da3-8640a32e42ab']";
            Log("Info: Searching payload for XDSDocumentEntry.uniqueId using XPath {0}", xPath);
            XmlNode uniqueIdNode = payload.SelectSingleNode(xPath, nsmgr);
            string uniqueId = uniqueIdNode.Attributes["value"].Value;
            string cdaDocId = document.GetDocId();
            bool isOid = Regex.IsMatch(cdaDocId, @"^\d+(\.\d+)*$");
            if (isOid)
            {
                LogAssert.AreEqual(cdaDocId, uniqueId, "the DocumentID field within the CDA document", "The XDSDocumentEntry.uniqueId");
            }
            else
            {
                LogAssert.AreEqual(CdaDocument.ConvertToOid(cdaDocId), uniqueId, "the DocumentID field within the CDA document (converted to an OID)", "The XDSDocumentEntry.uniqueId");
            }
        }

        /// <summary>
        /// Check P. The XDSDocumentEntry.sourcePatientId is identical to the XDSDocumentEntry.patientId field within the supplied CDA Document
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationP(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:ExternalIdentifier[@identificationScheme='urn:uuid:58a6f841-87b3-4a3e-92fd-a8ffeff98427']";
            Log("Info: Searching payload for XDSDocumentEntry.patientId using XPath {0}", xPath);
            XmlNode patientIdNode = payload.SelectSingleNode(xPath, nsmgr);
            string patientId = patientIdNode.Attributes["value"].Value;

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Slot[@name='sourcePatientId']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.sourcePatientId using XPath {0}", xPath);
            XmlNode sourcePatientIdNode = payload.SelectSingleNode(xPath, nsmgr);
            string sourcePatientId = sourcePatientIdNode.InnerText;

            LogAssert.AreEqual(patientId, sourcePatientId, "XDSDocumentEntry.patientId", "XDSDocumentEntry.sourcePatientId");
        }

        /// <summary>
        /// Check Q. The XDSDocumentEntry.formatCode field contains the Template ID of the Template to which the CDA Document asserts conformance
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        /// <param name="document">The CDA document.</param>
        /// <param name="documentFormatCode">The document format code.</param>
        private void EvaluationQ(XmlNode payload, XmlNamespaceManager nsmgr, CdaDocument document, string documentFormatCode)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:a09d5840-386c-46f2-b5ad-9c3699a4309d']";
            Log("Info: Searching payload for XDSDocumentEntry.formatCode using XPath {0}", xPath);
            XmlNode formatCodeNode = payload.SelectSingleNode(xPath, nsmgr);
            string formatCode = formatCodeNode.Attributes["nodeRepresentation"].Value;

            LogAssert.AreEqual(documentFormatCode, formatCode, "XDSDocumentEntry.formatCode");
        }

        /// <summary>
        /// Check R. The XDSDocumentEntry.languageCode field is set to a fixed value of 'en-AU'
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationR(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Slot[@name='languageCode']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.languageCode using XPath {0}", xPath);
            XmlNode languageCodeNode = payload.SelectSingleNode(xPath, nsmgr);
            string languageCode = languageCodeNode.InnerText;

            LogAssert.AreEqual("en-AU", languageCode, "XDSDocumentEntry.languageCode");
        }

        /// <summary>
        /// Check S: Only XDSSubmissionSet fields which are shown in the XDS.b
        /// Field Name of Table 3 in the  PCEHR Document Exchange Service TSS
        /// are provided.
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationS(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string[] requiredXDSSubmissionSetSlotNames =
            {
                "submissionTime",
            };
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage/rim:Slot";
            Log("Info: Searching payload for XDSSubmissionSet slots using XPath {0}", xPath);
            XmlNodeList xdsSubmissionSetSlots = payload.SelectNodes(xPath, nsmgr);
            foreach (XmlNode node in xdsSubmissionSetSlots)
            {
                string name = node.Attributes["name"].Value;
                bool validName = requiredXDSSubmissionSetSlotNames.Contains(name);
                string passMessage = string.Format("XDSSubmissionSet slot {0} is shown in the table.", name);
                string failMessage = string.Format("XDSSubmissionSet slot {0} is not shown in the table", name);
                LogAssert.IsTrue(validName, passMessage, failMessage);
            }

            Dictionary<string, string> requiredXDSSubmissionSetClassificationSchemes = new Dictionary<string, string>()
            {
                {"urn:uuid:a7058bb9-b4e4-4307-ba5b-e3f0ab85e12d", "XDSSubmissionSet.author"},
                {"urn:uuid:aa543740-bdda-424e-8c96-df4873be8500", "XDSSubmissionSet.contentTypeCode"},
                {"urn:uuid:96fdda7c-d067-4183-912e-bf5ee74998a8", "XDSSubmissionSet.uniqueId"},
                {"urn:uuid:554ac39e-e3fe-47fe-b233-965d2a147832", "XDSSubmissionSet.sourceId"},
                {"urn:uuid:6b5aea1a-874d-4603-a4bc-96a0a7b38446", "XDSSubmissionSet.patientId"}
            };

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage/rim:Classification";
            Log("Info: Searching payload for XDSSubmissionSet Classifications using XPath {0}", xPath);
            XmlNodeList xdsSubmissionSetClassifications = payload.SelectNodes(xPath, nsmgr);
            foreach (XmlNode node in xdsSubmissionSetClassifications)
            {
                string classificationScheme = node.Attributes["classificationScheme"].Value;
                bool validClassificationScheme = requiredXDSSubmissionSetClassificationSchemes.ContainsKey(classificationScheme);
                string passMessage = string.Format("XDSSubmissionSet classification {1} with UUID {0} is shown in the table.",
                    classificationScheme, requiredXDSSubmissionSetClassificationSchemes[classificationScheme]);
                string failMessage = string.Format("XDSSubmissionSet classification with UUID {0} is not shown in the table.",
                    classificationScheme);
                LogAssert.IsTrue(validClassificationScheme, passMessage, failMessage);
            }
        }

        /// <summary>
        /// Check T. The XDSSubmissionSet.entryUUID contains a symbolic identifier of 'SUBSET_SYMBOLICID'
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationT(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage";
            Log("Info: Searching payload for XDSSubmissionSet.entryUUID using XPath {0}", xPath);
            XmlNode entryUUIDNode = payload.SelectSingleNode(xPath, nsmgr);
            string entryUUID = entryUUIDNode.Attributes["id"].Value;
            LogAssert.AreEqual("SUBSET_SYMBOLICID_01", entryUUID, "XDSSubmissionSet.entryUUID");
        }

        /// <summary>
        /// Check U. The XDSSubmissionSet.authorPerson entity is identical to the value provided within the XDSDocumentEntry.authorPerson
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationU(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage/rim:Classification[@classificationScheme='urn:uuid:a7058bb9-b4e4-4307-ba5b-e3f0ab85e12d']/rim:Slot[@name='authorPerson']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSSubmissionSet.authorPerson using XPath {0}", xPath);
            XmlNode submissionSetAuthorPersonNode = payload.SelectSingleNode(xPath, nsmgr);
            string submissionSetAuthorPerson = submissionSetAuthorPersonNode.InnerText;

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:93606bcf-9494-43ec-9b4e-a7748d1a838d']/rim:Slot[@name='authorPerson']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.authorPerson using XPath {0}", xPath);
            XmlNode documentEntryAuthorPersonNode = payload.SelectSingleNode(xPath, nsmgr);
            string documentEntryAuthorPerson = documentEntryAuthorPersonNode.InnerText;

            LogAssert.AreEqual(documentEntryAuthorPerson, submissionSetAuthorPerson, "XDSDocumentEntry.authorPerson", "XDSSubmissionSet.authorPerson");
        }

        /// <summary>
        /// Check V. The XDSSubmissionSet.authorInstitution entity is identical to the value provided within the XDSDocumentEntry.authorInstitution
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationV(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage/rim:Classification[@classificationScheme='urn:uuid:a7058bb9-b4e4-4307-ba5b-e3f0ab85e12d']/rim:Slot[@name='authorInstitution']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSSubmissionSet.authorInstitution using XPath {0}", xPath);
            XmlNode submissionSetAuthorInstitutionNode = payload.SelectSingleNode(xPath, nsmgr);
            string submissionSetAuthorInstitution = submissionSetAuthorInstitutionNode.InnerText;

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:93606bcf-9494-43ec-9b4e-a7748d1a838d']/rim:Slot[@name='authorInstitution']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.authorInstitution using XPath {0}", xPath);
            XmlNode documentEntryAuthorInstitutionNode = payload.SelectSingleNode(xPath, nsmgr);
            string documentEntryAuthorInstitution = documentEntryAuthorInstitutionNode.InnerText;

            LogAssert.AreEqual(documentEntryAuthorInstitution, submissionSetAuthorInstitution, "XDSDocumentEntry.authorInstitution", "XDSSubmissionSet.authorInstitution");
        }

        /// <summary>
        /// Check W. The XDSSubmission.contentTypeCode entity is identical to the value in the XDSDocumentEntry.classCode entity
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationW(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage/rim:Classification[@classificationScheme='urn:uuid:aa543740-bdda-424e-8c96-df4873be8500']";
            Log("Info: Searching payload for XDSSubmissionSet.contentTypeCode using XPath {0}", xPath);
            string contentTypeCode = payload.SelectSingleNode(xPath, nsmgr).Attributes["nodeRepresentation"].Value;

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a']";
            Log("Info: Searching payload for XDSDocumentEntry.classCode using XPath {0}", xPath);
            string classCode = payload.SelectSingleNode(xPath, nsmgr).Attributes["nodeRepresentation"].Value;

            LogAssert.AreEqual(classCode, contentTypeCode, "XDSDocumentEntry.classCode", "XDSSubmissionSet.contentTypeCode");
        }

        /// <summary>
        /// Check X. The XDSSubmissionSet.contentTypeCodeDisplayName entity is identical with the value provided within the XDSDocumentEntry.classCodeDisplayName entity
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationX(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage/rim:Classification[@classificationScheme='urn:uuid:aa543740-bdda-424e-8c96-df4873be8500']/rim:Name/rim:LocalizedString";
            Log("Info: Searching payload for XDSSubmissionSet.contentTypeCodeDisplayName using XPath {0}", xPath);
            string contentTypeCodeDisplayName = payload.SelectSingleNode(xPath, nsmgr).InnerText;

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Classification[@classificationScheme='urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a']/rim:Name/rim:LocalizedString";
            Log("Info: Searching payload for XDSDocumentEntry.classCodeDisplayName using XPath {0}", xPath);
            string classCodeDisplayName = payload.SelectSingleNode(xPath, nsmgr).InnerText;

            LogAssert.AreEqual(classCodeDisplayName, contentTypeCodeDisplayName, "XDSDocumentEntry.classCodeDisplayName", "XDSSubmissionSet.contentTypeCodeDisplayName");
        }

        /// <summary>
        /// Check Z. The XDSSubmissionSet.personID entity is identical to the value provided within the XDSDocumentEntry.sourcePersonId entity
        /// (actually patientID not personID, and sourcePatientID not sourcePersonID)
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationZ(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:RegistryPackage/rim:ExternalIdentifier[@identificationScheme='urn:uuid:6b5aea1a-874d-4603-a4bc-96a0a7b38446']";
            Log("Info: Searching payload for XDSSubmissionSet.patientID using XPath {0}", xPath);
            string patientID = payload.SelectSingleNode(xPath, nsmgr).Attributes["value"].Value;

            xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:ExtrinsicObject/rim:Slot[@name='sourcePatientId']/rim:ValueList/rim:Value";
            Log("Info: Searching payload for XDSDocumentEntry.sourcePatientId using XPath {0}", xPath);
            string sourcePatientId = payload.SelectSingleNode(xPath, nsmgr).InnerText;

            LogAssert.AreEqual(sourcePatientId, patientID, "XDSDocumentEntry.sourcePatientId", "XDSSubmissionSet.patientID");
        }

        /// <summary>
        /// Check AB. The XDS.b Association Type is set to 'urn:ihe:ihi:2007:AssociationType:HasMember'
        /// Note: This test is incorrect, the Association Type should be 'urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember'.
        /// </summary>
        /// <param name="payload">The ProvideAndRegisterDocumentSetRequest node.</param>
        /// <param name="nsmgr">The namespace manager.</param>
        private void EvaluationAB(XmlNode payload, XmlNamespaceManager nsmgr)
        {
            string xPath = "lcm:SubmitObjectsRequest/rim:RegistryObjectList/rim:Association";
            Log("Info: Searching payload for Association using XPath {0}", xPath);
            string associationType = payload.SelectSingleNode(xPath, nsmgr).Attributes["associationType"].Value;

            LogAssert.AreEqual("urn:oasis:names:tc:ebxml-regrep:AssociationType:HasMember", associationType, "XDS.b Association Type");
        }

        /// <summary>
        /// Test No:         39
        /// Objective/Input:
        ///
        /// To ensure that the client system can handle a XDSRepositoryError
        /// (PCEHR_ERROR_3008 - Invalid template ID) when uploading a document
        /// with a wrong template ID. Upload a document with the wrong template id.
        /// The IHI to be used for this test is provided in the test data package.
        ///
        /// Expected Result:
        ///
        /// The request should be transmitted successfully to the PCEHR System
        /// and XDSRegistryError (PCEHR_ERROR_3008 - Invalid template ID) is
        /// returned. Client system is able to process that error response and
        /// inform the client system user appropriately.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_039()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            Log("Info: Adding Invalid Template ID to HIPS reference data.");
            SqlCommand command = patient.TestCommand;
            command.CommandText =
                @"INSERT hips.DocumentFormat
                  ([Code],[Description],[CodeSystemId],[DateCreated],[UserCreated],[DateModified],[UserModified])
                  VALUES
                  ('Invalid Template ID','Invalid Template ID',105,GETDATE(),'NOC TEST 39',GETDATE(),'NOC TEST 39')";
            command.ExecuteNonQuery();
            HipsResponse reloadResponse = ProxyHelper.ReferenceProxy.ReloadReferenceData(patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, reloadResponse, DialogueResource.HipsServiceReloadReferenceData);

            // The document should now go into the queue
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary3A);
            HipsResponse hipsResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), identifier, user, null, patient.TargetEpisode.AdmissionDate, "Invalid Template ID");
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, hipsResponse, DialogueResource.HipsServiceUploadDocument);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout, expectFailure: true);

            // Check the failure message
            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            LogAssert.AreEqual(
                "PcehrServiceError, Code: XDSRepositoryError, Description: PCEHR_ERROR_3008 - Invalid template ID for PCEHR",
                queueOps[0].Details,
                DialogueResource.QueueOperationStatusDetail);

            // Remove the invalid template ID from the database
            Log("Info: Removing Invalid Template ID from HIPS reference data.");
            SqlCommand command2 = patient.TestCommand;
            command2.CommandText =
                @"DELETE hips.DocumentFormat
                  WHERE Code = 'Invalid Template ID'";
            command2.ExecuteNonQuery();
            HipsResponse reloadResponse2 = ProxyHelper.ReferenceProxy.ReloadReferenceData(patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, reloadResponse2, DialogueResource.HipsServiceReloadReferenceData);
        }

        /// <summary>
        /// Test No:         40
        /// Objective/Input:
        ///
        /// Perform a ProvideAndRegisterDocument Set-b operation  with an IHI
        /// which does not exist in the PCEHR System. Client system uses an IHI
        /// number which does not exist in the PCEHR System. If the client
        /// system does not check for the existence of the PCEHR for the
        /// individual beforehand, the client system should be able to handle
        /// the error that the PCEHR system returns.
        ///
        /// Expected Result:
        ///
        /// The request should be transmitted successfully to the PCEHR System.
        /// The client system should be able to handle the response status type
        /// or error received.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_040()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 37");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary3A);
            HipsResponse hipsResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), identifier, user, null, patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, hipsResponse, DialogueResource.HipsServiceUploadDocument);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout, expectFailure: true);

            // Check the failure message
            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            LogAssert.AreEqual(
                "PcehrServiceError, Code: XDSRepositoryError, Description: PCEHR_ERROR_0004 - Authorisation denied",
                queueOps[0].Details,
                DialogueResource.QueueOperationStatusDetail);
        }

        /// <summary>
        /// Test No:         41
        /// Objective/Input:
        ///
        /// Perform a ProvideAndRegisterDocumentSet-b operation to amend a
        /// document which does not exist in the PCEHR system.
        /// Document being amended does not exist in the PCEHR system.
        /// AssociationType - ‘urn:ihe:iti:2007:AssociationType:RPLC’
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR system.
        /// The client system is able to handle the response details
        /// which contains the response status type Failure.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_041()
        {
            LogAssert.IsTrue(true, "HIPS does not allow the amendment of a document which does not exist in the PCEHR system.", "no");
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of a Discharge Summary.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_DS()
        {
            PCEHR_NOC_063(SampleDocumentType.DischargeSummary3A);
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of an Event Summary.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_ES()
        {
            PCEHR_NOC_063(SampleDocumentType.EventSummary);
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of a Specialist Letter.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_SL()
        {
            PCEHR_NOC_063(SampleDocumentType.SpecialistLetter);
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of a Shared Health Summary.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_SHS()
        {
            PCEHR_NOC_063(SampleDocumentType.SharedHealthSummary);
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of a PCEHR Prescription Record.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_PR()
        {
            PCEHR_NOC_063(SampleDocumentType.PcehrPrescriptionRecord);
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of a PCEHR Dispense Record.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_DR()
        {
            PCEHR_NOC_063(SampleDocumentType.PcehrDispenseRecord);
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of a Diagnostic Imaging Report.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_DI()
        {
            PCEHR_NOC_063(SampleDocumentType.DiagnosticImagingReport);
        }

        /// <summary>
        /// Executes PCEHR_NOC_63 to test replacement of a Pathology Report.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_063_PATH()
        {
            PCEHR_NOC_063(SampleDocumentType.PathologyReport);
        }

        /// <summary>
        /// Test No:         63
        /// Objective/Input:
        /// To show that the client system is able to replace a document that
        /// exists in the PCEHR System. The client system is able to submit a
        /// request with the document association type set to a value of
        /// ‘urn:ihe:iti:2007:AssociationType:RPLC’ (Replace).
        /// Document being amended can be specified using a UUID or an OID.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        private void PCEHR_NOC_063(SampleDocumentType docType)
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(docType);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

            CdaDocument document2 = patient.CreateAmendedDocument(document);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document2);
        }

        /// <summary>
        /// Test No:         111
        /// Objective/Input:
        /// Perform a ProvideAndRegisterDocumentSet-b operation to amend a
        /// document where the document identified by the Previous Version
        /// Document ID does not represent the latest version of the document
        /// in the PCEHR system. Document being amended is not represented by
        /// latest version ID in the PCEHR request. AssociationType -
        /// ‘urn:ihe:iti:2007:AssociationType:RPLC’.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. The
        /// client system is able to handle the response details which contains
        /// the response status type Failure.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_111()
        {
            Log(@"Info: Normally HIPS knows which is the latest version and
                submits the Document ID of the latest version. To force this
                situation to occur, after uploading the second version, the
                record of the second version will be deleted, and then a third
                version uploaded. HIPS will use the Document ID of the first
                version as the Previous Version Document ID, triggering this
                error.");
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document1 = patient.CreateNewDocument(SampleDocumentType.DischargeSummary3A);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document1);

            CdaDocument document2 = patient.CreateAmendedDocument(document1);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document2);

            Log("Info: Deleting Document {0} from HIPS Database", document2.GetDocId());
            using (SqlCommand command = patient.TestCommand)
            {
                command.CommandText = string.Format(
                    "DELETE hips.ClinicalDocumentVersion WHERE SourceSystemDocumentId = '{0}'",
                    document2.GetDocId());
                LogAssert.AreEqual(1, command.ExecuteNonQuery(), "Number of rows deleted");
            }

            CdaDocument document3 = patient.CreateAmendedDocument(document2);
            HipsResponse response = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(document3.GetBytes(), identifier, user, null, patient.TargetEpisode.AdmissionDate, document3.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response, DialogueResource.HipsServiceUploadDocument);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout, expectFailure: true);
            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            LogAssert.AreEqual(
                "PcehrServiceError, Code: XDSRepositoryError, Description: PCEHR_ERROR_3002 - Document metadata failed validation",
                queueOps[0].Details, DialogueResource.QueueOperationStatusDetail);
        }
    }
}