﻿using System;
using System.Collections.Generic;
using System.ServiceModel;
using HIPS.AppServer.HIPSServiceHost.Services;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Cda;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.Cda;
using HIPS.ServiceContracts.Cda.Message;
using HIPS.ServiceContracts.Common.Fault;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services that operate on CDA documents.
    /// </summary>
    public class CdaService : BaseService, ICdaService
    {
        #region Methods

        /// <summary>
        /// Creates a CDA discharge summary document that wraps a PDF document body.
        /// This may be for the purpose of uploading the discharge summary to the
        /// PCEHR or for provider-to-provider (P2P) secure message delivery (SMD).
        /// This takes in the PDF document, required metadata, and any attachments
        /// and creates a CDA document before packaging it.
        /// </summary>
        /// <param name="request">Request containing a PDF and required metadata for creating the CDA package.</param>
        /// <returns>Response containing details of the created document.</returns>
        public CreateDischargeSummaryLevel1AResponse CreateDischargeSummaryLevel1A(CreateDischargeSummaryLevel1ARequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;
            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var metadata = ObjectMapper.Map<HIPS.CommonSchemas.Cda.CdaHeaderMetadata>(request.CdaHeaderMetadata);
                var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(request.PatientIdentifier);
                var attachments = ObjectMapper.Map<List<HIPS.CommonSchemas.Cda.CdaAttachment>>(request.Attachments);

                // Invoke Business Service
                var businessService = new DischargeSummaryService(user);
                var businessResponse = businessService.CreateDischargeSummaryLevel1A(request.PdfDocument, metadata, patientIdentifier, attachments);

                // Map Common Model to DTO
                var svcResponse = ObjectMapper.Map<CreateDischargeSummaryLevel1AResponse>(businessResponse);
                return svcResponse;
            }
            catch (PcehrServiceException ex)
            {
                // Catch any business exceptions, log them in the “SystemErrorLog”
                // and map them to SOAP faults using AutoMapper. Ensure that the
                // message or code in each business exception is returned in the
                // SOAP fault.
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<PcehrServiceFault>>(ex);
            }
            catch (HiServiceException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<HiServiceFault>>(ex);
            }
            catch (ItemNotFoundException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<ItemNotFoundFault>>(ex);
            }
            catch (InvalidUserException ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToUploadDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<InvalidUserFault>>(ex);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ResponseStrings.ErrorUnableToCreateDischargeSummaryLevel1A, ex, user, LogMessage.HIPS_MESSAGE_169);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        #endregion Methods
    }
}