﻿#region Using Directives

using System.Collections.Generic;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PatientBusinessLogic;
using HIPS.PatientSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.ServiceContracts.ServiceInterfaces;

using DataSchemas = HIPS.PcehrDataStore.Schemas;
using ServiceSchemas = HIPS.PatientSchemas;

#endregion Using Directives

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// Implementation of web service operations on patients in the HIPS database.
    /// </summary>
    public partial class PatientService : BaseService, IPatientService
    {
        #region Methods

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital with or without a PCEHR.
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="withPcehr">True to list patients with a PCEHR, false to list patients without a PCEHR. Null for all patients.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <returns>A patient list response with status and a list of patients in hospital.</returns>
        public PatientListResponse ListPatientsInHospital(UserDetails user, string hospitalCodeSystem, string hospitalCode, bool? withPcehr, int daysDischarged)
        {
            PatientListLogic logic = new PatientListLogic();
            List<DataSchemas.PatientInHospital> data;
            HipsResponse response = logic.ListPatientsInHospital(user, hospitalCodeSystem, hospitalCode, withPcehr, daysDischarged, out data);

            PatientListResponse listResponse = new PatientListResponse();

            listResponse.HipsResponse = response;
            if (data != null)
            {
                listResponse.PatientInHospitalList = ObjectMapper.Map<List<ServiceSchemas.PatientInHospital>>(data);
            }
            return listResponse;
        }

        /// <summary>
        /// Lists the episodes for a specific patient in a defined hospital
        /// </summary>
        /// <param name="patientIdentifier">The local, state or national patient identifier</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <returns>
        /// Response containing the a list of the admitted or discharged episodes for the patient with in the specified number of days.,
        /// </returns>
        public PatientEpisodeListResponse ListPatientEpisodesInHospital(UserDetails user, PatientIdentifierBase patientIdentifier, int daysDischarged)
        {
            PatientListLogic logic = new PatientListLogic();
            List<PatientInHospitalEpisode> data;
            HipsResponse response = logic.ListPatientEpisodesInHospital(user, patientIdentifier, daysDischarged, out data);

            PatientEpisodeListResponse listResponse = new PatientEpisodeListResponse();
            listResponse.HipsResponse = response;
            if (data != null)
            {
                listResponse.PatientEpisodes = ObjectMapper.Map<List<PatientEpisode>>(data);
            }
            return listResponse;
        }

        /// <summary>
        /// Gets the single episode for a specific patient and contains the consent information
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="sourceSystemEpisodeId">The source system episode identifier for the episode in the hospital</param>
        /// <param name="patientIdentifier">The local, state or national patient identifier</param>
        /// <returns>
        /// Response containing the a list of the admitted or discharged episodes for the patient with in the specified number of days.,
        /// </returns>
        public EpisodePatientDetailsResponse GetEpisodeDetails(UserDetails user, string sourceSystemEpisodeId, PatientIdentifierBase patientIdentifier)
        {
            PatientListLogic logic = new PatientListLogic();
            EpisodePatientExtendedDetails data;
            HipsResponse response = logic.GetEpisodeDetails(user, patientIdentifier, sourceSystemEpisodeId, out data);

            EpisodePatientDetailsResponse episodeResponse = new EpisodePatientDetailsResponse();

            if (data != null && response.Status == HipsResponseIndicator.OK)
            {
                episodeResponse = ObjectMapper.Map<EpisodePatientDetailsResponse>(data);
            }
            episodeResponse.HipsResponse = response;

            return episodeResponse;
        }

        /// <summary>
        /// Gets the details of the patient’s discloser record for a specific HPIO
        /// </summary>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="patientIdentifier">The local, state or national patient identifier</param>
        /// <returns>
        /// Response containing the details of the patient's disclosure.
        /// </returns>
        public PatientDisclosureDetailsResponse GetPatientDisclosureDetails(UserDetails user, PatientIdentifierBase patientIdentifier)
        {
            PatientListLogic logic = new PatientListLogic();
            PatientDisclosureDetails data;
            HipsResponse response = logic.GetPatientDisclosureDetails(user, patientIdentifier, out data);

            PatientDisclosureDetailsResponse patientDisclosureResponse = new PatientDisclosureDetailsResponse();

            if (data != null && data.PatientInHospital.Mrn != null)
            {
                patientDisclosureResponse = ObjectMapper.Map<PatientDisclosureDetailsResponse>(data);
            }
            patientDisclosureResponse.HipsResponse = response;

            return patientDisclosureResponse;
        }

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital without an IHI.
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <returns>A patient list response with status and a list of patients in hospital.</returns>
        //public PatientListResponse ListPatientsWithoutIhi(UserDetails user, string hospitalCodeSystem, string hospitalCode, int daysDischarged)
        //{
        //    PatientListLogic logic = new PatientListLogic();
        //    List<DataSchemas.PatientInHospital> data;
        //    HipsResponse response = logic.ListPatientsWithoutIhi(user, hospitalCodeSystem, hospitalCode, daysDischarged, out data);

        //    ServiceSchemas.PatientListResponse listResponse = new ServiceSchemas.PatientListResponse();
        //    listResponse.HipsResponse = response;
        //    if (data != null)
        //    {
        //        listResponse.PatientInHospitalList = ObjectMapper.Map<List<ServiceSchemas.PatientInHospital>>(data);
        //    }
        //    return listResponse;
        //}

        /// <summary>
        /// Lists admitted or recently discharged patients in hospital. Filters are available
        /// to list patients with or without an IHI; with or without a PCEHR; whether to
        /// exclude patients with special values in the Medicare card number field; and the
        /// number of days after discharge that patients are included in the list.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="withIhi">Optional filter on having an active verified IHI, or no IHI.</param>
        /// <param name="withPcehr">Optional filter on having a PCEHR found, or no PCEHR found</param>
        /// <param name="excludeMedicareExclusions">Whether to exclude patients with special Medicare numbers.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <param name="user">The user who is requesting the information.</param>
        /// <returns>A patient list response with status and a list of patients in hospital.</returns>
        public AdmittedPatientListResponse ListAdmittedPatients(string hospitalCodeSystem, string hospitalCode, bool? withIhi, bool? withPcehr, bool excludeMedicareExclusions, int daysDischarged, UserDetails user)
        {
            PatientListLogic logic = new PatientListLogic();
            List<DataSchemas.PatientInHospital> data;
            HipsResponse response = logic.ListAdmittedPatients(hospitalCodeSystem, hospitalCode, withIhi, withPcehr, excludeMedicareExclusions, daysDischarged, user, out data);

            ServiceSchemas.AdmittedPatientListResponse listResponse = new ServiceSchemas.AdmittedPatientListResponse();
            listResponse.HipsResponse = response;
            if (data != null)
            {
                listResponse.AdmittedPatientList = ObjectMapper.Map<List<ServiceSchemas.AdmittedPatient>>(data);
            }
            return listResponse;
        }

        #endregion Methods
    }
}