﻿using System;
using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Author, Legal Authenticator and Responsible Health Professional using the ParticipatingProvider entity.
    /// </summary>
    internal class ProviderProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationLegalAuthenticator>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateLegalAuthenticator())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.ILegalAuthenticator>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForLegalAuthenticator())
                .ForMember(dest => dest.DateTimeAuthenticated, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src.Metadata.LegalAuthenticator));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.IParticipationResponsibleHealthProfessional>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateResponsibleHealthProfessional())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode
                {
                    Code = "253000",
                    CodeSystem = "2.16.840.1.113883.13.62",
                    CodeSystemName = "1220.0 - ANZSCO - Australian and New Zealand Standard Classification of Occupations, First Edition, 2006",
                    DisplayName = "Medical Practitioners nfd"
                }));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.IResponsibleHealthProfessional>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForResponsibleHealthProfessional())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPersonWithOrganisation>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonWithOrganisation())
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IPerson>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePerson())
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => ObjectMapper.Map<List<ModelSchemas.Cda.EntityIdentifier>>(src)))
                .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider>() { src }));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, List<ModelSchemas.Cda.EntityIdentifier>>()
                .ConvertUsing((ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider src) => new List<ModelSchemas.Cda.EntityIdentifier>()
                    {
                        src.Hpii != null ? new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = "HPI-I",
                            AssigningGeographicArea = "National Identifier",
                            Root = "1.2.36.1.2001.1003.0." + src.Hpii
                        } : null,
                        src.LocalIdentifier != null ? new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = src.EmployerName,
                            Root = "1.2.36.1.2001.1005.41." + src.EmployerHpio,
                            Extension = src.LocalIdentifier,
                            Code = new ModelSchemas.Cda.CdaCode
                            {
                                Code = "EI",
                                CodeSystem = "2.16.840.1.113883.12.203",
                                CodeSystemName = "Identifier Type (HL7)"
                            }
                        } : null,
                    }.Where(a => a != null).ToList());

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames.Split(' ')))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffix.Split(' ')))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => src.Title.Split(' ')));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.IEmploymentOrganisation>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateEmploymentOrganisation())
                .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.EmployerName))
                .ForMember(
                    dest => dest.Identifiers,
                    opt => opt.MapFrom(src => new List<ModelSchemas.Cda.EntityIdentifier>()
                    {
                        new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = "HPI-O",
                            AssigningGeographicArea = "National Identifier",
                            Root = "1.2.36.1.2001.1003.0." + src.EmployerHpio,
                        }
                    }));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationDocumentAuthor>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateAuthor())
                .ForMember(dest => dest.AuthorParticipationPeriodOrDateTimeAuthored, opt => opt.MapFrom(src => src.Metadata.DocumentCreationDateTime))
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src.Metadata.DocumentAuthor))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode
                    {
                        Code = "253000",
                        CodeSystem = "2.16.840.1.113883.13.62",
                        CodeSystemName = "1220.0 - ANZSCO - Australian and New Zealand Standard Classification of Occupations, First Edition, 2006",
                        DisplayName = "Medical Practitioners nfd"
                    }));

            this.CreateMap<ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider, NehtaSchemas.SCSModel.IAuthor>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForAuthor())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.ResolveUsing((ModelSchemas.Cda.ParticipatingIndividual.ParticipatingProvider src) => Guid.NewGuid()));
        }

        #endregion Methods
    }
}