﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using HIPS.Base.Schemas;
using HIPS.PcehrDataStore.Schemas.Schemas;
using BaseEnumerators = HIPS.Base.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.Schemas
{
    /// <summary>
    /// This class represents a hospital
    /// </summary>
    [KnownType(typeof(Hospital))]
    [Serializable]
    [DataContract]
    public class Hospital : BaseListSchema
    {
        #region Private members

        private string authorisedEmployeeName;
        private string authorisedEmployeeUserId;
        private int healthProviderOrganisationId;
        private int healthProviderOrganisationNetworkId;
        private string hiCertSerial;
        private bool hiCsp;
        private string hpiO;
        private string hpioName;
        private string hpoCertSerial;
        private byte[] logo;
        private string name;
        private string pcehrCertSerial;
        private bool pcehrCsp;
        private int uploadDocumentMinimumAge;
        private int visitorHospitalId;

        #endregion Private members

        #region Properties

        /// <summary>
        /// Gets or sets the addresses for the hospital.
        /// </summary>
        /// <value>
        /// The addresses.
        /// </value>
        [DataMember]
        public List<Address> Addresses { get; set; }

        /// <summary>
        /// Gets or sets the name of the authorised employee.
        /// </summary>
        /// <value>
        /// The name of the authorised employee.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string AuthorisedEmployeeName
        {
            get
            {
                return authorisedEmployeeName;
            }
            set
            {
                authorisedEmployeeName = value;
            }
        }

        /// <summary>
        /// Gets or sets the authorised employee user id.
        /// </summary>
        /// <value>
        /// The authorised employee user id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string AuthorisedEmployeeUserId
        {
            get
            {
                return authorisedEmployeeUserId;
            }
            set
            {
                authorisedEmployeeUserId = value;
            }
        }

        /// <summary>
        /// Gets or sets the list of codes.
        /// </summary>
        /// <value>
        /// The list of codes.
        /// </value>
        [DataMember]
        public List<HospitalCode> Codes { get; set; }

        /// <summary>
        /// Gets or sets the contact details for the hospital.
        /// </summary>
        /// <value>
        /// The contacts.
        /// </value>
        [DataMember]
        public List<Contact> Contacts { get; set; }

        /// <summary>
        /// Gets or sets the health provider organisation ID.
        /// </summary>
        /// <value>
        /// The health provider organisation ID.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int HealthProviderOrganisationId
        {
            get
            {
                return healthProviderOrganisationId;
            }
            set
            {
                healthProviderOrganisationId = value;
            }
        }

        /// <summary>
        /// Specifies which HPO network this organisation belongs to.
        /// </summary>
        /// <value>
        /// Health Provider Organisation Network Id
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public int HealthProviderOrganisationNetworkId
        {
            get
            {
                return healthProviderOrganisationNetworkId;
            }
            set
            {
                healthProviderOrganisationNetworkId = value;
            }
        }

        /// <summary>
        /// Gets or sets the serial number of the HI certificate.
        /// </summary>
        /// <value>
        /// The serial number of the HI certificate.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string HiCertSerial
        {
            get
            {
                return hiCertSerial;
            }
            set
            {
                hiCertSerial = value;
            }
        }

        /// <summary>
        /// Whether HIPS will connect to the HI Service as a CSP on behalf of this organisation.
        /// </summary>
        /// <value>
        /// HI Service CSP
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public bool HiCsp
        {
            get
            {
                return hiCsp;
            }
            set
            {
                hiCsp = value;
            }
        }

        /// <summary>
        /// Gets or sets the hospital id.
        /// </summary>
        /// <value>
        /// The hospital id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ApplicationGeneratedKey)]
        public int? HospitalId
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }

        /// <summary>
        /// Gets or sets the HPI-O.
        /// </summary>
        /// <value>
        /// The HPI-O.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string HpiO
        {
            get
            {
                return hpiO;
            }
            set
            {
                hpiO = value;
            }
        }

        /// <summary>
        /// Gets or sets the name of the hpio.
        /// </summary>
        /// <value>
        /// The name of the hpio.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string HpioName
        {
            get
            {
                return hpioName;
            }
            set
            {
                hpioName = value;
            }
        }

        /// <summary>
        /// Hexadecimal serial number for certificate used to sign CDA documents.
        /// This must be an HPO certificate. If not provided, HIPS will be unable to upload documents to the PCEHR.
        /// </summary>
        /// <value>
        /// HPO Certificate Serial Number
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string HpoCertSerial
        {
            get
            {
                return hpoCertSerial;
            }
            set
            {
                hpoCertSerial = value;
            }
        }

        /// <summary>
        /// Gets or sets the logo.
        /// </summary>
        /// <value>
        /// The logo.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public byte[] Logo
        {
            get
            {
                return logo;
            }
            set
            {
                logo = value;
            }
        }

        /// <summary>
        /// Gets or sets the name.
        /// </summary>
        /// <value>
        /// The name.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string Name
        {
            get
            {
                return name;
            }
            set
            {
                name = value;
            }
        }

        /// <summary>
        /// Gets or sets the serial number of the PCEHR certificate.
        /// </summary>
        /// <value>
        /// The serial number of the PCEHR certificate.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string PcehrCertSerial
        {
            get
            {
                return pcehrCertSerial;
            }
            set
            {
                pcehrCertSerial = value;
            }
        }

        /// <summary>
        /// Whether HIPS will connect to the PCEHR system as a CSP on behalf of this organisation.
        /// </summary>
        /// <value>
        /// Pcehr Service CSP
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public bool PcehrCsp
        {
            get
            {
                return pcehrCsp;
            }
            set
            {
                pcehrCsp = value;
            }
        }

        /// <summary>
        /// Gets or sets the number of years old the patient must have been at
        /// the time of admission in order to allow their documents to be
        /// uploaded to the PCEHR. When set 0 this disables the age filter.
        /// </summary>
        /// <value>
        /// The minimum age the patient must have been at admission to allow
        /// uploading documents to PCEHR.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int UploadDocumentMinimumAge
        {
            get
            {
                return uploadDocumentMinimumAge;
            }
            set
            {
                uploadDocumentMinimumAge = value;
            }
        }

        /// <summary>
        /// Gets or sets the visitor hospital identifier.
        /// </summary>
        /// <value>
        /// The visitor hospital identifier.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public int VisitorHospitalId
        {
            get
            {
                return visitorHospitalId;
            }
            set
            {
                visitorHospitalId = value;
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Gets the hospital code for a given code system. This method assumes this hospital only has one code in the
        /// given system, which is typically true. If more than one code is found, an exception will be thrown.
        /// For example, given "pasFacCd" it may return "RAH", or given "HPI-O" it may return "8003620123456789".
        /// </summary>
        /// <param name="hospitalCodeSystem">A code from CodeSystem.Code that specifies which type of hospital code to get</param>
        /// <returns>The hospital code of the requested type, or empty string if none found</returns>
        public string GetCode(string hospitalCodeSystem)
        {
            List<HospitalCode> codes = GetCodes(hospitalCodeSystem);
            if (codes.Count > 1)
            {
                throw new Exception(string.Format("Method should only return one code. It returned {0}", codes.Count));
            }
            if (codes.Count == 1)
            {
                return codes[0].Code;
            }
            return string.Empty;
        }

        /// <summary>
        /// Gets all the hospital codes for a given code system.
        /// This could be useful in cases where an external system uses more than one code to identify the same hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <returns>A list of hospital codes.</returns>
        public List<HospitalCode> GetCodes(string hospitalCodeSystem)
        {
            return Codes.Where(result => result.CodeSystemCode == hospitalCodeSystem).ToList();
        }

        #endregion Methods
    }
}