﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;
using HIPS.PcehrSchemas;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Xml;
using System.IO;
using System.Windows.Forms;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_502
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 7 March 2013
    /// Use Case #:	    UC.CIS.502
    /// Use Case Name:	Assisted Registration of an adult
    /// Test Case ID:   PCEHR_CIS_020204
    /// Objective:
    /// To assist a consumer with registering for an eHealth record. 
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_502_020204 : CcaTest
    {
        /// <summary>
        /// Prepare registration details, and complete all necessary registration form fields to result in a successful registration.
        /// a. Submit the registration request.
        /// b. Confirm that the registration process is successful.
        /// c. Verify the audit log entry contains at minimum the following information:
        ///     • Authorised Employee’s unique local system identifier (including HPI-I & full name, if available)
        ///     • Consumer’s IHI or Medicare number or DVA number or demographics that include the full name, date of birth and sex
        ///     • Authorised Representative’s IHI or Medicare number or DVA number or demographics that include the  full name, date of birth and sex
        ///     • Date (and time if available) of the assisted registration request.

        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_502")]
        public void CIS_502_020204()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 02");

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 10");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };
            
            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse =  ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetPatientIdentifier,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            if (registrationResponse.HipsResponse.Status == HipsResponseIndicator.PcehrServiceError && registrationResponse.HipsResponse.ResponseCode == "PCEHR_ERROR_9009")
            {
                MessageBox.Show("Have you reset the test data?");
            }

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that HIPS audited the request
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyRegisterPatientPayload(pcehrAudit, patientRep.TargetDemographic);
            
            // Check that the audit record contains a date and time
            LogAssert.IsNotNull(pcehrAudit.DateCreated, DialogueResource.AuditRecordContainsDateAndTime);

        }

        private void VerifyRegisterPatientPayload(PcehrAudit audit, Demographic representative)
        {
            // Confirm that the payload is well formed and complies with the gainPCEHRAccess Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RegisterPCEHR/2.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");

            // Check that the patient ID was provided in the request
            XmlNode ihiNumberNode = doc.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.AreEqual(patient.IhiInformation.Ihi, ihiNumberNode.InnerText, "IHINumber element value");

            // Check the representative demographics
            XmlNode repFamilyNameNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:representative/p:demographics/c:name/c:familyName", nsmgr);
            LogAssert.AreEqual(representative.FamilyName, repFamilyNameNode.InnerText, "Family Name element value");
            XmlNode repGivenNameNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:representative/p:demographics/c:name/c:givenName", nsmgr);
            LogAssert.AreEqual(representative.GivenName, repGivenNameNode.InnerText, "Given Name element value");
            XmlNode repSexNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:representative/p:demographics/c:sex", nsmgr);
            LogAssert.AreEqual(AuditHelper.GetSex(representative.Sex), repSexNode.InnerText, "Sex element value");
            XmlNode repDateOfBirthNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:representative/p:demographics/c:dateOfBirth", nsmgr);
            LogAssert.AreEqual(representative.DateOfBirth, DateTime.Parse(repDateOfBirthNode.InnerText), "Date of Birth element value");
            XmlNode repMedicareNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:representative/p:demographics/c:medicareCardNumber", nsmgr);
            LogAssert.AreEqual(representative.MedicareNumber, repMedicareNode.InnerText, "Medicare Card Number element value");
            XmlNode repMedicareIrnNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:representative/p:demographics/c:medicareIRN", nsmgr);
            LogAssert.AreEqual(representative.MedicareIrn, repMedicareIrnNode.InnerText, "Medicare IRN element value");
            
            // Check that the user ID was supplied
            XmlNode userIdNode = doc.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:User/c:ID", nsmgr);
            LogAssert.AreEqual(patient.GetTestUser().Login, userIdNode.InnerText, "User ID element value");

        }
    }
}
